# -*- coding: utf-8 -*-

import cgi
import six
import json
import logging
import traceback
import xml.dom.minidom
from operator import itemgetter
from collections import defaultdict
import re


from google.protobuf import message

from sandbox.projects.common import templates
from sandbox.projects.common.differ import printers
from .node import Node
from .querydata_pb2 import TQueryData

COLOR_ADDED = "#aaffaa"
COLOR_CHANGED = "#ffffbb"
COLOR_REMOVED = "#ffaaaa"
COLOR_NODIFF = "#e0e0e0"

NO_DIFF_BLOCK = "nd"

_NODE_POSITION_PROPERTY = "__ node position __"


class DiffError(Exception):
    pass


def StartBlock(title, backgroundColor=None, name=None, additionalStyles=None, open=True):
    """
        !!! DEPRECATED !!! try to use projects.common.search.response.diff.printers.HtmlBlock
    """
    params = {
        "title": title,
        "style": "",
        "name": "" if name is None else 'name="{}"'.format(name),
        "display": "block",
        "plusminus": "-",
    }

    if backgroundColor is not None:
        params["style"] += "background-color: {};".format(backgroundColor)

    if additionalStyles is not None:
        params["style"] += (' ' + additionalStyles)

    # do not put empty attributes
    if params["style"]:
        params["style"] = 'style="{}"'.format(params["style"])

    if not open:
        params["display"] = "none"
        params["plusminus"] = "+"

    return (
        '<div {style} {name}>'
        '<a href="#" OnClick="return openclose(this);"><b>[{plusminus}]</b></a> {title}'
        '<br>'
        '<div class="ml" style="display: {display}">'
    ).format(**params)


def EndBlock():
    """
        !!! DEPRECATED !!! try to use projects.common.search.response.diff.printers.HtmlBlock
    """
    return "</div></div>"


def colored_title(title, key):
    if type(key) is unicode:
        key = key.encode('utf-8')
    return "<b>{}</b>&nbsp;<b style='color: blue'>{}</b>".format(title, cgi.escape(key))


class NodeTypes:
    KEYNODE = 1
    MAP = 2


class NodeType(object):
    def __init__(self, name, path, type):
        self.Name = name
        self.Path = path  # e.g.: ["Grouping", "Group", "Document"]; None means any path
        self.Type = type  # see NodeTypes

    def WriteDiff(self, file, queryNumber, propertyPath, values1, values2, changedProps):
        return False


class KeyNode(NodeType):
    """
        Node which have unique key
    """

    def __init__(self, name, path):
        super(KeyNode, self).__init__(name, path, NodeTypes.KEYNODE)

    def GetKeyValue(self, node):
        raise Exception("GetKeyValue not overriden")


class KeyValueMapNode(NodeType):
    """
        Node which represents map key and value. e.g.:
            SearcherProp {
                Key: "La"
                Value: "22"
            },
            SearcherProp {
                Key: "Nctx"
                Value: "1"
            }
    """

    def __init__(self, name, path):
        super(KeyValueMapNode, self).__init__(name, path, NodeTypes.MAP)

    def GetKey(self, node):
        return node.GetPropValue("Key")

    def GetValue(self, node):
        return node.GetPropValue("Value", parse=False)


def node_types_dict(node_types_list):
    return {(t.Name, tuple(t.Path)): t for t in node_types_list}


def _get_node_type(name, path, node_types):
    return node_types.get((name, tuple(path)))


def _find_pairs(list1, list2):
    # At first try fastest stuff: just compare sums of hashes
    hashes1 = [item._hash for item in list1]
    hashes2 = [item._hash for item in list2]
    if sum(hashes1) == sum(hashes2):
        # sum of hashes are equal now, but we need to pair them according to hashes
        values1 = [item._value for item in list1]
        values2 = [item._value for item in list2]
        if values1 == values2:
            return zip(list1, list2, values1), [], []

    # Probably some nodes was added/removed, so try to find all pairs with equal hashes
    pairs = []
    set1 = set(list1)
    set2 = set(list2)

    hash_to_item1 = {item._hash: item for item in set1}
    hash_to_item2 = {item._hash: item for item in set2}
    for _hash in set(hashes1) & set(hashes2):
        item1 = hash_to_item1[_hash]
        item2 = hash_to_item2[_hash]

        pairs.append((item1, item2, item1._value))

        set1.remove(item1)
        set2.remove(item2)

    # Now find best pairs in nodes that left
    weights = []
    for n1 in set1:
        for n2 in set2:
            if n1._hash == n2._hash and n1._value == n2._value:
                logging.debug("bragic")
                weights.append((n1._value, n1, n2))
            else:
                weights.append((get_tree_similarity(n1, n2), n1, n2))
    weights.sort(
        key=itemgetter(0),
        reverse=True
    )

    for weight, n1, n2 in weights:
        if n1 in set1 and n2 in set2:
            pairs.append((n1, n2, weight))

            set1.remove(n1)
            set2.remove(n2)

    # pairs, added, removed
    return pairs, list(set2), list(set1)


def _find_only_pairs_weight(list1, list2, weights):
    c1 = defaultdict(int)
    for l1 in list1:
        c1[l1] += 1
    c2 = defaultdict(int)
    for l2 in list2:
        c2[l2] += 1

    result = 0
    weights.sort(key=itemgetter(0), reverse=True)

    for weight, n1, n2 in weights:
        if c1[n1] != 0 and c2[n2] != 0:
            result += weight + 1
            c1[n1] -= 1
            c2[n2] -= 1

    return result


def get_tree_similarity(node1, node2):
    result = 0

    if node1._hash == node2._hash:
        if node1._value == node2._value:
            return node1._value
        else:
            logging.debug("OMG, hashes are equal, but values are not")

    for key, list1 in node1._nodes.iteritems():
        list2 = node2._nodes.get(key, [])
        weights = []
        for n1 in list1:
            for n2 in list2:
                if n1._hash == n2._hash and n1._value == n2._value:
                    weights.append((n1._value, n1, n2))
                else:
                    weights.append((get_tree_similarity(n1, n2), n1, n2))
        result += _find_only_pairs_weight(list1, list2, weights)

    for key, list1 in node1._props.iteritems():
        list2 = node2._props.get(key, [])
        if 1 == len(list1) == len(list2):
            result += 2 if list1[0] == list2[0] else 1
        else:
            weights = [(1 if n1 == n2 else 0, n1, n2) for n1 in list1 for n2 in list2]
            result += _find_only_pairs_weight(list1, list2, weights)

    return result


def write_nodiff_tree(file_obj, data, title):
    file_obj.write(StartBlock(title, COLOR_NODIFF, NO_DIFF_BLOCK, "display: none;"))
    file_obj.write(printers.HtmlBlock.simple_data(data))
    file_obj.write(EndBlock())


def WriteData(file, text, backgroundColor=None, preprocessText=cgi.escape):
    """
        !!! DEPRECATED !!! try to use projects.common.search.response.diff.printers.HtmlBlock
    """
    if type(text) is unicode:
        text = text.encode('utf-8')
    htext = preprocessText(text)
    if backgroundColor is not None:
        file.write("<pre style='background-color: {}'>{}</pre>".format(backgroundColor, htext))
    else:
        file.write("<pre>" + htext + "</pre>")


def WriteDataBlock(file, title, text, backgroundColor=None):
    """
        !!! DEPRECATED !!! try to use projects.common.search.response.diff.printers.HtmlBlock
    """
    file.write(StartBlock(title, backgroundColor))
    WriteData(file, text)
    file.write(EndBlock())


def _write_nodes_list_grouped_by_key(file_obj, title, nodes_by_key_list, background_color):
    for key, nodes in nodes_by_key_list:
        node_name = colored_title(title, key)

        for node in nodes:
            WriteDataBlock(file_obj, node_name, node.ToStr(), background_color)


def _prop_vals_to_str(values):
    if len(values) == 1:
        return values[0]
    else:
        return "\r\n\r\n".join(values)


def _write_props_list_grouped_by_name(file_obj, title, props_by_name_list, background_color):
    for name, values in props_by_name_list:
        block_title = "<b>{}</b>".format(name)
        WriteDataBlock(file_obj, block_title, _prop_vals_to_str(values), background_color)


def _group_nodes_by_key(nodes, node_type):
    nodes_by_key = {}

    for node in nodes:
        key_val = node_type.GetKeyValue(node)
        nodes_by_key.setdefault(key_val, []).append(node)

    return nodes_by_key


def _cmp_objects_grouped_by_key(
    file_obj,
    query_number,
    objects_dict1,
    objects_dict2,
    obj_name,
    write_list_func,
    compare_func,
    path,
    is_prop,
    changed_props
):
    keys1 = set(objects_dict1.keys())
    keys2 = set(objects_dict2.keys())

    added = keys2 - keys1
    removed = keys1 - keys2

    if len(added) > 0:
        objects_by_key_tuple = [(key, objects_dict2[key]) for key in added]

        block_title = "<b>{}</b>".format(obj_name)

        if is_prop:
            for key, values in objects_by_key_tuple:
                changed_props.add(query_number, path + [key], COLOR_ADDED, ",".join(values))
        else:
            for key in added:
                for node in objects_dict2[key]:
                    node.remove_prop(_NODE_POSITION_PROPERTY)
            changed_props.add(query_number, path, COLOR_ADDED)

        write_list_func(file_obj, block_title, objects_by_key_tuple, COLOR_ADDED)

    if len(removed) > 0:
        objects_by_key_tuple = [(key, objects_dict1[key]) for key in removed]

        block_title = "<b>{}</b>".format(obj_name)

        if is_prop:
            for key, values in objects_by_key_tuple:
                changed_props.add(query_number, path + [key], COLOR_REMOVED, ",".join(values))
        else:
            for key in removed:
                for node in objects_dict1[key]:
                    node.remove_prop(_NODE_POSITION_PROPERTY)
            changed_props.add(query_number, path, COLOR_REMOVED)

        write_list_func(file_obj, block_title, objects_by_key_tuple, COLOR_REMOVED)

    has_diff = len(added) or len(removed)
    for key in keys1 & keys2:
        epsilon = None
        try:
            key1 = objects_dict1.get("Key")
            key2 = objects_dict2.get("Key")
            if (key1 == ['SuperMindMult'] and key2 == ['SuperMindMult']) or \
                    (key1 == ['"SuperMindMult"'] and key2 == ['"SuperMindMult"']):
                epsilon = 0.03
        except Exception:
            logging.debug(traceback.format_exc())

        has_diff = compare_func(
            file_obj,
            query_number,
            key,
            objects_dict1[key],
            objects_dict2[key],
            path,
            epsilon=epsilon,
        ) or has_diff
    return has_diff


def write_tree_diff(
    file_obj, query_num,
    node1, node2, node_types,
    changed_props,
    current_path=[]
):
    node1.build_hash()
    node2.build_hash()
    node1.build_value()
    node2.build_value()
    return _write_tree_diff(
        file_obj, query_num,
        node1, node2, node_types,
        changed_props,
        current_path
    )


def _write_tree_diff(
    file_obj, query_num,
    node1, node2, node_types,
    changed_props,
    current_path
):
    def group_and_compare(file, query_number, nodes1, nodes2, path_to_nodes, node_type):
        if len(nodes1) == 1 and len(nodes2) == 1:
            title = colored_title(path_to_nodes[-1], node_type.GetKeyValue(nodes1[0]))

            if nodes1[0].Compare(nodes2[0]):
                write_nodiff_tree(file, nodes1[0].ToStr(), title)
                return False

            file.write(StartBlock(title, COLOR_CHANGED))
            has_diff = _write_tree_diff(
                file, query_number,
                nodes1[0], nodes2[0], node_types,
                changed_props, path_to_nodes
            )
            file.write(EndBlock())
            return has_diff

        for index, node in enumerate(nodes1, start=1):
            node.SetPropValue(_NODE_POSITION_PROPERTY, str(index))
        for index, node in enumerate(nodes2, start=1):
            node.SetPropValue(_NODE_POSITION_PROPERTY, str(index))

        return _cmp_objects_grouped_by_key(
            file,
            query_number,
            _group_nodes_by_key(nodes1, node_type),
            _group_nodes_by_key(nodes2, node_type),
            path_to_nodes[-1],
            _write_nodes_list_grouped_by_key,
            fuzzy_cmp_nodes,
            path_to_nodes,
            False,
            changed_props
        )

    def fuzzy_cmp_nodes(file, query_number, key, nodes1, nodes2, path_to_nodes, epsilon=None):
        if len(nodes1) == 1 and len(nodes2) == 1:
            pairs, added, removed = [(nodes1[0], nodes2[0], 0)], [], []
        else:
            pairs, added, removed = _find_pairs(nodes1, nodes2)

        if len(added) > 0:
            node_name = "<b>{} {}</b>".format(path_to_nodes[-1], key)

            for node in added:
                changed_props.add(query_number, path_to_nodes, COLOR_ADDED)
                WriteDataBlock(file, node_name, node.ToStr(), COLOR_ADDED)

        if len(removed) > 0:
            node_name = "<b>{} {}</b>".format(path_to_nodes[-1], key)

            changed_props.add(query_number, path_to_nodes, COLOR_REMOVED)
            for node in removed:
                WriteDataBlock(file, node_name, node.ToStr(), COLOR_REMOVED)

        has_diff = len(added) or len(removed)
        for n1, n2, weight in pairs:
            title = colored_title(path_to_nodes[-1], key)

            if n1.Compare(n2):
                write_nodiff_tree(file, n1.ToStr(), title)
                continue

            file.write(StartBlock(title, COLOR_CHANGED))
            has_diff = _write_tree_diff(
                file, query_number,
                n1, n2, node_types,
                changed_props, path_to_nodes
            ) or has_diff

            file.write(EndBlock())
        return has_diff

    def cmp_nodes(file, query_number, node_name, nodes1, nodes2, path_to_nodes, epsilon=None):
        node_type = _get_node_type(node_name, current_path, node_types)

        path_to_nodes = current_path[:]
        path_to_nodes.append(node_name)

        if node_type is not None and node_type.Type == NodeTypes.KEYNODE:
            return group_and_compare(file, query_number, nodes1, nodes2, path_to_nodes, node_type)
        else:
            return fuzzy_cmp_nodes(file, query_number, "", nodes1, nodes2, path_to_nodes)

    def cmp_props(file, query_number, prop_name, values1, values2, path_to_nodes, epsilon=None):
        title = "<b>{}</b>".format(prop_name)
        if values1 == values2:
            write_nodiff_tree(file, _prop_vals_to_str(values1), title)
            return False

        if epsilon is not None:
            try:
                logging.debug("epsilon is not None")
                if len(values1) == len(values2) == 1:
                    value1_stripped, value2_stripped = values1[0], values2[0]
                    try:
                        value1_stripped = _strip_quotes(value1_stripped)
                        value2_stripped = _strip_quotes(value2_stripped)
                    except Exception:
                        pass
                    if abs(float(value1_stripped) - float(value2_stripped)) < epsilon:
                        logging.debug("Return false")
                        write_nodiff_tree(file, _prop_vals_to_str(values1), title)
                        return False
            except Exception as e:
                logging.debug(e)

        file.write(StartBlock(title))

        diff_written = False
        node_type = _get_node_type(path_to_nodes[-1], path_to_nodes[:-1], node_types) if path_to_nodes else None

        prop_path = path_to_nodes + [prop_name]

        if node_type is not None:
            diff_written = node_type.WriteDiff(
                file,
                query_number,
                prop_path,
                [_strip_quotes(v) for v in values1],
                [_strip_quotes(v) for v in values2],
                changed_props
            )

        if not diff_written:
            values1 = _prop_vals_to_str(values1)
            values2 = _prop_vals_to_str(values2)
            changed_props.add_property(query_number, prop_path, values1, values2)

            values_list1, values_list2 = printers.match_string_color_changed(values1, values2)
            file.write(printers.HtmlBlock.colored_data(values_list1))
            file.write(printers.HtmlBlock.colored_data(values_list2))

        file.write(EndBlock())
        return True

    def convert_map_nodes_to_props(nodes, node_type):
        node = Node()

        for n in nodes:
            key = node_type.GetKey(n)
            value = node_type.GetValue(n)

            if isinstance(value, Node):
                # value is node, attach subnode
                add_dict = node._nodes
            else:
                # value is string, attach prop
                add_dict = node._props

            if key not in add_dict:
                add_dict[key] = [value]
            else:
                add_dict[key].append(value)

        return node

    def patch_nodes(original_nodes):
        patched_nodes = {}

        for node_name, nodes in original_nodes.iteritems():
            node_type = _get_node_type(node_name, current_path, node_types)

            if node_type is None or node_type.Type != NodeTypes.MAP:
                # stay unchanged
                patched_nodes[node_name] = nodes
                continue

            # convert MAP nodes to props
            node = convert_map_nodes_to_props(nodes, node_type)
            patched_nodes[node_name] = [node]

        return patched_nodes

    d1 = _cmp_objects_grouped_by_key(
        file_obj,
        query_num,
        node1.beautify_props(),
        node2.beautify_props(),
        "",
        _write_props_list_grouped_by_name,
        cmp_props,
        current_path,
        True,
        changed_props,
    )
    d2 = _cmp_objects_grouped_by_key(
        file_obj,
        query_num,
        patch_nodes(node1._nodes),
        patch_nodes(node2._nodes),
        "",
        _write_nodes_list_grouped_by_key,
        cmp_nodes,
        current_path,
        False,
        changed_props,
    )
    return d1 or d2


def _strip_quotes(string):
    if string.startswith("\"") and string.endswith("\""):
        return string[1:-1]
    return string


_HTML_HEADER = templates.get_html_template('header.html')


_LEGEND = """
<table border=1>
    <tr> <th colspan=4> Legend </th> </tr>
    <tr> <td class="{}">Added</td>
         <td class="{}">Changed</td>
         <td class="{}">Removed</td>
         <td class="{}">No diffs</td>
    </tr>
</table><p>
""".format(
    printers.DiffType.ADDED,
    printers.DiffType.CHANGED,
    printers.DiffType.REMOVED,
    printers.DiffType.NODIFF
)

_BUTTONS = """
<input id="btnShowNodes" type="button" onclick="showNodes()" value="Show all nodes" />
<input id="btnHideNodes" type="button" onclick="hideNodes()" value="Show only diffs" style="display: none;"/><p>
"""

_HTML_FOOTER = "</body></html>"


def WriteDiff(file, WriteBody, addLegend=True, addButtons=True, title=None):
    if title is None:
        title = "diff"
    file.write(_HTML_HEADER % title)

    if addLegend:
        file.write(_LEGEND)

    if addButtons:
        file.write(_BUTTONS)

    have_diff = WriteBody()

    file.write(_HTML_FOOTER)

    return have_diff


class ChangedProps(dict):
    NUM_VALUES = 10

    def _add(self, queryNumber, key, value):
        values = self.get(key, set([]))
        if value is not None:
            values.add((queryNumber, value))
        else:
            values.add((queryNumber, '{changed nodes-tree - see full_diff for details}'))
        self[key] = values

    def add(self, queryNumber, path, color, value=None):
        self._add(queryNumber, ("/".join(path), color), value)

    def add_property(self, queryNumber, path, value1, value2):
        self.add(queryNumber, path, COLOR_CHANGED, (value1, value2))

    def combine(self, another):
        for key, values in another.iteritems():
            old_values = self.get(key, set([]))
            self[key] = set(list(old_values | values)[:self.NUM_VALUES])

    def write(self, file):
        def writeBody():
            for key in sorted(self.keys()):
                path, color = key
                values = self[key]
                if len(values) == 0:
                    WriteData(file, path, color)
                else:
                    file.write(StartBlock(path, color))

                    for query_number, value in values:
                        if len(values) > 1:
                            file.write(StartBlock("query: {}".format(query_number), color))
                        else:
                            file.write("query: {}".format(query_number))

                        if type(value) == tuple:
                            values0, values1 = printers.match_string_color_changed(value[0], value[1])
                            file.write(printers.HtmlBlock.colored_data(values0))
                            file.write(printers.HtmlBlock.colored_data(values1))
                        else:
                            if value is None:
                                raise Exception(path)
                            WriteData(file, value, color)

                        if len(values) > 1:
                            file.write(EndBlock())

                    file.write(EndBlock())

        WriteDiff(file, writeBody, addButtons=False, title="changed props")


def json_dict_to_tree(text):
    j = json.loads(text, encoding='utf-8')
    return _convert_json_dict_to_tree_impl(j)


def _legalize_node_name(name):
    """
        make sure that name is legal protobuf field name (only alnum or '_')
    """
    if not name:
        return '_'  # empty node name also not allowed
    if type(name) is unicode:
        name = name.encode('utf-8')
    return re.sub(r'\W', '_', name)


def _append_list_item(dict_obj, k, v):
    if k in dict_obj:
        dict_obj[k].append(v)
    else:
        dict_obj[k] = [v]


def _try_unpack_as_string(v, node, key):
    if (isinstance(v, unicode) or isinstance(v, str)) and len(v) > 200:
        try:
            matreshka = json_dict_to_tree(v)
            _append_list_item(node._nodes, key, matreshka)
            return
        except Exception:
            pass
    _append_list_item(node._props, key, _format_value(v))


def _convert_json_dict_to_tree_impl(dict_obj):
    node = Node()
    for key, value in dict_obj.iteritems():
        key = _legalize_node_name(key)
        if isinstance(value, dict):
            _append_list_item(node._nodes, key, _convert_json_dict_to_tree_impl(value))
        elif isinstance(value, list):
            for v in value:
                if isinstance(v, dict):
                    _append_list_item(node._nodes, key, _convert_json_dict_to_tree_impl(v))
                else:
                    _try_unpack_as_string(v, node, key)
        else:
            _try_unpack_as_string(value, node, key)
    return node


def _format_value(val):
    if isinstance(val, unicode):
        val = val.encode('utf-8')
    elif isinstance(val, str):
        pass
    elif isinstance(val, (bool, int, float)):
        return str(val)
    else:
        val = str(val)
    return '"{}"'.format(val.replace('\\', '\\\\').replace('"', '\\"').replace('\n', '\\n'))


def _legalize_string(unknown_string):
    if isinstance(unknown_string, unicode):
        return unknown_string.encode('utf-8')
    return unknown_string


def _legalize_xml_name(unknown_string):
    return _legalize_string(unknown_string).replace(':', '__')


def convert_xml_to_tree(xml_str, quote_ampersands=False, attributes_as_suffix=[]):
    if quote_ampersands:
        xml_str = xml_str.replace('&', '&amp;')
    try:
        doc_elem = xml.dom.minidom.parseString(xml_str).documentElement
    except Exception:
        exc_text = traceback.print_exc()
        logging.error("Cannot parse XML:\n%s", exc_text)
        raise DiffError("Cannot parse XML. Error:\n{}".format(exc_text))
    root = Node()
    root._nodes[_legalize_string(doc_elem.tagName)] = [_convert_xml_to_tree(doc_elem, attributes_as_suffix)]
    return root


def _convert_xml_to_tree(dom_obj, attributes_as_suffix=[]):
    node = Node()

    def _add_elem(add_dict, name, value):
        if value is None:
            value = ''
        if name in add_dict:
            add_dict[name].append(value)
        else:
            add_dict[name] = [value]

    def _add_node_with_suffix(add_dict, orig_name, value, props_as_suffix):
        name = orig_name
        for p in props_as_suffix:
            if p in value._props:
                name += '_' + '_'.join(value._props[p])
                del value._props[p]
        _add_elem(add_dict, name, value)

    txt = ''
    if dom_obj.attributes:
        for i in six.moves.xrange(dom_obj.attributes.length):
            attr = dom_obj.attributes.item(i)
            local_name = _legalize_xml_name(attr.localName)
            prop_name = '{}__{}'.format(_legalize_xml_name(attr.prefix), local_name) if attr.prefix else local_name
            _add_elem(node._props, prop_name, attr.value)

    for n in dom_obj.childNodes:
        if n.nodeType in (n.TEXT_NODE, n.CDATA_SECTION_NODE):
            txt += _legalize_string(n.nodeValue.strip())
        elif n.nodeType == n.ATTRIBUTE_NODE:
            _add_elem(node._props, _legalize_string(n.nodeName), _legalize_string(n.nodeValue))
        elif n.nodeType == n.ELEMENT_NODE:
            tagName = _legalize_xml_name(n.tagName)
            _add_node_with_suffix(node._nodes, tagName, _convert_xml_to_tree(n), attributes_as_suffix)

    if len(txt):
        node._props['_text'] = ['"{}"'.format(txt.encode("string_escape").replace('"', '\\"'))]

    return node


def is_iterable(obj):
    try:
        iter(obj)
    except TypeError:
        return False
    else:
        return type(obj) not in [str, unicode]


def convert_query_data_proto_to_tree(s):
    qd = TQueryData()
    try:
        qd.ParseFromString(s)
    except:
        logging.error("bad query data proto: {0}".format(s))
        raise
    return _query_data_proto_to_tree(qd)


def _query_data_proto_to_tree(msg):
    node = Node()
    for field, value in msg.ListFields():
        if is_iterable(value):
            node._nodes[field.name] = [_query_data_proto_to_tree(sub) for sub in value]
        elif isinstance(value, message.Message):
            node._nodes[field.name] = [_query_data_proto_to_tree(value)]
        # elif field.full_name == 'NQueryData.TFactor.StringValue':
        #     node._nodes[field.name] = [] if value is None else [json_dict_to_tree(value)]
        else:
            node._props[field.name] = [] if value is None else [str(value)]
    return node
