# -*- coding: utf-8 -*-

import hashlib
from sandbox.projects.common import string as us
from . import exception as ne


class Node(object):
    __slots__ = ['_props', '_nodes', '_hash', '_value']

    class EvalFailed():
        pass

    def __init__(self):
        self._props = {}
        self._nodes = {}

    def build_hash(self):
        def _magic_hash(s):
            if isinstance(s, unicode):
                s = str(hash(s))  # questionable decision, see also https://bugs.python.org/issue2948
            return int("0x" + hashlib.md5(s).hexdigest(), 0)

        self._hash = 0
        for k, v in self._props.iteritems():
            self._hash += _magic_hash(k)
            for p in v:
                self._hash += _magic_hash(p)
        for k, v in self._nodes.iteritems():
            self._hash += _magic_hash(k)
            for n in v:
                self._hash += n.build_hash()
        return self._hash

    def build_value(self):
        self._value = 0
        for v in self._props.itervalues():
            self._value += len(v) * 2
        for v in self._nodes.itervalues():
            self._value += len(v)
            for n in v:
                self._value += n.build_value()
        return self._value

    def nodes(self):
        return self._nodes

    def props(self):
        return self._props

    def get_node(self, node_name, required=True):
        if node_name not in self._nodes:
            if not required:
                return None
            raise ne.ElementNotFoundException("{} node not found".format(node_name))
        if len(self._nodes[node_name]) > 1:
            s = ("\n" + "-" * 100 + "\n").join([
                us.shift_right(n.ToStr()) for n in self._nodes[node_name]
            ])
            raise ne.UnknownError("Found more than 1 occurrences of {}:\n{}".format(node_name, s))
        return self._nodes[node_name][0]

    def beautify_props(self):
        props = {}

        for key, values in self._props.iteritems():
            props[key] = [self._parse_prop_value(value, remove_quotes=False) for value in values]

        return props

    def get_prop_values(self, propName, required=True, parse=True):
        values = self._props.get(propName, [])
        if not values:
            if required:
                raise ne.ElementNotFoundException("{} property not found".format(propName))
            return values
        if parse:
            values = [self._parse_prop_value(value, remove_quotes=True) for value in values]
        return values

    def _parse_prop_value(self, value, remove_quotes=False):
        if value and not (value[0] == value[-1] == '"'):
            return value
        try:
            return eval(value) if remove_quotes else '"{}"'.format(eval(value))
        except Exception:
            return value

    def GetPropValue(self, propName, required=True, parse=True):
        values = self.get_prop_values(propName, required, False)
        if not values:
            return None
        if len(values) > 1:
            raise ne.UnknownError("Found more than 1 occurences of {} property".format(propName))
        if parse:
            return self._parse_prop_value(values[0], remove_quotes=True)
        return values[0]

    def SetPropValue(self, propName, value):
        self._props[propName] = [value]

    def remove_prop(self, prop_name):
        self._props.pop(prop_name, None)

    def ToStr(self):
        return "\n".join(self.to_line_arr())

    def to_line_arr(self, level=0):
        indention = "  " * level

        result = []

        for name in sorted(self._props.keys()):
            values = self._props[name]
            if type(name) is unicode:
                name = name.encode('utf-8')
            for value in values:
                if type(value) is unicode:
                    value = value.encode('utf-8')
                result.append("{}{}: {}".format(indention, name, value))

        for name in sorted(self._nodes.keys()):
            items = self._nodes[name]
            for item in items:
                if type(name) is unicode:
                    name = name.encode('utf-8')
                result.append(indention + name + " {")
                result.extend(item.to_line_arr(level + 1))
                result.append(indention + "}")

        return result

    def Compare(self, anotherObj):
        """
            Returns true when no diffs
        """
        return self._compare_objects(self, anotherObj)

    @classmethod
    def _compare_dicts(cls, dict1, dict2, is_props):
        keys1 = set(dict1.keys())
        keys2 = set(dict2.keys())

        if len(keys2 ^ keys1) > 0:
            return False

        for key in keys1:  # keys are same
            list1 = dict1[key]
            list2 = dict2[key]

            if is_props:
                if list1 != list2:
                    return False
            else:
                if not cls._compare_lists(list1, list2):
                    return False

        return True

    @classmethod
    def _compare_lists(cls, list1, list2):
        if len(list1) != len(list2):
            return False

        for v1, v2 in zip(list1, list2):
            if not cls._compare_objects(v1, v2):
                return False

        return True

    @classmethod
    def _compare_objects(cls, node1, node2):
        return (
            cls._compare_dicts(node1._props, node2._props, True) and
            cls._compare_dicts(node1._nodes, node2._nodes, False)
        )


def _crop_string(s):
    if len(s) > 1000:
        return ''.join((s[:1000], "(showing only first 1000 symbols)"))
    else:
        return s


def WalkPath(node, path, callback):
    child_nodes = node._nodes.get(path[0])
    if not child_nodes:
        return
    if len(path) > 1:
        for child_node in child_nodes:
            WalkPath(child_node, path[1:], callback)
    else:
        for child_node in child_nodes:
            callback(child_node)
