#!/usr/bin/env python

import abc
import requests
import logging


class ApiInterface(object):
    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def beta_exists(self, beta_name):
        pass

    @abc.abstractmethod
    def get_beta_info(self, beta_name):
        pass

    @abc.abstractmethod
    def get_beta_state(self, beta_name):
        pass

    @abc.abstractmethod
    def start_beta(self, beta_name):
        pass

    @abc.abstractmethod
    def stop_beta(self, beta_name):
        pass

    @abc.abstractmethod
    def get_instances(self, beta_name, source):
        pass

    @abc.abstractmethod
    def retrieve_api_beta_template(self, template_name):
        pass

    @abc.abstractmethod
    def update_patches(self, beta_name, patches):
        pass

    @abc.abstractmethod
    def retrieve_api_beta(self, beta_name):
        pass


class ApiCommon(ApiInterface):
    def __init__(self, srv, token=None, timeout=30, ssl_verify=True, id=None):
        self._srv = srv.rstrip('/')
        self.__token = token
        self.timeout = timeout
        self.headers = self._set_headers(token)
        self.ssl_verify = ssl_verify
        self.logger = logging.getLogger(__name__)
        self.id = id

    @staticmethod
    def _set_headers(token):
        headers = {'Content-type': 'application/json'}
        if token:
            headers['Authorization'] = 'OAuth ' + token
        return headers

    def _check_response(self, response, request_type):
        self.logger.debug("Checking response of url: %s", response.url)
        self.logger.debug("Response headers: %s", response.headers)
        self.logger.debug('Return code: %s. Content:\n%s', response.status_code, response.content)
        if response.status_code == requests.codes.ok:
            return response.json()
        else:
            raise Exception('{} request fails with status code {}: {}'.format(
                request_type, response.status_code, response.content
            ))

    def _make_req(self, url):
        r = requests.get(self._srv + url, timeout=self.timeout, headers=self.headers, verify=self.ssl_verify)
        return self._check_response(r, "GET")

    def _make_req_put(self, url, data=None):
        r = requests.put(
            self._srv + url,
            timeout=self.timeout,
            data=data,
            headers=self.headers,
            verify=self.ssl_verify,
        )
        return self._check_response(r, "PUT")

    def _make_req_post(self, url, data=None):
        r = requests.post(
            self._srv + url,
            timeout=self.timeout,
            data=data,
            headers=self.headers,
            verify=self.ssl_verify,
        )
        return self._check_response(r, "POST")

    def _make_req_del(self, url, data=None):
        r = requests.delete(
            self._srv + url,
            timeout=self.timeout,
            data=data,
            headers=self.headers,
            verify=self.ssl_verify,
        )
        if r.status_code == requests.codes.no_content:
            return None
        return self._check_response(r, "DELETE")
