#!/usr/bin/env python

from __future__ import print_function

import six

import logging
import json
import time

from sandbox.projects.common import decorators
from sandbox.projects.common.betas.api_interface import ApiCommon
import sandbox.projects.common.link_builder as lb


class YappyApi(ApiCommon):
    PATCHES_DIR = "arcadia:/arc/trunk/arcadia/search/priemka/yappy/data/patches"

    def check_status(self, beta_name, statuses):
        if isinstance(statuses, six.string_types):
            statuses = [statuses]
        got_status = self.get_beta_state(beta_name).get('status', 'STOPPED')
        self.logger.info("Beta %s is in %s status. Desired statuses: %s", beta_name, got_status, statuses)
        return got_status in statuses

    def sleep_for_status(self, beta_name, status, retries=8, delay=20, multiplier=1.6):
        sleep_time = delay
        for attempt in range(1, retries + 1):
            if self.check_status(beta_name, status):
                return True
            self.logger.info("Sleep %s attempt (%s seconds) for status", attempt, sleep_time)
            time.sleep(sleep_time)
            sleep_time = int(sleep_time * multiplier)
        return self.check_status(beta_name, status)

    def process_lineage_request(self, method, data):
        r = self._make_req_post('/api/yappy.services.Lineage2/{}'.format(method), data=data)

        if r.get('status') != 'SUCCESS':
            raise RuntimeError('{} failed:\n{}'.format(method, r))
        return r

    @decorators.retries(4, delay=5, default_instead_of_raise=False)
    def beta_exists(self, beta_name):
        """
        Check if beta exists in Yappy
        :param beta_name: beta name
        :return: True/False (to keep API consistent with sdms_api)
        """
        return self._beta_exists(beta_name)

    @decorators.retries(4, delay=5)
    def beta_exists_no_defaults(self, beta_name):
        """
        Same as `beta_exists` but do not force False by default: raises exception if non-2xx response received.

        :param beta_name: beta name
        :return: True/False
        """
        return self._beta_exists(beta_name)

    def _beta_exists(self, beta_name):
        """
        Check if beta exists in Yappy
        :return: True/False (to keep API consistent with sdms_api)
        """
        return self._make_req_post(
            '/api/yappy.services.Model/betaExists',
            data=json.dumps({'name': beta_name}),
        ).get('exists', False)

    @decorators.retries(4, delay=5)
    def get_beta_info(self, beta_name):
        """
        Get info beta from Yappy
        :return: *almost* the same as sdms_api

        The only difference between sdms_api and yappy_api atm: STOPPPED (sdms_api) -> STOPPED (yappy_api)
        """
        return self._make_req_post('/api/yappy.services.Model/getBetaStatus', data=json.dumps({'name': beta_name}))

    @decorators.retries(4, delay=5)
    def get_beta_state(self, beta_name):
        """
            Get 'state' portion of beta info or all beta info if 'state' doesn't exist
        """
        # atm yappy doesn't have designated api method for this
        beta_info = self.get_beta_info(beta_name)
        return beta_info.get('state') or beta_info

    @decorators.retries(4, delay=5)
    def start_beta(self, beta_name):
        if not (self.beta_exists(beta_name) and self.check_status(beta_name, ["CONSISTENT", "INCONSISTENT"])):
            # beta was not allocated, so allocate it!
            self.process_lineage_request('allocateBeta', json.dumps({'name': beta_name}))

        if not self.sleep_for_status(beta_name, "CONSISTENT"):
            raise RuntimeError("Failed to start beta")

    @decorators.retries(4, delay=5)
    def stop_beta(self, beta_name):
        if not self.beta_exists(beta_name) or self.check_status(beta_name, "STOPPED"):
            self.logger.info("No need to stop beta: %s", beta_name)
            return

        self.process_lineage_request('deallocateBeta', json.dumps({'name': beta_name}))

        if not self.sleep_for_status(beta_name, "STOPPED"):
            raise RuntimeError("Failed to stop beta")

    @decorators.retries(4, delay=5)
    def get_instances(self, beta_name, source):
        r = self._make_req_post(
            '/api/yappy.services.Model/retrieveBeta',
            data=json.dumps({'name': beta_name, 'withHistory': False})
        )
        logging.debug(
            "All sources for %s: %s",
            beta_name,
            [c.get('type', {}).get('name', '') for c in r.get('components', [])]
        )
        for component in r.get('components', []):
            if component.get('type', {}).get('name', '') == source:
                if not component.get('slot'):
                    logging.debug('NO SLOT!!! Response is:')
                    logging.debug(json.dumps(r, indent=2))
                return component.get('slot', {}).get('instances', [])
        return []

    @decorators.retries(4, delay=5)
    def get_beta_domain_name_without_tld(self, beta_name):
        r = self._make_req_post('/api/yappy.services.Model/retrieveBeta', data=json.dumps({'name': beta_name}))
        domain_name_without_tld = r.get('domainNameWithoutTld', '')
        logging.debug('Domain name without tld: %s', domain_name_without_tld)
        return domain_name_without_tld

    @decorators.retries(4, delay=5)
    def create_beta_from_template(self, template_name, suffix, patch_map):
        api_spec = {'sandbox_id': lb.task_wiki_link(self.id)}
        patches = [{'component_id': k, 'patch': v} for k, v in patch_map.items()]

        self.process_lineage_request(
            'createBetaFromBetaTemplate',
            json.dumps({'template_name': template_name, 'suffix': suffix, 'patches': patches, 'api_spec': api_spec}),
        )

    @decorators.retries(4, delay=5)
    def retrieve_api_beta_template(self, template_name):
        return self._make_req_post(
            '/api/yappy.services.Lineage2/retrieveApiBetaTemplate',
            data=json.dumps({'name': template_name}),
        )

    @decorators.retries(4, delay=5)
    def retrieve_api_beta(self, beta_name):
        return self._make_req_post(
            '/api/yappy.services.Lineage2/retrieveApiBeta',
            data=json.dumps({'name': beta_name}),
        )

    @decorators.retries(4, delay=5)
    def update_patches(self, beta_name, patches):
        api_spec = {'sandbox_id': lb.task_wiki_link(self.id)}

        self.process_lineage_request(
            'updatePatches',
            data=json.dumps({'beta_name': beta_name, 'patches': patches, 'api_spec': api_spec}),
        )


if __name__ == '__main__':
    logging.basicConfig(level='DEBUG')
    api = YappyApi('https://yappy.z.yandex-team.ru/', ssl_verify=False)

    print(api.beta_exists('hamster'))
    print(api.beta_exists('rd-demo'))
    print(api.beta_exists('hamster111'))
    print(api.get_beta_info('hamster'))
    print(api.get_beta_state('hamster'))
    print(api.get_beta_info('hamster111'))
