import os
import re
import logging
import collections

from sandbox.sandboxsdk.parameters import (
    LastReleasedResource,
    SandboxArcadiaUrlParameter,
    SandboxStringParameter,
)
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.utils import check_if_tasks_are_ok
import sandbox.projects.common.build.parameters as build_params


def snake2camel(name):
    return re.sub(r"^(\w)|_(\w)", lambda n: n.group(1).upper() if n.group(1) else n.group(2).upper(), name)


class RequiredBinary(object):
    required = True

    def __init__(self, project_path, art_name):
        self.project_path = project_path
        self.art_name = art_name

    def __hash__(self):
        return hash(self.project_path)

    def __eq__(self, other):
        return self.project_path == other.project_path

    def local_path(self, task):
        return task.ctx.get(self.art_name + "_executable")

    def get_task_params(self):
        class_prefix = snake2camel(self.art_name)
        resource_type = type(
            class_prefix + "ExecResourceParameter",
            (LastReleasedResource,),
            {
                "bin_name": self.art_name,
                "name": self.art_name + "_resid",
                "description": self.art_name + " resource id:",
                "required": False,
                "default_value": None  # do not fetch unnecessary released resources
            },
        )

        selector = type(
            class_prefix + "ObtainSelector",
            (SandboxStringParameter,),
            {
                "name": self.art_name + "_selector",
                "description": "Get " + self.art_name + " from:",
                "choices": [
                    ("Any", None),
                    ("resource", "resource"),
                    ("build", "build")
                ],
                "sub_fields": {
                    "resource": [resource_type.name],
                    "build": []
                },
                "default_value": "resource",
                "required": False,
            },
        )
        return [selector, resource_type]


class OptionalBinary(RequiredBinary):
    required = False


class ArcadiaSourceParameter(SandboxArcadiaUrlParameter):
    name = "arcadia_url"
    description = "Build binaries from this branch:"
    required = False
    default_value = Arcadia.trunk_url()


class BinariesProvider(object):
    def __init__(self, binaries, message="Building requirements for parent task", group=None):
        self._targets = binaries
        self._subtask_description = message
        self._group = group or build_params.BASE_BUILD_GROUP_NAME
        self._task_parameters = self.create_task_parameters()

    def set_subtask_description(self, msg):
        self._subtask_description = msg

    def create_task_parameters(self):
        out = [ArcadiaSourceParameter, build_params.ArcadiaPatch]
        for b in self._targets:
            out += b.get_task_params()
        for param in out:
            param.group = self._group
        return out

    def task_parameters(self):
        return self._task_parameters

    def fetch_prebuild_binaries(self, task):
        resources = collections.defaultdict(list)
        for binary in self._targets:
            select = task.ctx.get(binary.art_name + "_selector")
            if select is None and binary.required:
                raise SandboxTaskFailureError("Required binary {} not provided".format(binary.name))

            if select == "resource":
                resid = task.ctx[binary.art_name + "_resid"]
                if resid:
                    resources[resid].append(binary.art_name)
                elif binary.required:
                    raise SandboxTaskFailureError("Required binary {} not provided".format(binary.name))

        to_build = self._get_build_targets(task)

        if to_build:
            lst = channel.sandbox.list_resources(
                resource_type="ARCADIA_PROJECT",
                task_id=task.ctx["build_subtask_id"]
            )
            if len(lst) != 1:
                raise SandboxTaskFailureError(
                    "Task {} has no resource ARCADIA_PROJECT (or has many)".format(task.ctx["build_subtask_id"])
                )
            build_resource = lst[0]
            if not build_resource.is_ready():
                raise SandboxTaskFailureError("Resource {} is not ready (probably failed".format(build_resource.id))

            for binary in to_build:
                resources[build_resource.id].append(binary.art_name)

        for resid, binaries in resources.items():
            resource = channel.sandbox.get_resource(resid)
            if not resource.is_ready():
                raise SandboxTaskFailureError("Resource {} is not ready (probably failed".format(resid))
            path = task.sync_resource(resid)
            for name in binaries:
                real_name = path
                if os.path.exists(os.path.join(path, name)):
                    real_name = os.path.join(path, name)
                if not os.path.isfile(real_name):
                    raise SandboxTaskFailureError("{} is not an executable file".format(real_name))
                logging.info("using path {} for executable {}".format(real_name, name))
                task.ctx[name + "_executable"] = real_name

    def _get_build_targets(self, task):
        res = []
        for binary in self._targets:
            select = task.ctx.get(binary.art_name + "_selector")
            if select is None and binary.required:
                raise SandboxTaskFailureError("Required binary {} not provided".format(binary.art_name))
            if select == "build":
                res.append(binary)
        return res

    def run_build_task(self, task):
        if "build_subtask_id" in task.ctx or task.ctx.get("build_subtask_done"):
            return
        logging.info("going to run build task")

        to_build = self._get_build_targets(task)

        if not to_build:
            logging.info("nothing to build, skipping")
            task.ctx["build_subtask_done"] = True
            return

        input_parameters = {
            "checkout_arcadia_from_url": task.ctx["arcadia_url"],
            "arcadia_patch": task.ctx.get(build_params.ArcadiaPatch.name),
            "clear_build": False,
            "targets": ";".join([b.project_path for b in to_build]),
            "arts": ";".join([os.path.join(b.project_path, b.art_name) for b in to_build])
        }
        subtask = task.create_subtask(
            task_type="YA_MAKE",
            description=self._subtask_description,
            input_parameters=input_parameters,
            execution_space=80000
        )
        logging.info("started task {} YA_MAKE for {}".format(subtask.id, input_parameters["targets"]))
        task.ctx["build_subtask_id"] = subtask.id

    def join_build_task(self, task):
        if task.ctx.get("build_subtask_done"):
            return
        task.wait_all_tasks_completed_restart_once([task.ctx["build_subtask_id"]])
        task.ctx["build_subtask_done"] = True
        check_if_tasks_are_ok([task.ctx["build_subtask_id"]])

    def build_binaries(self, task):
        self.run_build_task(task)
        self.join_build_task(task)

    def fetch_binaries(self, task):
        self.build_binaries(task)
        self.fetch_prebuild_binaries(task)
