import copy
import logging
import six

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common import kosher_release

logger = logging.getLogger(__name__)


class LastBinaryTaskRelease(object):
    """
    Mixin class to automatically find last resource with binary task executor according to actual parameter value
    and setup proper task requirement. Overrides on_save and on_execute methods.
    Search of the resource is performed by "task_type" attribute which should be explicitly set and
    "released" attribute which is set automatically when resource is released (see
    https://wiki.yandex-team.ru/sandbox/releases/). To configure resource search,
    you need to override the binary_executor_query property, which should return a dictionary with valid arguments
    for sdk2.Resource.find().

    Parameters of the task being mixed should either inherit LastBinaryReleaseParameters or include it.

    One can use module methods instead of inheriting mixin class.
    """

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": self.type.name, "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def on_save(self):
        setup_requirement(self)

    def on_execute(self):
        validate_resource_executor(self)


def binary_release_parameters(stable=False, prestable=False, testing=False, unstable=False, none=False, custom=False, testing_or_upper=False):
    assert len(list(filter(lambda x: bool(x), (stable, prestable, testing, unstable, none, custom, testing_or_upper)))) == 1, ""

    class _LastBinaryReleaseParameters(sdk2.Parameters):
        with sdk2.parameters.RadioGroup(
            "Release type of binary executor resource", group="Binary task executor"
        ) as binary_executor_release_type:
            binary_executor_release_type.values.stable = binary_executor_release_type.Value(
                ctt.ReleaseStatus.STABLE, default=stable)
            binary_executor_release_type.values.prestable = binary_executor_release_type.Value(
                ctt.ReleaseStatus.PRESTABLE, default=prestable)
            binary_executor_release_type.values.testing = binary_executor_release_type.Value(
                ctt.ReleaseStatus.TESTING, default=testing)
            binary_executor_release_type.values.unstable = binary_executor_release_type.Value(
                ctt.ReleaseStatus.UNSTABLE, default=unstable)
            binary_executor_release_type.values.none = binary_executor_release_type.Value("none", default=none)
            binary_executor_release_type.values.custom = binary_executor_release_type.Value("custom", default=custom)
            binary_executor_release_type.values.testing_or_upper = binary_executor_release_type.Value("testing_or_upper", default=testing_or_upper)

    return _LastBinaryReleaseParameters()


class LastBinaryReleaseParameters(sdk2.Parameters):
    _lbrp = binary_release_parameters(custom=True)


BINARY_TYPES = [
    ctt.ReleaseStatus.STABLE,
    ctt.ReleaseStatus.PRESTABLE,
    ctt.ReleaseStatus.TESTING,
    ctt.ReleaseStatus.UNSTABLE,
    'none',
    'custom',
    'last_successful_stable',
]


def binary_release_parameters_list(
    stable=False,
    prestable=False,
    testing=False,
    unstable=False,
    none=False,
    custom=False,
    last_successful_stable=False,
):
    kws = locals().copy()
    release = None

    for _release, value in six.iteritems(kws):
        if value:
            if release:
                raise ValueError('Only one of {} can be True'.format(BINARY_TYPES))
            else:
                release = _release
    if not release:
        raise ValueError('Release must be one of {}'.format(BINARY_TYPES))

    class _LastBinaryReleaseParameters(sdk2.Parameters):
        with sdk2.parameters.Group('Binary task executor') as binary_executor_release_type_group:
            binary_executor_release_type = sdk2.parameters.String(
                'Release type of binary executor resource',
                required=True,
                default=release,
                choices=[(t, t) for t in BINARY_TYPES], ui=sdk2.parameters.String.UI('select')
            )

    return _LastBinaryReleaseParameters()


def setup_requirement(task):
    if task.Parameters.binary_executor_release_type == "custom":
        return

    if task.Parameters.binary_executor_release_type == "none":
        task.Requirements.tasks_resource = None
    elif task.Parameters.binary_executor_release_type == "last_successful_stable":
        last_success_stable_task = sdk2.Task.find(
            task_type=task.type,
            status=ctt.Status.Group.SUCCEED,
            input_parameters={"binary_executor_release_type": "stable"}
        ).first()
        task.Requirements.tasks_resource = (
            last_success_stable_task.Requirements.tasks_resource if last_success_stable_task else None
        )
    elif task.Parameters.binary_executor_release_type == "testing_or_upper":
        query = copy.deepcopy(task.binary_executor_query)
        for env in (ctt.ReleaseStatus.TESTING, ctt.ReleaseStatus.PRESTABLE, ctt.ReleaseStatus.STABLE,):
            if 'attrs' not in query:
                query['attrs'] = {}
            query['attrs']['released'] = str(env)
            tasks_resource = (
                sdk2.service_resources.SandboxTasksBinary.find(**query).first()
            )
            if tasks_resource:
                task.Requirements.tasks_resource = tasks_resource
                break
        else:
            task.Requirements.tasks_resource = None
    else:
        task.Requirements.tasks_resource = (
            sdk2.service_resources.SandboxTasksBinary.find(**task.binary_executor_query).first()
        )

    validate_resource_executor(task)


def validate_resource_executor(task):
    if (
        task.Parameters.binary_executor_release_type not in ("none", "custom") and
        task.Requirements.tasks_resource is None
    ):
        raise errors.TaskFailure(
            "Can't find any SandboxTasksBinary resource for task type {} with attribute released=={}".format(
                task.type.name, task.Parameters.binary_executor_release_type
            )
        )


class LastRefreshableBinary(LastBinaryTaskRelease):
    """
    Class to automatically find and update last resource with binary task after rerunning.
    Overrides on_enqueue method. History is saved in Context in `tasks_resource_history` field.
    By default doesn't refresh resource after waiting subtasks
    """

    _refresh_on_wait = False

    def on_enqueue(self):
        """
        :param sdk2.Task self:
        """
        if not self.Context._refresh:
            self.Context._refresh = True
            return

        old_tasks_resource = self.Requirements.tasks_resource
        setup_requirement(self)
        if self.Requirements.tasks_resource != old_tasks_resource:
            if not self.Context.tasks_resource_history:
                self.Context.tasks_resource_history = [old_tasks_resource and old_tasks_resource.id]
            self.Context.tasks_resource_history.append(
                self.Requirements.tasks_resource and self.Requirements.tasks_resource.id
            )

    def on_wait(self, prev_status, status):
        if not self._refresh_on_wait:
            self.Context._refresh = False


class LastBinaryTaskKosherReleaseMixin(LastBinaryTaskRelease):
    """
    A "kosher" version of LastBinaryTaskRelease based on the so called "kosher release scheme"
    (aka Kvasov-Korum scheme).

    - https://a.yandex-team.ru/arc_vcs/sandbox/projects/common/kosher_release
    - https://st.yandex-team.ru/RMDEV-2380

    The key difference between this class and its parent is in the way of resource resolving.
    LastBinaryTaskKosherRelease uses kosher release scheme.

    Key features:

        - Independent release stages. E.g., once released to testing a resource remains released to testing even
        after its stable release (and thus it will be recognized as latest testing release until a newer release
        to testing occurs)
        - Forward rollbacks. When you need to roll your release back to a particular version you just need to
        release the appropriate resource to the desired stage. This will do the job. Additionally it is possible to
        cancel a bad release (mark the respective resource as bad)

    Note:

        - NOT compatible with original Sandbox releases (Sandbox-only releases won't be detected)
        - 'testing_or_upper' stage is NOT supported (see https://st.yandex-team.ru/RMDEV-3194#6267d9e17205116ce6b58f51)
        - 'testing_or_upper' stage when selected is replaced with 'testing'
    """

    K_RELEASE__RESOURCE_TYPE = "SANDBOX_TASKS_BINARY"
    K_RELEASE__TASK_TYPE_ATTR = "task_type"
    K_RELEASE__REPLACE_TESTING_OR_UPPER_WITH = "testing"

    def _setup_task_requirement(self):
        if self.Parameters.binary_executor_release_type == "custom":
            return

        if self.Parameters.binary_executor_release_type == "none":
            self.Requirements.tasks_resource = None

        if self.Parameters.binary_executor_release_type == "testing_or_upper":
            self.Parameters.binary_executor_release_type = self.K_RELEASE__REPLACE_TESTING_OR_UPPER_WITH

        self.Requirements.tasks_resource = kosher_release.find_release(
            resource_type=self.K_RELEASE__RESOURCE_TYPE,
            stage=self.Parameters.binary_executor_release_type,
            sb_rest_client=self.server,
            custom_attrs={
                self.K_RELEASE__TASK_TYPE_ATTR: self.type.name,
            },
        )

        self._validate_resource_executor()

    def _validate_resource_executor(self):
        if (
            self.Parameters.binary_executor_release_type not in ("none", "custom") and
            self.Requirements.tasks_resource is None
        ):
            raise errors.TaskFailure(
                "Can't find any {} resource with attributes `{}={}` and `{}` with ISO 8601 datetime".format(
                    self.K_RELEASE__RESOURCE_TYPE,
                    self.K_RELEASE__TASK_TYPE_ATTR,
                    self.type.name,
                    kosher_release.private.get_release_attr_name(self.Parameters.binary_executor_release_type),
                )
            )

    def on_save(self):
        self._setup_task_requirement()
