import logging

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt


logger = logging.getLogger(__name__)


class LastBinaryTaskRelease(object):
    """
    Mixin class to automatically find last resource with binary task executor according to actual parameter value
    and setup proper task requirement. Overrides on_save and on_execute methods.
    Search of the resource is performed by "task_type" and "release" attributes which should be explicitly set.

    Parameters of the task being mixed should either inherit LastBinaryReleaseParameters or include it.

    One can use module methods instead of inheriting mixin class.
    """

    def on_save(self):
        setup_requirement(self)

    def on_execute(self):
        validate_resource_executor(self)


def binary_release_parameters(stable=False, test=False, none=False):
    assert len(list(filter(lambda x: bool(x), (stable, test, none)))) == 1, ""

    class _LastBinaryReleaseParameters(sdk2.Parameters):
        with sdk2.parameters.RadioGroup("Release type of binary executor resource", group="Binary task executor") as binary_executor_release_type:
            binary_executor_release_type.values.stable = binary_executor_release_type.Value(ctt.ReleaseStatus.STABLE, default=stable)
            binary_executor_release_type.values.test = binary_executor_release_type.Value(ctt.ReleaseStatus.TESTING, default=test)
            binary_executor_release_type.values.none = binary_executor_release_type.Value("none", default=none)
            binary_executor_release_type.values.custom = binary_executor_release_type.Value("custom", default=False)

    return _LastBinaryReleaseParameters()


class LastBinaryReleaseParameters(sdk2.Parameters):
    _lbrp = binary_release_parameters(none=True)


def setup_requirement(task):
    if task.Parameters.binary_executor_release_type == "custom":
        return
    task.Requirements.tasks_resource = None if task.Parameters.binary_executor_release_type == "none" else \
        sdk2.service_resources.SandboxTasksBinary.find(
            attrs={"task_type": task.type.name, "release": task.Parameters.binary_executor_release_type},
            state=[ctr.State.READY],
        ).first()


def validate_resource_executor(task):
    if task.Parameters.binary_executor_release_type != "none" and task.Requirements.tasks_resource is None:
        raise errors.TaskFailure("Can't find any SandboxTasksBinary resource for task type {} with attribute release=={}".format(task.type.name, task.Parameters.binary_executor_release_type))
