import itertools
import json
import os
import urllib2

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.bno.http import fetch
from sandbox.projects.common.bno.params import EnvTypeParam
from sandbox.projects.common.bno.resources import save_resource
from sandbox.projects.common.bno.utils import run_cmd
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.task import SandboxTask
import upload_trie_to_saas


class VpsHost(sdk_parameters.SandboxStringParameter):
    name = 'vps_host'
    description = 'Vps host'
    default_value = 'vps-beta.n.yandex-team.ru'


class VaultTokenItemNameParam(sdk_parameters.SandboxStringParameter):
    name = 'vault_item'
    description = 'Vault item name for mr token'
    default_value = 'mrtoken'
    required = True


class YtExportPath(sdk_parameters.SandboxStringParameter):
    name = 'yt_export_path'
    description = 'Yt export path'
    default_value = ':'


class UploadToSaas(sdk_parameters.SandboxBoolParameter):
    name = 'upload_to_saas'
    description = 'upload data to saas instead of querysearch'
    default_value = False


class SaasNamespace(sdk_parameters.SandboxStringParameter):
    name = 'saas_namespace'
    description = 'saas namespace'
    default_value = ''


class SaasRootFolder(sdk_parameters.SandboxStringParameter):
    name = 'saas_root_folder'
    description = 'saas root folder'
    default_value = upload_trie_to_saas.DEFAULT_ROOT


class BnoBaseBuildTask(SandboxTask):
    input_parameters = [EnvTypeParam, VpsHost, VaultTokenItemNameParam, YtExportPath, UploadToSaas, SaasNamespace, SaasRootFolder]
    keysemantic = 'snipdocid'

    environment = (
        environments.PipEnvironment("yandex-yt"),
        environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
    )

    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.Group.LINUX

    def __init__(self, task_id=0):
        SandboxTask.__init__(self, task_id)

    def deploy_data(self, path, host):
        rsync_path = "{}::querydata{}/".format(host, '' if self.ctx[EnvTypeParam.name] == 'production' else '-beta')
        run_cmd(['tar', '-xf', path])
        run_cmd(['rsync', '-vr', '000', rsync_path])
        run_cmd(['rsync', '-vr', '001', rsync_path])
        run_cmd(['rsync', '-vr', self.namespace + '.trie.tag', rsync_path])

    def build_data(self, data):
        tool = utils.sync_last_stable_resource(resource_types.BNO_QUERYDATA_INDEXER_EXECUTABLE, arch='linux')
        path = self.path('deploy')
        if not os.path.exists(path):
            os.makedirs(path)
        trie_path = self.namespace + '.trie'
        run_cmd(
            [tool, "--enable-keyref", "-E", "-w", "-i", data, "-D", os.path.join(path, trie_path), "-N",
             self.keysemantic,
             "-S", "docid_setprops/" + self.namespace, "-n", "2", "-f", "4096,32768"])
        trie_path += '.tar'
        run_cmd(['tar', '-C', path, '-cf', trie_path, '.'])
        return save_resource(self, path=trie_path, resource_type=resource_types.BNO_QUERYDATA_TRIE)

    def build_value(self, value):
        plugin = self.content_plugin.copy()
        plugin[self.namespace]['SerpData']['viewport'] = (json.loads(value))
        return 'Snippet=' + json.dumps(plugin, ensure_ascii=False)

    def prepare_data(self, data):
        data = [(item[2], self.build_value(item[-1])) for item in data]
        return data, data and save_resource(self, data, self.namespace + '.txt') or None

    def build_queries(self):
        raise NotImplementedError

    def fetch_data(self, urls):
        urls = [(x + y) for x, y in itertools.izip(urls, fetch([url[-1] for url in urls]))]
        correct = [url for url in urls if url[-2] == 200]
        errors = [url for url in urls if url[-2] != 200]
        save_resource(self, errors, 'vps.errors.txt')
        return correct, save_resource(self, correct, 'vps.txt')

    def export_to_yt(self, data):
        from yt.wrapper import YtClient
        table = self.ctx[YtExportPath.name]
        cluster, path = table.split(':')
        vault_item = self.ctx[VaultTokenItemNameParam.name]
        token = self.get_vault_data(vault_item)
        if cluster and path and token:
            client = YtClient(cluster, token)
            rows = (dict(('col%s' % (k, ), v) for k, v in enumerate(item)) for item in data)
            client.write_table(os.path.join(path, self.namespace), rows)

    @staticmethod
    def get_querydata_shares(custom_hosts):
        return []
        # don't upload news_bno trie because of saas-kv
        try:
            querydata_shares = "http://querydata.n.yandex-team.ru/get_trie_hosts/"
            bno_trie_name = "news_bno.trie"
            return urllib2.urlopen("".join((querydata_shares, bno_trie_name))).read().split()
        except:
            return custom_hosts

    def upload_to_saas(self, trie_resource):
        saas_namespace = self.ctx[SaasNamespace.name]
        saas_root_folder = self.ctx[SaasRootFolder.name]
        saas_target = self.ctx[EnvTypeParam.name]
        vault_item = self.ctx[VaultTokenItemNameParam.name]
        yt_token = self.get_vault_data(vault_item)

        fields = saas_namespace.split(':', 1)
        if len(fields) != 2 or fields[-1] not in ('docid_setprops', 'categflag'):
            raise Exception('Expected namespace format: some_name:docid_setprops or some_name:categflag')

        uploader = upload_trie_to_saas.Uploader(ns_name=saas_namespace, root_folder=saas_root_folder, yt_token=yt_token)
        with open(trie_resource.path) as f:
            uploader.upload_table(f=f, with_tld=False)
        uploader.deliver_table(env=saas_target)

    def on_execute(self):
        with self.current_action('Building queries'):
            urls, _ = self.build_queries()
        with self.current_action('Fetching data'):
            urls, _ = self.fetch_data(urls)
        with self.current_action('Prepare data'):
            _, resource = self.prepare_data(urls)

        if self.ctx[UploadToSaas.name]:
            with self.current_action('Uploading data to SAAS'):
                self.upload_to_saas(resource)
        else:
            with self.current_action('Building data'):
                resource = self.build_data(resource.path)
            for host in self.get_querydata_shares(['ws37-754.search.yandex.net', 'ws31-437.search.yandex.net']):
                with self.current_action('Deploying data to ' + host):
                    self.deploy_data(resource.path, host)
        # with self.current_action('Export data to yt'):
        #    self.export_to_yt(urls)

    def on_release(self, additional_parameters):
        SandboxTask.on_release(self, additional_parameters)
