# -*- coding: utf-8 -*-

"""
    Модуль с общими классами для задач сборки из Аркадии
"""

import re
import os
import copy
import shlex
import logging
import platform
import tempfile

from sandbox import common
import sandbox.common.types.misc as ctm

import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts

from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import task as sandbox_task
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects.common import apihelpers


LIST_LIMIT = 10
VAULT_PATTERN = r'\$\(vault:(?P<dst>file|value):(?P<owner>[^:]+):(?P<name>[^:]+)\)'
ENV_VAR_PATTERN = r'^\s*(\b[a-zA-Z_]\w*=((\'[^\']*\')|(\"[^\"]*\")|([^\s\'\"]+))(\s+|$))+$'
TMP_STORE = []


class UseObjCache(parameters.SandboxBoolParameter):
    name = 'use_obj_cache'
    description = 'Use cached object files'
    group = 'Common build params'
    default_value = False


def create_input_class(class_name):
    """
        @param class_name: задается маленькими буквами, слова разделяются подчеркиванием
    """
    class InputClass(parameters.SandboxBoolParameter):
        name = 'build_{}'.format(class_name)
        description = '---> build {}'.format(class_name)
        hidden = None
        group = 'Projects'

    return InputClass


def create_info_class(class_name, class_sub_fields):
    """
        @param class_name: задаётся маленькими буквами, слова разделяются подчеркиванием
        @param class_sub_fields: список подполей
    """
    targets = ['build_{}'.format(target) for target in class_sub_fields]

    class InfoClass(parameters.SandboxStringParameter):
        name = class_name
        description = class_name.replace('_', ' ')
        choices = [
            ('build_all', 'build_all'),
            ('choose', 'choose')
        ]
        default_value = 'choose'
        sub_fields = {
            'choose': targets,
        }
        group = 'Projects'

    return InfoClass


def gen_input_params(build_fields):
    params = []
    for name, sub_fields in build_fields.items():
        params.append(create_info_class(name, sub_fields))
        for sub_field in sub_fields:
            params.append(create_input_class(sub_field))

    return params


def on_enqueue_input_params(ctx, build_fields):
    for name, sub_fields in build_fields.items():
        if ctx.get(name) == 'build_all':
            for sub_field in sub_fields:
                ctx['build_{}'.format(sub_field)] = True


def modify_resources_for_win32(resources):
    result = []
    for res_type, target, path in resources:
        _, ext_target = os.path.splitext(target)
        _, ext_result = os.path.splitext(path)
        new_target_name = target
        new_path = path
        if not ext_target:
            new_target_name += '.exe'
        if not ext_result:
            new_path += '.exe'
        logging.info('Rename: {} -> {}'.format(target, new_target_name))
        logging.info('Rename: {} -> {}'.format(path, new_path))
        result.append((res_type, new_target_name, new_path))
    return result


class ArcadiaTask(sandbox_task.SandboxTask):
    """
        Базовый класс для задач сборки из Аркадии
    """
    environment = (environments.SvnEnvironment(), )

    build_fields = {}

    input_parameters = build_params.get_arcadia_params()

    # дополнительные параметры для контекста подзадач при мультиархитектурной сборке
    # ключ - название архитектуры; например, если хотим, чтобы linux-сборка запустилась
    # с дополнительными параметрами, пишем что-то вроде
    # additional_bundle_params = {'linux': {'param1': 1, 'param2': 2}}
    additional_bundle_params = {}

    TARGET_PATH_TO_NAME_MAP = {}
    TARGET_RESOURCES = []

    wait_build_bundle_tasks = []
    arcadia_src_dir = None

    def get_target_name(self, path):
        return self.TARGET_PATH_TO_NAME_MAP.get(path, os.path.basename(path))

    def on_enqueue(self):
        sandbox_task.SandboxTask.on_enqueue(self)
        checkout_arcadia_from_url = self.ctx.get(consts.ARCADIA_URL_KEY)
        if checkout_arcadia_from_url and checkout_arcadia_from_url[-1] == '/':
            self.ctx[consts.ARCADIA_URL_KEY] = self.ctx[consts.ARCADIA_URL_KEY][0:-1]
        if common.platform.get_arch_from_platform(self.arch) == ctm.OSFamily.OSX:
            self.cores = 4

    def is_build_bundle_task(self):
        """
            Является ли задача мета-задачей для сборки под все архитектуры

            :return: True, если является; False в противном случае
        """
        return bool(self.ctx.get(consts.BUILD_BUNDLE_KEY))

    def get_target_resources(self):
        """
            Список описаний (тип ресурса, имя результата, путь к собранному ресурсу в Аркадии), которые нужно собрать.
            Включает в себя описания из self.TARGET_RESOURCE_TYPES с путями, взятыми из метаданных ресурсов,
            и описания из self.TARGET_RESOURCES как есть.
        """
        result = []
        for resource_type in getattr(self, 'TARGET_RESOURCE_TYPES', []):
            arcadia_build_name = getattr(
                resource_type, 'arcadia_build_name', self.get_target_name(resource_type.arcadia_build_path)
            )
            result.append((resource_type, arcadia_build_name, resource_type.arcadia_build_path))

        for item in getattr(self, 'TARGET_RESOURCES', []):
            if len(item) == 2:
                resource_type, arcadia_build_path = item
                result.append((resource_type, self.get_target_name(arcadia_build_path), arcadia_build_path))
            else:
                result.append(item)

        target_platform = self.ctx.get('target_platform_alias') or platform.platform()
        return result if target_platform not in ['win32', 'cygwin'] else modify_resources_for_win32(result)

    def check_build_bundle_worker(self, arch):
        """
            Проверить задачи сборки под определённую архитектуру

            :param arch: название архитектуры
        """
        worker_id = self.ctx.get('{0}_worker_task_id'.format(arch))
        if not worker_id:
            sub_ctx = copy.deepcopy(self.ctx)
            # подзадачи не должны быть build_bundle, иначе возникнет рекурсия
            sub_ctx[consts.BUILD_BUNDLE_KEY] = False
            sub_ctx['notify_via'] = ''
            sub_ctx['parent_id_for_resources'] = str(self.id)
            # устанавливаем дополнительные параметры для архитектуры, если есть
            if arch in self.additional_bundle_params:
                sub_ctx.update(self.additional_bundle_params[arch])
            if '_log_resource' in sub_ctx:
                del sub_ctx['_log_resource']
            subtask = self.create_subtask(
                task_type=self.type,
                description="{0} worker for task #{1}: {2}".format(arch, self.id, self.descr),
                input_parameters=sub_ctx,
                arch=arch,
                important=self.important
            )
            self.ctx['{0}_worker_task_id'.format(arch)] = subtask.id
            self.wait_build_bundle_tasks.append(subtask.id)
        else:
            build_task = channel.sandbox.get_task(worker_id)
            if build_task.is_failure():
                raise errors.SandboxTaskFailureError(
                    'Worker task:{0} is failed. '
                    'Cannot complete the task successfully. '
                    'Try to check the worker logs.'.format(worker_id)
                )
            elif build_task.is_deleted():
                raise errors.SandboxTaskFailureError(
                    'Worker task:{0} is deleted. '
                    'Cannot complete the task successfully. '.format(worker_id)
                )
            elif build_task.is_finished():
                # если сборка успешно завершилась, собираем ресурсы
                offset = 0
                while True:
                    task_resources = apihelpers.list_task_resources(
                        build_task.id, limit=LIST_LIMIT, offset=offset
                    ) or []
                    for resource in task_resources:
                        if resource.type == 'TASK_LOGS':
                            continue
                        if resource.arch == 'any' and build_task.arch != self.arch:
                            continue
                        src_path = self.sync_resource(resource.id)
                        dst_path = os.path.join(resource.arch, resource.file_name)
                        paths.copy_path(src_path, dst_path)
                        copy_resource = self.create_resource(
                            resource.description, dst_path, str(resource.type),
                            arch=resource.arch,
                            attributes=resource.attributes
                        )
                        logging.info('Resource %s was copied to %s', resource.id, copy_resource)
                    if len(task_resources) < LIST_LIMIT:
                        break
                    offset += LIST_LIMIT
            else:
                self.wait_build_bundle_tasks.append(worker_id)

    def check_build_bundle_tasks(self):
        """
            Проверить задачи сборки под все архитектуры
        """
        self.wait_build_bundle_tasks = []
        for arch in self.archs_for_bundle:
            self.check_build_bundle_worker(arch)
        if self.wait_build_bundle_tasks:
            self.wait_all_tasks_completed(self.wait_build_bundle_tasks)

    def on_execute(self):
        """
            Запускаем сразу сборку или создание подзадач, если указан параметр build_bundle
        """
        checkout_arcadia_from_url = self.ctx.get(consts.ARCADIA_URL_KEY)
        if not checkout_arcadia_from_url:
            eh.check_failed('{} parameter is not specified'.format(consts.ARCADIA_URL_KEY))
        elif "/arcadia" not in checkout_arcadia_from_url:
            logging.info("WARNING! Path should contain '/arcadia' word")
            # eh.fail('Wrong svn path:{}\nPath should contain "/arcadia" word'.format(checkout_arcadia_from_url))
        logging.info('Svn url to checkout: %s', checkout_arcadia_from_url)

        try:
            self.ctx[consts.ARCADIA_URL_KEY] = svn.Arcadia.freeze_url_revision(checkout_arcadia_from_url)
        except errors.SandboxSvnError as error:
            logging.exception("freeze_url_revision failed:")
            raise errors.SandboxTaskUnknownError(
                'Arcadia URL {0} does not exist. Error: {1}'.format(checkout_arcadia_from_url, error)
            )
        if not self.is_build_bundle_task():
            return self.do_execute()
        else:
            return self.check_build_bundle_tasks()

    def do_execute(self):
        """
            Основной код сборки, должен быть определён в наследнике

            :return:
        """
        raise NotImplementedError

    def _target_enabled(self, target):
        return self.ctx.get('build_{}'.format(target), False)

    def _target_resource_id(self, target):
        return self.ctx.get('{}_resource_id'.format(target), 0)

    def arcadia_info(self):
        """
            Получение информации для релиза

            :return: кортеж из значений - ревизия, тег, бранч
            :rtype: list
        """
        arcadia_url_key = self.ctx.get(consts.ARCADIA_URL_KEY)
        if not arcadia_url_key:
            return None, None, None
        branch, tag = utils.branch_tag_from_svn_url(arcadia_url_key)
        return self.ctx.get('arcadia_revision'), tag, branch

    def is_good_src_dir(self, d):
        """
            Проверяет, является ли директория кеша подходящей для репозитория из параметра checkout_arcadia_from_url

            :param d: путь до директории
            :return: True, если подходит; False в противном случае
        """
        p = process.run_process(['svn', 'info', d], check=False, log_prefix='svnurl')
        if p.returncode:
            return False

        cached_url = filter(lambda x: x.startswith('URL:'), open(p.stdout_path).readlines())[0][5:].rstrip('\n')
        checkout_url = self.ctx.get(consts.ARCADIA_URL_KEY)

        cached_path = svn.Arcadia.parse_url(cached_url).path
        checkout_path = svn.Arcadia.parse_url(checkout_url).path

        # check if remaining parts are same
        if ''.join(cached_path.split('/')[1:]) != ''.join(checkout_path.split('/')[1:]):
            return False

        return True

    def get_arcadia_src_dir(self, ignore_externals=False, copy_trunk=False):
        """
            Получить репозиторий с аркадией.

            Сначала ищем директорию в кеше, если не нашли то делаем полный чекаут.
            Директории с кешом удаляются специльным таском раз в неделю.

            :param ignore_externals: использовать ли параметр svn-а --ignore_externals
        """
        arcadia_src_dir = svn.Arcadia.get_arcadia_src_dir(
            self.ctx[consts.ARCADIA_URL_KEY], copy_trunk=copy_trunk)
        if not arcadia_src_dir:
            raise errors.SandboxTaskFailureError(
                'Cannot get repo for url {0}'.format(self.ctx[consts.ARCADIA_URL_KEY])
            )
        return arcadia_src_dir

    LOCAL_RELEASE_DIR = 'release'

    def configure_ccache(self):
        """
        Настроить ccache - прописать нужные переменные окружения, получить ресурс ccache
        """
        return environments.CCache().prepare()

    def fill_system_info(self):
        """
            Добавить информацио про ОС в контекст
        """
        self.ctx['target_os'] = platform.system() + ' ' + platform.release()
        self.ctx['target_arch'] = platform.machine() or 'unknown'

    def clean_release_dir(self, release_dir):
        """
            Очистить директорию сборки

            :param release_dir: путь до директории сборки
        """
        paths.remove_path(self.abs_path('arcadia'))
        if not utils.get_or_default(self.ctx, UseObjCache):
            paths.remove_path(self.abs_path(release_dir))

    def save_make_paths_func(self, make_log_file_name, arcadia_src_dir, target):
        """
            Сохранить make-пути после сборки в ресурсах

            :param make_log_file_name:
            :param arcadia_src_dir: путь до репозитория Аркадия
            :param target: какие таргеты собирались
        """
        logging.info("saving make paths for target %s", target)

        make_log = open(make_log_file_name).read()

        arcadia_src_dir += "/"

        paths = set([])

        for path in re.findall(arcadia_src_dir + "(\\S+)", make_log):
            while (not os.path.isdir(arcadia_src_dir + path)) and path:
                path = "/".join(path.split("/")[:-1])
            if path:
                paths.add(os.path.normpath(path))

        resource_file_name = self.abs_path("dir_list_" + target)
        fu.write_lines(resource_file_name, sorted(paths))

        r = self._create_resource("make dir list for " + target, resource_file_name, 'DIR_LIST')
        if self.ctx.get('save_make_paths_attr'):
            r.attrs.update({self.ctx['save_make_paths_attr']: target})
        r.mark_ready()

    def should_checkout(self):
        checkout_auto = self.ctx.get(consts.CHECKOUT_MODE) == consts.CHECKOUT_MODE_AUTO
        if checkout_auto:
            parsed_url = svn.Arcadia.parse_url(self.ctx.get(consts.ARCADIA_URL_KEY))
            checkout = not parsed_url.trunk and not self.ctx.get(consts.ARCADIA_PATCH_KEY)
        else:
            checkout = self.ctx.get(consts.CHECKOUT)
        logging.info("Use selective checkout: %s", checkout)
        return checkout

    def deref(self, s):
        def deref_vault(match):
            secret = self.get_vault_data(match.group('owner'), match.group('name'))

            if match.group('dst') == 'file':
                tmp = tempfile.NamedTemporaryFile()
                TMP_STORE.append(tmp)
                deref_path = tmp.name
                fu.write_file(deref_path, secret)
                return deref_path

            return secret

        s = re.sub(VAULT_PATTERN, deref_vault, s)

        return s

    def get_env_vars(self):
        env_vars = utils.get_or_default(self.ctx, build_params.EnvironmentVarsParam)

        if env_vars and not re.match(ENV_VAR_PATTERN, env_vars):
            raise errors.SandboxTaskFailureError("Incorrect 'Environment variables' parameter '{}'".format(env_vars))

        logging.info("{} checked".format(build_params.EnvironmentVarsParam.name))

        env_vars = {k: self.deref(v) for k, v in (x.split('=', 1) for x in shlex.split(env_vars))}

        if 'GSID' not in env_vars.keys() and 'GSID' in os.environ.keys():
            env_vars['GSID'] = os.getenv('GSID')

        return env_vars
