# -*- coding: utf-8 -*-

import logging
import os

from sandbox.projects import resource_types

from sandbox.projects.common.build.CommonBuildTask import CommonBuildTask
import sandbox.projects.common.constants as consts
from sandbox.projects.common.utils import get_short_branch_name

import sandbox.common.types.resource as ctr

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.svn import Arcadia


class BaseSimpleBuildTaskWithPkg(CommonBuildTask):
    """
        This build module is DEPRECATED by Unity (devtools@),
        although it is still widely used.

        In new build tasks please use YA_PACKAGE.

        Extension class for CommonBuildTask.
        Adds functionality to run and control child tasks to create packages.
    """

    class UseCcache(parameters.SandboxBoolParameter):
        name = 'use_ccache'
        description = 'Use ccache'
        default_value = False

    class InfoFieldPkg(parameters.SandboxInfoParameter):
        name = 'info_field_pkg'
        description = 'Packages'

    class CreatePackages(parameters.SandboxBoolParameter):
        name = 'create_packages'
        description = 'Create Packages'
        default_value = False

    input_parameters = CommonBuildTask.input_parameters + [UseCcache, InfoFieldPkg, CreatePackages]

    RESOURCE_TO_PKG_MAP = {
        # ## Example:
        #
        # # Key is resource architecture.
        #
        # 'freebsd':  {
        #     'RESOURCE':        'svn/path/to/package',
        # },
        # 'linux': {
        #
        #     # Key is resource type.
        #     #
        #     # Value is either:
        #     #    * subpath to package in svn, in this case package architecture
        #     #      is the same as resource architecture (the most usual);
        #     #
        #     #    * dictionary where keys are package architectures and values
        #     #      are svn subpaths for appropriate architectures (it's
        #     #      sometimes necessary to pack e.g. Linux binary in both Linux
        #     #      and FreeBSD packages.
        #
        #     'RESOURCE':        'svn/path/to/package1',
        #     'OTHER_RESOURCE':  'svn/path/to/package2',
        #     'RES_FOR_2_ARCH':  {
        #                            'freebsd': 'svn/path/to/freebsd_package',
        #                            'linux': 'svn/path/to/linux_package',
        #                        },
        # },
    }

    def __init__(self, task_id=0):
        CommonBuildTask.__init__(self, task_id)

    def initCtx(self):
        CommonBuildTask.initCtx(self)
        self.ctx[consts.BUILD_SYSTEM_KEY] = consts.YMAKE_BUILD_SYSTEM

    def are_pkgs_enabled(self):
        """Checks whether packages are enabled or not.
        :return: True if packages are enabled, False otherwise.
        """
        return self.ctx.get('create_packages')

    def do_pkg_tasks_exist(self):
        """Checks whether package tasks exist or not.
        :return: True if package tasks exist, False otherwise.
        """
        return self.ctx.get('create_packages') and self.ctx.get('pkg_tasks_ids')

    def mark_resources_ready(self):
        """Marks non-logs resources as ready to be able to create packages and
        to prevent breaking task after it is waked up on other host.
        :return: Nothing.
        """
        logging.info('Mark resources as ready.')
        for resource in self.list_resources():
            resource = channel.sandbox.get_resource(resource.id)
            if resource.type != resource_types.TASK_LOGS and resource.status == ctr.State.NOT_READY:
                self.mark_resource_ready(resource)

    def get_pkg_tasks_props(self):
        """Walks through all the task's resources and creates dictionary to
        save all necessary properties for creating package-building sub-tasks.
        :return: Dictionary with task properties.
        """
        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)
        pkg_tasks_props = {}
        for resource in self.list_resources():
            logging.info('Check resource %s.' % resource.id)
            resource_to_pkg_map = self.RESOURCE_TO_PKG_MAP.get(resource.arch)
            if not resource_to_pkg_map:
                logging.info('Unsupported architecture %s.' % resource.arch)
                continue

            pkg_raw_info = resource_to_pkg_map.get(str(resource.type))
            if not pkg_raw_info:
                logging.info('%s is not used in packaging.' % resource.type)
                continue

            if isinstance(pkg_raw_info, str):
                pkg_subpaths = {resource.arch: pkg_raw_info}
            elif isinstance(pkg_raw_info, dict):
                pkg_subpaths = pkg_raw_info
            else:
                logging.info('Unexpected map structure: %s.' % pkg_raw_info)
                continue

            for pkg_arch, pkg_subpath in pkg_subpaths.iteritems():
                pkg_svn_url = Arcadia.replace(url, path=os.path.join(parsed_url.path, pkg_subpath))

                if not pkg_tasks_props.get(pkg_svn_url):
                    # This is the first resource for this package.
                    pkg_tasks_props[pkg_svn_url] = {
                        'pkg_arch': pkg_arch,
                        'pkg_name': os.path.basename(pkg_subpath),
                        'resource_ids': [],
                    }
                pkg_tasks_props[pkg_svn_url]['resource_ids'].append(resource.id)

        return pkg_tasks_props

    def create_pkg_tasks(self, pkg_ctx={}):
        """Creates sub-tasks to build packages for Robot binaries.
        :return: Nothing.
        """
        logging.info('Create package tasks.')

        arcadia_url = self.ctx[consts.ARCADIA_URL_KEY]
        branch_name = get_short_branch_name(arcadia_url)
        if not branch_name:
            raise TaskFail('Cannot get branch name from %s.' % arcadia_url)
        revision = Arcadia.parse_url(arcadia_url).revision
        pkg_version = '.'.join([revision, branch_name, str(self.id)])

        pkg_tasks_props = self.get_pkg_tasks_props()
        pkg_tasks_ids = []
        for pkg_svn_url, task_props in pkg_tasks_props.iteritems():
            pkg_arch = task_props['pkg_arch']
            task_type = 'CREATE_%s_PACKAGE' % pkg_arch.upper()
            sub_arch_hint = pkg_ctx.get('sub_arch_hint', {}).get(pkg_arch)
            task_arch = sub_arch_hint or pkg_arch
            task_descr = 'Worker for %s: create package %s.' % \
                         (self.id, task_props['pkg_name'])
            task_ctx = {
                'pkg_svn_url': pkg_svn_url,
                'pkg_version': pkg_version,
                'resource_ids': task_props['resource_ids'],
                # We don't need notifications for subtasks.
                'notify_via': '',
                'notify_if_finished': '',
                'notify_if_failed': '',
            }
            task_ctx.update(pkg_ctx)
            logging.info('Task context: %s' % task_ctx)
            pkg_task = self.create_subtask(
                task_type=task_type,
                arch=task_arch,
                input_parameters=task_ctx,
                description=task_descr
            )
            pkg_tasks_ids.append(pkg_task.id)
        self.ctx['pkg_tasks_ids'] = pkg_tasks_ids
        if self.ctx['pkg_tasks_ids']:
            self.wait_all_tasks_completed(pkg_tasks_ids)

    def check_pkg_tasks(self):
        """Checks package-building sub-tasks and their resources.
        :return: Nothing.
        """
        logging.info('Check package tasks.')
        pkg_full_names = []
        for pkg_task_id in self.ctx['pkg_tasks_ids']:
            logging.info('Check subtask %s.' % pkg_task_id)
            pkg_task = channel.sandbox.get_task(pkg_task_id)
            if pkg_task.is_failure():
                raise TaskFail('Subtask %s failed.' % pkg_task.id)
            pkg_full_name = pkg_task.ctx.get('pkg_full_name')
            if pkg_full_name:
                pkg_full_names.append(pkg_full_name)
        self.ctx['pkg_full_names'] = pkg_full_names
        logging.info('Package tasks are checked successfully.')
        self.set_info('Packages are created: %s.' % ' '.join(pkg_full_names))
