# -*- coding: utf-8 -*-

import os
import logging
import tarfile
from contextlib import closing

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import make_folder

from sandbox.projects import resource_types
from sandbox.projects.common.utils import svn_revision
from sandbox.projects.common.vcs import arc
import sandbox.projects.common.build.ArcadiaTask as bbt
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.projects.common.arcadia import sdk


def get_arcadia_binary_build_params(group_name='Binary build params'):
    class BuildParams(object):
        class BuildPath(parameters.SandboxStringParameter):
            name = 'build_path'
            description = 'Build paths (e.g. path/one;path/t/wo)'
            default_value = 'quality/mapreduce'
            group = group_name

        class Binaries(parameters.SandboxStringParameter):
            name = 'binaries'
            description = 'Files to copy (e.g. path/one/one;path/two/two)'
            default_value = 'quality/mapreduce/mapreduce'
            group = group_name

        class NonStrip(parameters.SandboxBoolParameter):
            name = 'non_strip'
            description = 'Build non-stripped binaries'
            group = group_name

        params = [
            BuildPath,
            Binaries,
            NonStrip,
        ]

    return BuildParams


class BuildArcadiaBinary(bbt.ArcadiaTask):
    """
        This build module is DEPRECATED by Unity (devtools@),
        although it is still widely used.

        In new build tasks please use common/build/YaMake.py module (YA_MAKE task in sandbox interface).
    """

    type = 'BUILD_ARCADIA_BINARY'

    execution_space = 70000  # 70 Gb for build should be enough

    LOCAL_BIN_DIR = 'binaries'

    input_parameters = build_params.get_common_build_params(consts.YMAKE_BUILD_SYSTEM) + get_arcadia_binary_build_params().params

    def on_enqueue(self):
        bbt.ArcadiaTask.on_enqueue(self)

        resource_path = os.path.join(self.LOCAL_BIN_DIR, 'binaries.tar.gz')
        resource = self._create_resource(
            self.descr,
            resource_path,
            resource_types.ARCADIA_BINARY_ARCHIVE)
        self.ctx['out_resource_id'] = resource.id

    def _prepare_to_build(self):
        os.unsetenv('LD_LIBRARY_PATH')
        self.ctx[bbt.UseObjCache.name] = True

        logging.info(
            'Starting build %s from %s',
            self.ctx['build_path'],
            self.ctx[consts.ARCADIA_URL_KEY],
        )

        self.ctx['arcadia_revision'] = svn_revision(self.ctx[consts.ARCADIA_URL_KEY] or "")
        self.fill_system_info()

        logging.info('Arcadia checkout completed')
        self.set_info('checkout')

    def _generate_results(self):
        logging.info('Create the tar-file with binaries')
        binaries_dir = self.abs_path(self.LOCAL_BIN_DIR)
        binaries_tar = os.path.join(binaries_dir, 'binaries.tar.gz')
        make_folder(binaries_dir)

        binaries = self.ctx['binaries'].replace(' ', '')
        binaries = filter(len, binaries.split(';'))

        with closing(tarfile.open(binaries_tar, 'w:gz', dereference=True)) as tar_file:
            for binary in binaries:
                binary_path = os.path.realpath(os.path.join(self.release_dir, binary))
                if not os.path.exists(binary_path):
                    raise SandboxTaskFailureError(
                        "Binary '{0}' does not exist after build at {1}. "
                        "Check its name for misspells".format(binary, binary_path))

                tar_file.add(binary_path, arcname=os.path.basename(binary))

        self.clean_release_dir(self.release_dir)
        logging.info('Cleanup completed')

    def build(self):
        build_system = self.ctx.get(build_params.BuildSystem.name, consts.YMAKE_BUILD_SYSTEM)
        target_dirs = filter(None, [e.strip() for e in self.ctx['build_path'].split(';')])
        if not target_dirs:
            logging.info('No target dirs defined. Build arcadia from root instead.')
            target_dirs = ['']
        def_flags = self.ctx.get(consts.DEFINITION_FLAGS_KEY, '')
        if self.ctx['non_strip']:
            logging.info('Building non-stripped binaries')
            def_flags += ' -DNO_STRIP=yes'
        else:
            def_flags += ' -DNO_STRIP=no'

        self.release_dir = self.abs_path(self.LOCAL_RELEASE_DIR)

        sdk.do_build(
            build_system, self.arcadia_src_dir, target_dirs, consts.RELEASE_BUILD_TYPE, clear_build=True,
            results_dir=self.release_dir, def_flags=def_flags
        )

    def on_execute(self):
        with arc.Arc().mount_path(None, None, fetch_all=False) as arcadia_src_dir:
            self.arcadia_src_dir = arcadia_src_dir
            Arcadia.apply_patch(self.arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())
            self._prepare_to_build()
            self.build()
            self._generate_results()
