# coding: utf-8

# noinspection PyProtectedMember
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import sandbox.projects.common.build.BuildForAllTask as bfa
import sandbox.projects.common.constants as consts
import sandbox.projects.common.build.parameters as build_params
from sandbox.projects.common.build.BuildArcadiaProjectForAllSubtask import BuildArcadiaProjectForAllSubtask, ArcadiaProjectResourceType

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters


class PathToProject(parameters.String):
    name = 'project'
    description = 'Path to build artefact (<path/to/build_dir>/<target_name>)'
    default_value = None


class PlatformsList(bfa.PlatformsListParameter):
    choices = [
        ('Linux', 'linux'),
        ('Win32', 'win32'),
        ('Win32-i686', 'win32_i686'),
        ('Darwin', 'darwin'),
        ('Darwin-arm64', 'darwin-arm64'),
        ('Linux-aarch64', 'linux-aarch64'),
        ('Linux-ppc64le', 'linux-ppc64le'),
        ('Cygwin', 'cygwin'),
    ]


class Parameters(sdk2.Parameters):
    do_not_remove_resources = bfa.DoNotRemoveResourcesParameter()
    resource_type = ArcadiaProjectResourceType()
    platforms_list = PlatformsList()
    checkout_arcadia_from_url = build_params.ArcadiaUrl()
    build_system = build_params.BuildSystem()
    definition_flags = build_params.DefinitionFlags()
    strip_binaries = build_params.StripBinaries()
    arcadia_patch = build_params.ArcadiaPatch()
    backup_to_mds = bfa.BackupToMDS()
    mds_namespace = bfa.MDSNamespaceParameter()
    mds_token_vault_name = bfa.MDSTokenVaultNameParameter()
    mds_token_vault_owner = bfa.MDSTokenVaultOwnerParameter()
    mds_token_from_yav = bfa.MDSTokenYavSecret()
    mds_download_url = bfa.MDSDownloadURLParameter()


def get_build_arcadia_project_for_all_params():
    return list(Parameters)


class BuildArcadiaProjectForAll(bfa.BuildForAllTask):
    type = 'BUILD_ARCADIA_PROJECT_FOR_ALL'
    release_to = 'sandbox-releases@yandex-team.ru'
    subtask_type = BuildArcadiaProjectForAllSubtask

    target_to_host_map = {
        'linux': consts.SANDBOX_LINUX_UBUNTU_12_PRECISE,
        'win32': consts.SANDBOX_LINUX_UBUNTU_12_PRECISE,
        'win32_i686': consts.SANDBOX_LINUX_UBUNTU_12_PRECISE,
        'darwin': consts.SANDBOX_LINUX_UBUNTU_12_PRECISE,
        'darwin-arm64': consts.SANDBOX_LINUX_UBUNTU_12_PRECISE,
        'linux-aarch64': consts.SANDBOX_LINUX_UBUNTU_12_PRECISE,
        'linux-ppc64le': consts.SANDBOX_LINUX_UBUNTU_12_PRECISE,
        'cygwin': consts.SANDBOX_LINUX_UBUNTU_12_PRECISE,
    }

    toolchain_map = {
        'linux': 'DEFAULT-LINUX-X86_64',
        'win32': 'DEFAULT-WIN-X86_64',
        'win32_i686': 'DEFAULT-WIN-i686',
        'darwin': 'DEFAULT-DARWIN-X86_64',
        'darwin-arm64': 'DEFAULT-DARWIN-ARM64',
        'linux-aarch64': 'DEFAULT-LINUX-AARCH64',
        'linux-ppc64le': 'DEFAULT-LINUX-PPC64LE',
        'cygwin': 'GCC49-CYGWIN-X86_64',
    }

    input_parameters = get_build_arcadia_project_for_all_params() + [PathToProject] + \
        build_params.get_aapi_parameters() + build_params.get_lto_parameters() + build_params.get_musl_parameters() + \
        build_params.get_use_prebuilt_tools_parameters()

    @property
    def resource_type(self):
        return sdk2.resource.Resource[self.ctx[ArcadiaProjectResourceType.name]]

    def get_resource_description(self):
        if self.resource_description is None:
            self.resource_description = '{} for {}'.format(self.ctx.get(PathToProject.name, ''), '{target}')
        return self.resource_description

    def get_subtask_description(self):
        if self.subtask_description is None:
            self.subtask_description = 'Building {} for {}'.format(self.ctx.get(PathToProject.name, ''), '{target}')
        return self.subtask_description

    def update_params_for_build(self, params, platform):
        if platform not in self.toolchain_map:
            raise SandboxTaskFailureError('Toolchain for platform {platform} is unavailable.'.format(platform=platform))
        if platform != 'linux':
            params[consts.MUSL] = False
        params[consts.TARGET_PLATFORM_FLAGS] = '--target-platform=' + self.toolchain_map[platform]
        params['target_platform_alias'] = platform
        params['build_target'] = self.ctx.get(PathToProject.name, '')
        params[consts.STRIP_BINARIES] = self.ctx.get(consts.STRIP_BINARIES)
        return params
