# coding: utf-8

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters

import sandbox.sdk2.vcs.svn as svn
import sandbox.common.errors as errors

from sandbox.projects.common import utils

import sandbox.common.types.task as ctt
import sandbox.projects.common.build.BuildArcadiaProjectForAll as bapfa


class PathToProjects(parameters.String):
    name = 'projects'
    description = 'Path to build artifacts ;-separated (<path/to/build_dir>/<target_name>)'
    default_value = None
    required = True


class BuildArcadiaProjectsForAll(sdk2.Task):
    class Parameters(bapfa.Parameters):
        projects = PathToProjects()

    class Context(sdk2.Context):
        child_tasks_ids = None
        arcadia_url = None

    def on_execute(self):
        if not self.Context.child_tasks_ids:
            try:
                self.Context.arcadia_url = svn.Arcadia.freeze_url_revision(self.Parameters.checkout_arcadia_from_url)
                self.set_info('Svn url for arcadia: {}'.format(self.Context.arcadia_url))
            except svn.SvnError as error:
                raise errors.TaskFailure(
                    'Arcadia URL {0} does not exist. Error: {1}'.format(self.Parameters.checkout_arcadia_from_url, error)
                )
            self.Context.child_tasks_ids = self._create_subtasks()
            raise sdk2.WaitTask(self.Context.child_tasks_ids.values(), ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True)
        else:
            self._on_process_finished_tasks()
            utils.check_if_tasks_are_ok(self.Context.child_tasks_ids.values())

    def _create_subtasks(self):
        projects = str(self.Parameters.projects).split(';')
        child_tasks_ids = {}

        subtask_class = sdk2.Task[bapfa.BuildArcadiaProjectForAll.type]
        for project in reversed(projects):
            ctx = {
                bapfa.PathToProject.name: project
            }
            for param in bapfa.get_build_arcadia_project_for_all_params():
                ctx[param.name] = getattr(self.Parameters, param.name)
            ctx['checkout_arcadia_from_url'] = self.Context.arcadia_url

            subtask = subtask_class(self, description='{}'.format(project), inherit_notifications=True, **ctx).enqueue()
            child_tasks_ids[project] = subtask.id

        return child_tasks_ids

    def _on_process_finished_tasks(self):
        child_tasks_ids = self.Context.child_tasks_ids
        self.set_info("<br>".join(self._gen_helpful_links(child_tasks_ids)), do_escape=False)

    @staticmethod
    def _gen_helpful_links(tasks_ids):
        helpful_links = []
        for project, task_id in tasks_ids.items():
            helpful_links.append(
                "<b>{project}</b>: "
                "<a href='https://sandbox.yandex-team.ru/task/{task_id}/view' target='_blank'>https://sandbox.yandex-team.ru/task/{task_id}/view</a>".format(
                    project=project, task_id=task_id)
            )
        return helpful_links
