import os
import copy
import shutil
import logging
import json

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.common.hash import md5sum
from sandbox.projects.common.build.CreateProjectSourcePackage import CreateProjectSourcePackage
from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
import sandbox.projects.common.constants as consts
from sandbox.projects.common.build import build_for_all_task_2
import sandbox.projects.common.build.parameters as build_params

import sandbox.sdk2.parameters as sdk2params

logger = logging.getLogger(__name__)


class VersionParameter(sdk2params.String):
    name = 'build_version'
    description = 'Build version'


class DoNotRemoveResourcesParameter(sdk2params.Bool):
    description = 'Set "do not remove" for resources'
    name = 'do_not_remove_resources'
    default_value = False


class LinuxPlatformParameter(sdk2params.String):
    name = 'linux_platforms_list'
    description = 'Choose linux platform'
    choices = [
        (consts.SANDBOX_LINUX_UBUNTU_10_LUCID, consts.SANDBOX_LINUX_UBUNTU_10_LUCID),
        (consts.SANDBOX_LINUX_UBUNTU_12_PRECISE, consts.SANDBOX_LINUX_UBUNTU_12_PRECISE),
        (consts.SANDBOX_LINUX_UBUNTU_14_TRUSTY, consts.SANDBOX_LINUX_UBUNTU_14_TRUSTY),
        (consts.SANDBOX_LINUX_UBUNTU_16_XENIAL, consts.SANDBOX_LINUX_UBUNTU_16_XENIAL),
        (consts.SANDBOX_LINUX_ANY, consts.SANDBOX_LINUX_ANY),
    ]
    group = 'Platforms'
    default_value = consts.SANDBOX_LINUX_UBUNTU_10_LUCID


class PlatformsListParameter(sdk2params.CheckGroup):
    name = 'platforms_list'
    description = 'list of target platforms'
    choices = [('Linux', 'linux'), ('Darwin', 'darwin')]
    group = 'Platforms'
    default_value = None


class SourcesPackagePlatformsListParameter(sdk2params.String):
    name = 'sources_package_platforms_list'
    description = 'list of target platforms for source package'
    default_value = None


class BackupToMDS(sdk2params.Bool):
    name = "backup_to_mds"
    description = "Backup to MDS"
    default_value = True


class MDSNamespaceParameter(sdk2params.String):
    name = 'mds_namespace'
    description = 'MDS namespace'


class MDSTokenVaultNameParameter(sdk2params.String):
    name = 'mds_token_vault_name'
    description = 'MDS token vault name [Deprecated]'


class MDSTokenVaultOwnerParameter(sdk2params.String):
    name = 'mds_token_vault_owner'
    description = 'MDS token vault owner [Deprecated]'


class MDSTokenYavSecret(sdk2params.YavSecretWithKey):
    name = 'mds_token_from_yav'
    description = 'MDS token from Yav'


class MDSDownloadURLParameter(sdk2params.String):
    name = 'mds_download_url'
    description = 'MDS download url base'


def get_build_for_all_params():
    return [
        DoNotRemoveResourcesParameter,
        PlatformsListParameter,
        build_params.ArcadiaUrl,
        BackupToMDS,
        MDSNamespaceParameter,
        MDSTokenVaultNameParameter,
        MDSTokenVaultOwnerParameter,
    ]


class BuildForAllTask(SandboxTask):
    cores = 1
    subtask_type = None
    subtask_description = None
    resource_type = None
    resource_description = None
    inherit_notifications = False
    target_to_host_map = {
        'linux': consts.SANDBOX_LINUX_OLD_VERSION,
        'darwin': consts.SANDBOX_DARWIN_OLD_VERSION,
    }
    bind_osx_host = False

    def get_version(self):
        return self.ctx.get(VersionParameter.name)

    def get_do_not_remove(self):
        return self.ctx.get(DoNotRemoveResourcesParameter.name, False)

    def update_params_for_build(self, params, platform):
        params['target_platform'] = platform
        params['target_platform_alias'] = platform
        return params

    def get_host_platform_for_build(self, target_platform):
        if target_platform not in self.target_to_host_map:
            raise SandboxTaskFailureError('Target platform {!r} is unavailable.'.format(target_platform))
        if target_platform == 'linux' and LinuxPlatformParameter.name in self.ctx:
            return self.ctx.get(LinuxPlatformParameter.name)
        return self.target_to_host_map[target_platform]

    @staticmethod
    def get_host_platform_for_package(target_platform):
        return 'linux'

    def run_subtasks(self, task_type, description, target_platforms):
        logging.debug("target_platforms={}".format(target_platforms))

        subtasks = []
        common_params = {p.name: self.ctx[p.name] for p in self.input_parameters if p.name in self.ctx}
        for target_platform_alias in target_platforms:
            params = dict(common_params)
            params['ya_path'] = 'devtools/ya'  # XXX: remove
            params['do_not_restart'] = self.ctx.get('do_not_restart', False)
            params[consts.BUILD_SYSTEM_KEY] = self.ctx.get(consts.BUILD_SYSTEM_KEY, consts.YMAKE_BUILD_SYSTEM)
            host_name = None
            if task_type.type == CreateProjectSourcePackage.type:
                task_arch = self.get_host_platform_for_package(target_platform_alias)
                params['target_platform'] = params['target_platform_alias'] = target_platform_alias
            else:
                task_arch = self.get_host_platform_for_build(target_platform_alias)
                # TODO: temporary hack for darwin, DEVTOOLS-3708
                if task_arch == consts.SANDBOX_DARWIN_OLD_VERSION and self.bind_osx_host:
                    host_name = 'ymake-dev-mac'
                    params[consts.USE_SANDBOX_GCC] = False
                self.update_params_for_build(params, target_platform_alias)
            logging.debug("Task arch is %s %s", task_arch, params)
            task = self.create_subtask(
                task_type.type,
                description.format(target=target_platform_alias, version=self.get_version()),
                execution_space=self.get_subtask_execution_space(),
                input_parameters=params,
                host=host_name,
                arch=task_arch,
                important=self.important,
                inherit_notifications=self.inherit_notifications
            )
            subtasks.append(task)

        self.wait_all_tasks_stop_executing(subtasks)

    def save_resources(self, resource_type, description):
        by_platform = {"data": {}}
        for task in self.list_subtasks(load=True):
            if task.status == self.Status.FAILURE:
                raise SandboxTaskFailureError('Build in worker {id} failed'.format(id=task.id))
            if resource_type is None:
                continue

            resource_name = resource_type.name
            sub_resources = [r for r in apihelpers.list_task_resources(task.id) if r.type == resource_name]
            if len(sub_resources) == 0:
                continue

            resource = max(sub_resources, key=lambda res: res.id)

            alias = task.ctx['target_platform_alias']

            src_path = self.sync_resource(resource.id)

            dst_dir = alias
            dst_path = os.path.join(dst_dir, os.path.basename(resource.file_name))

            if not os.path.exists(dst_dir):
                os.makedirs(dst_dir)
            if os.path.isdir(src_path):
                shutil.copytree(src_path, dst_dir)
            else:
                shutil.copy(src_path, dst_dir)

            attrs = copy.deepcopy(resource.attributes)
            if self.get_version():
                attrs['version'] = self.get_version()
            if self.get_do_not_remove():
                attrs['ttl'] = 'inf'
            mds_url = self._upload_to_mds(attrs, dst_path)

            resulting_resource = self._create_resource(
                resource_desc=description.format(target=alias, version=self.get_version()),
                resource_filename=dst_path,
                resource_type=resource_type,
                arch=resource.arch,
                attrs=attrs,
                complete=True,
            )
            urls = [resulting_resource.proxy_url]
            if mds_url:
                urls.append(mds_url)

            by_platform["data"][alias] = {
                "url": resulting_resource.proxy_url,
                "md5": resulting_resource.file_md5,
                "urls": urls,
            }

        return by_platform

    def _upload_to_mds(self, attrs, path):
        """ Mutate attrs! Decide hot to upload data to MDS """
        if self.ctx.get(BackupToMDS.name):
            mds_url = self.save_resource_to_mds(path)

            if not mds_url:
                attrs["sync_upload_to_mds"] = True
            else:
                attrs["sync_upload_to_mds"] = False
                attrs["mds_url"] = mds_url

            return mds_url

    def save_resource_to_mds(self, path):
        yav_secret_raw = self.ctx.get(MDSTokenYavSecret.name)  # type: str
        mds_token = None
        if yav_secret_raw:
            yav_secret = sdk2params.YavSecretWithKey.cast(yav_secret_raw)  # type: sdk2params.YavSecretWithKey
            mds_token = yav_secret.value()

        if not mds_token:
            logger.info("MDS Token from Yav NOT found")
            return

        mds_namespace = self.ctx.get(MDSNamespaceParameter.name)

        from sandbox import common
        from sandbox.common import mds

        mds_key = mds.MDS().upload(
            path=path,
            mds_name=md5sum(path) + "/" + os.path.basename(path),
            token=mds_token,
            namespace=mds_namespace,
        )
        logging.debug("Path %s uploaded to MDS, key is %s", path, mds_key)

        download_url = self.ctx.get(MDSDownloadURLParameter.name)

        if mds_namespace:
            mds_url = "https://{}/get-{}/{}".format("storage.yandex-team.ru", mds_namespace, mds_key)
        elif download_url:
            mds_url = "{}/{}".format(download_url, mds_key)
        else:
            mds_url = "{}/{}".format(common.config.Registry().client.mds.dl.url, mds_key)

        logging.debug("Data from %s backed up to MDS %s", path, mds_url)

        return mds_url

    def save_platform_mapping(self, by_platform):
        by_platform_file = "by_platform.json"
        with open(by_platform_file, "w") as f:
            json.dump(by_platform, f, indent=4)

        attrs = {"ttl": "inf", "sync_upload_to_mds": True} if self.get_do_not_remove() else {}
        self._upload_to_mds(attrs, by_platform_file)

        mapping_resource = self._create_resource(
            resource_desc="",
            resource_filename=by_platform_file,
            resource_type=resource_types.PLATFORM_MAPPING,
            attrs=attrs
        )

        logger.debug("mapping_resource.attrs after creation %s", mapping_resource.attrs)

        mapping_resource.mark_ready()

    def get_resource_description(self):
        return self.resource_description

    def get_subtask_description(self):
        return self.subtask_description

    def get_subtask_execution_space(self):
        return None

    def on_execute(self):
        if 'build_subtasks_created' not in self.ctx:
            self.ctx['build_subtasks_created'] = True
            self.freeze_arcadia_revision()
            target_platforms = build_for_all_task_2.get_target_platforms(
                self.ctx.get(PlatformsListParameter.name)
            )
            if target_platforms:
                self.run_subtasks(self.subtask_type, self.get_subtask_description(), target_platforms)

        if 'sources_package_subtasks_created' not in self.ctx:
            self.ctx['sources_package_subtasks_created'] = True
            target_platforms = build_for_all_task_2.get_target_platforms(
                self.ctx.get(SourcesPackagePlatformsListParameter.name)
            )
            if target_platforms:
                self.run_subtasks(CreateProjectSourcePackage, self.get_subtask_description() + ' (sources)',
                                  target_platforms)

        self.save_resources(resource_types.PROJECT_SOURCES_PACKAGE, self.get_resource_description())
        self.save_platform_mapping(self.save_resources(self.resource_type, self.get_resource_description()))

    def arcadia_info(self):
        return '', None, self.get_version()

    def freeze_arcadia_revision(self):
        if build_params.ArcadiaUrl.name in self.ctx:
            self.ctx[build_params.ArcadiaUrl.name] = Arcadia.freeze_url_revision(
                self.ctx.get(build_params.ArcadiaUrl.name)
            )
