# -*- coding: utf-8 -*-
import logging
import os
import time

import sandbox.sdk2 as sdk2
import sandbox.sdk2.helpers
from sandbox.common import errors as common_errors
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common import solomon
from sandbox.sdk2.helpers import subprocess
import sandbox.sandboxsdk.ssh as sdk_ssh
import sandbox.common.types.client as ctc

logger = logging.getLogger(__name__)


class CheckAutobuildTime(sdk2.Task):
    """
    Check auto build time
    """

    class Parameters(sdk2.Parameters):
        vault_owner = sdk2.parameters.String("Vault owner")
        vault_key_name = sdk2.parameters.String("Vault key name")
        committer = sdk2.parameters.String("Committer")
        solomon_token_owner = sdk2.parameters.String("Vault owner")
        solomon_token_key_name = sdk2.parameters.String("Vault key name")

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        cores = 1
        ram = 1024
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        if self.Parameters.vault_owner and self.Parameters.vault_key_name:
            self.token = sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.vault_key_name)
        else:
            self.token = None

        with sdk_ssh.Key(self, "akastornov", "arcadia-devtools-ssh-key"):
            ya_tool = sdk2.svn.Arcadia.export('arcadia:/arc/trunk/arcadia/ya', 'ya')
            clone_env = os.environ.copy()
            clone_env["SVN_SSH"] = "ssh -l {}".format(self.Parameters.committer)
            src_path = os.path.abspath("arcadia")
            with sandbox.sdk2.helpers.ProcessLog(self, 'clone') as pl:
                p = subprocess.Popen([ya_tool, "-v", "clone", "--no-junk", src_path], stdout=pl.stdout, stderr=pl.stderr, env=clone_env)
                p.communicate()

            proc = subprocess.Popen(
                [sdk._get_ya_tool(src_path), 'tool', 'svn', 'up', '--parents', 'devtools/dummy_arcadia/simple_review_autobuild_check'],
                cwd=src_path,
            )
            proc.wait()

            pr_file_path = os.path.join(src_path, "devtools/dummy_arcadia/simple_review_autobuild_check", "lib/__init__.py")
            with open(pr_file_path) as f:
                text = f.read()

            logging.info("Text read:\n{}".format(text))
            if "a + b" in text:
                logging.info("Replace 'a + b'")
                text = text.replace("a + b", "b + a")
            elif "b + a" in text:
                logging.info("Replace 'b + a'")
                text = text.replace("b + a", "a + b")
            else:
                logging.error("Failed to change text")

            logging.info("Text to write:\n{}".format(text))

            with open(pr_file_path, 'w') as f:
                logging.info("Writing text")
                f.write(text)

            start_time, commit_duration, check_duration = self.timed_commit(src_path)
            logging.info("Commit Duration: {} s".format(int(commit_duration)))
            self.set_info("Commit Duration: {} s".format(int(commit_duration)))
            self.send_time_to_solomon(start_time, commit_duration, check_duration)

    def timed_commit(self, source_dir):
        ya_tool = sdk._get_ya_tool(source_dir)
        ya_pr_cmd = [
            sdk._python(),
            ya_tool,
            'pr',
            'create',
            '-m',
            'Commit for autobuild check',
            '-A',
        ]
        ya_pr_st = [
            sdk._python(),
            ya_tool,
            'pr',
            'status',
            '-v',
        ]
        env = os.environ.copy()
        if self.token:
            env['YA_TOKEN'] = self.token
            env['YA_USER'] = self.Parameters.committer
            env['SVN_SSH'] = 'ssh -l {}'.format(self.Parameters.committer)
        try:
            proc = subprocess.Popen(ya_pr_cmd, cwd=source_dir, stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=env)
            out, err = proc.communicate()
            logs = out + err
            self.set_info("PR created:\n{}".format(logs))
            logging.info("PR created:\n{}".format(logs))

            prefix = "https://a.yandex-team.ru/review/"
            for line in logs.splitlines():
                if prefix in line:
                    review_id = line[len(prefix):]
                    logging.info("ReviewId: {}".format(review_id))
                    ya_pr_st.extend(['-i', review_id])

            t_start = time.time()
            check_duration = -1
            while True:
                t_end = time.time()
                proc = subprocess.Popen(ya_pr_st, cwd=source_dir, stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=env)
                out, err = proc.communicate()
                output = out + err
                logging.info("Current output: %s", output)
                if "Success" in output and "In progress" not in output and check_duration < 0:
                    check_duration = t_end - t_start
                if "merged as" in output:
                    return t_start, t_end - t_start, check_duration

                time.sleep(1)

        except subprocess.CalledProcessError as cpe:
            raise common_errors.TaskFailure("PR failed:\n{}".format(cpe.output))

        return -1

    def send_time_to_solomon(self, start_time, commit_duration, check_duration):
        common_labels = {
            'project': 'autocheck',
            'cluster': 'push_autocheck',
            'service': 'push_autocheck',
        }
        token = sdk2.Vault.data(self.Parameters.solomon_token_owner, self.Parameters.solomon_token_key_name)
        sensors = [
            {
                'labels': {'sensor': 'sample_merge_time', },
                'ts': int(start_time),
                'value': int(commit_duration),
            }
        ]
        solomon.push_to_solomon_v2(token=token, params=common_labels, sensors=sensors)

        sensors = [
            {
                'labels': {'sensor': 'sample_check_time', },
                'ts': int(start_time),
                'value': int(check_duration),
        }
        ]
        solomon.push_to_solomon_v2(token=token, params=common_labels, sensors=sensors)
