# -*- coding: utf-8 -*-
import logging
import os
import platform

from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxSubprocessError, SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.common import context_managers
from sandbox.projects.common import utils
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
import sandbox.projects.common.build.ArcadiaTask as bbt
from sandbox.projects.common.build import dependency_banner


class CommonBuildTask(bbt.ArcadiaTask):
    """
        This build module is DEPRECATED by Unity (devtools@),
        although it is still widely used.

        In new build tasks please use common/build/YaMake.py module.
    """
    input_parameters = bbt.ArcadiaTask.input_parameters + build_params.get_build_system_params() + \
        build_params.get_aapi_parameters_as_default() + build_params.get_lto_parameters() + \
        build_params.get_musl_parameters() + build_params.get_use_prebuilt_tools_parameters() + [
            build_params.ForceVCSInfoUpdate,
        ]

    LOCAL_BIN_DIR = 'binaries'

    def initCtx(self):
        self.ctx['notify_via'] = 'email'
        for _, target, _ in self.get_target_resources():
            self.ctx['build_%s' % target] = True

    def on_enqueue(self):
        bbt.ArcadiaTask.on_enqueue(self)
        if self.is_build_bundle_task():
            return
        self.prepare_resources()

    def prepare_resources(self):
        for resource_type, target, target_path in self.get_target_resources():
            if resource_type is not None and self._target_enabled(target):
                filename = os.path.basename(target_path)
                resource_path = os.path.join(self.LOCAL_BIN_DIR, filename)
                resource_name = u'{} ({})'.format(self.descr, filename)
                resource = self._create_resource(resource_name,
                                                 resource_path,
                                                 resource_type,
                                                 arch=self.arch)
                self.ctx[self.get_prepared_resource_ctx_name(target)] = resource.id
        dependency_banner.create_resources(self)

    def get_prepared_resource_id(self, target):
        """
            Получить идентификатор созданного заранее ресурса по ключевому имени target
            :param target: ключевое имя ресурса
            :return: идентификатор ресурса или None, если по переданному ключевому имени не было найдено ресурсов
        """
        return self.ctx.get(self.get_prepared_resource_ctx_name(target))

    def get_prepared_resource_ctx_name(self, target):
        """
            Получить название параметра в контексте, в котором нужно сохранять созданный заранее ресурс по ключевому
                имени target
            :param target: ключевое имя ресурса
            :return: идентификатор ресурса или None, если по переданному ключевому имени не было найдено ресурсов
        """
        return '{0}_resource_id'.format(target)

    def get_definition_flags(self):
        """
            Дополнительные сборочные параметры.
        """

        def_flags = sdk.parse_flags(self.ctx.get(consts.DEFINITION_FLAGS_KEY, ''))
        def_flags["SANDBOX_TASK_ID"] = self.id
        return def_flags

    def _get_src_dir(self):
        arcadia_url = self.ctx.get(consts.ARCADIA_URL_KEY, "")

        use_aapi = utils.get_or_default(self.ctx, build_params.UseArcadiaApiFuseAsDefault)
        use_arc_instead_of_aapi = utils.get_or_default(self.ctx, build_params.UseArcInsteadOfArcadiaApiAsDefault)
        aapi_fallback = utils.get_or_default(self.ctx, build_params.AllowArcadiaApiFallback)
        use_arc_vcs = sdk.is_arc_path(arcadia_url)
        use_fuse = use_aapi or use_arc_vcs
        minimize_arc_mount_path = utils.get_or_default(self.ctx, build_params.MinimizeArcMountPath)

        if use_aapi and aapi_fallback:
            if not sdk.wait_aapi_url(arcadia_url):
                use_aapi = False
                logging.info('Disable AAPI, because service is temporary unavailable')

        if use_fuse and not sdk.fuse_available():
            raise SandboxTaskFailureError('Fuse is not available {} yet'.format(platform.platform()))

        if use_fuse:
            logging.info('Mount = true (using %s)', 'Arc VCS' if use_arc_vcs else 'AAPI')
            return sdk.mount_arc_path(arcadia_url, use_arc_instead_of_aapi=use_arc_instead_of_aapi, fetch_all=False,
                                      minimize_mount_path=minimize_arc_mount_path,
                                      )

        return context_managers.nullcontext(self.get_arcadia_src_dir())

    def do_execute(self):
        os.chdir(self.abs_path())
        binaries_dir = self.abs_path(self.LOCAL_BIN_DIR)
        paths.make_folder(binaries_dir)

        arcadia_ctx = self._get_src_dir()
        with arcadia_ctx as arcadia_src_dir:
            self.ctx['arcadia_src_dir'] = arcadia_src_dir
            Arcadia.apply_patch(arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())
            self.ctx['arcadia_revision'] = utils.svn_revision(arcadia_src_dir)
            self.fill_system_info()

            release_dir = self.abs_path(self.LOCAL_RELEASE_DIR)
            paths.make_folder(release_dir)

            try:
                target_dirs = []
                filtered_target_resources = []
                for resource_type, target_name, target_path in self.get_target_resources():
                    if self._target_enabled(target_name):
                        filtered_target_resources.append((resource_type, target_name, target_path))
                        target_dirs.append(os.path.dirname(target_path))

                if target_dirs:
                    dependency_banner.check_dependencies(self, arcadia_src_dir, target_dirs)

                    build_type = self.ctx.get(consts.BUILD_TYPE_KEY, consts.RELEASE_BUILD_TYPE)
                    target_platform = self.ctx.get(consts.TARGET_PLATFORM_KEY)
                    target_platform_flags = self.ctx.get(consts.TARGET_PLATFORM_FLAGS)
                    host_platform = self.ctx.get(consts.HOST_PLATFORM_KEY)
                    def_flags = self.get_definition_flags()
                    build_system = self.ctx.get(build_params.BuildSystem.name, consts.YMAKE_BUILD_SYSTEM)
                    clear_build = self.ctx.get(build_params.ClearBuild.name, True)
                    use_dev_version = self.ctx.get(build_params.UseDevVersion.name, False)
                    strip_binaries = self.ctx.get(consts.STRIP_BINARIES, False)
                    musl = self.ctx.get(consts.MUSL, False)
                    sdk.do_build(
                        build_system, arcadia_src_dir, target_dirs, build_type,
                        clear_build=clear_build,
                        results_dir=release_dir,
                        def_flags=def_flags,
                        host_platform=host_platform,
                        target_platform=target_platform,
                        target_platform_flags=target_platform_flags,
                        use_dev_version=use_dev_version,
                        strip_binaries=strip_binaries,
                        sanitize=utils.get_or_default(self.ctx, build_params.Sanitize),
                        musl=musl,
                        lto=utils.get_or_default(self.ctx, build_params.LTO),
                        thinlto=utils.get_or_default(self.ctx, build_params.ThinLTO),
                        force_vcs_info_update=utils.get_or_default(self.ctx, build_params.ForceVCSInfoUpdate),
                        use_prebuilt_tools=self.ctx.get(build_params.UsePrebuiltTools.name),
                    )

                logging.debug("Release dir content: %s", os.listdir(release_dir))

                for resource_type, target, target_path in filtered_target_resources:
                    if resource_type is not None:
                        resource_path = channel.sandbox.get_resource(self.get_prepared_resource_id(target)).path
                        if os.path.exists(resource_path):
                            os.unlink(resource_path)
                        paths.copy_path(
                            os.path.join(release_dir, target_path),
                            resource_path,
                            copy_function=os.link,
                        )

            except SandboxSubprocessError:
                self.set_info('Build is failed. Try to check logs folder for details.')
                raise

            self.set_info('Build completed.')
