# -*- coding: utf-8 -*-

import os
import json
import logging
import sys
import tarfile

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.environments import SvnEnvironment
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskUnknownError, SandboxTaskFailureError

from sandbox.projects import resource_types
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.constants as consts
from sandbox.projects.common.import_tools import import_path


class TaskOptions(object):
    def __init__(self, ctx):
        self.ya_path = ctx.get('ya_path')
        self.target_platform = ctx.get('target_platform')
        self.project = ctx.get('project')


class SvnAdapter(object):
    def __init__(self, task, url):
        self.url = url
        self.task = task
        parsed_url = Arcadia.parse_url(url)
        path = parsed_url.path.rstrip('/')
        if path.endswith('/arcadia'):
            path = path[:-8]
        self.path = path
        self.revision = int(parsed_url.revision)
        self.source_root = None

    def get_source_url(self):
        return Arcadia.replace(self.url, path=os.path.join(self.path, 'arcadia'))

    def get_source_root(self):
        if self.source_root is None:
            url = self.get_source_url()
            self.source_root = Arcadia.get_arcadia_src_dir(url)
            if not self.source_root:
                raise SandboxTaskUnknownError('Cannot get repo for url %s' % url)
        return self.source_root


class CreateProjectSourcePackage(ArcadiaTask):
    type = 'CREATE_PROJECT_SOURCE_PACKAGE'
    execution_space = 250000

    environment = [SvnEnvironment()]

    class YaPath(parameters.SandboxStringParameter):
        name = 'ya_path'
        description = 'Path to ya'
        default_value = 'devtools/ya'

    class TargetPlatform(parameters.SandboxStringParameter):
        name = 'target_platform'
        description = 'Target platform'
        default_value = 'darwin'

    class Project(parameters.SandboxStringParameter):
        name = 'project'
        description = 'Project to pack'
        default_value = 'devtools/ymake'

    input_parameters = ArcadiaTask.input_parameters + [
        YaPath,
        TargetPlatform,
        Project,
    ]

    cores = 17
    client_tags = ctc.Tag.LINUX_PRECISE

    def import_ya_module(self):
        logging.info(os.path.join(self.ya_path, 'core', 'checkout.py'))
        return import_path(os.path.join(self.ya_path, 'core', 'checkout.py'), [self.ya_path])

    def __init__(self, task_id=0):
        ArcadiaTask.__init__(self, task_id)
        self.ya_path = None
        self.source_root = None

    def run_ya(self, ya_args, project, stdout_path=None, stderr_path=None):
        command = [sys.executable, os.path.join(self.ya_path, 'ya'), '-v'] + ya_args
        logging.info('Run ya: {}, stdout to {}, stderr to {}'.format(command, stdout_path, stderr_path))
        if os.path.exists(self.ya_path):
            process = run_process(
                command,
                check=False,
                stderr=open(stderr_path, 'w'),
                stdout=open(stdout_path, 'w'),
                wait=True,
                work_dir=os.path.join(self.source_root, project),
                outputs_to_one_file=False,
                log_prefix='build'
            )

            ret_code = process.wait()
            process.communicate()
            if ret_code != 0:
                raise SandboxTaskFailureError('Process {0} terminated with nonzero code {1}'.format(command, ret_code))
        else:
            raise SandboxTaskFailureError('Process {0} was not created: ya does not exist'.format(command))

    def on_execute(self):
        ArcadiaTask.on_execute(self)

    def tar_sources(self, output_filename, source_map, build_plan):
        with tarfile.open(output_filename, "w:gz") as tar:
            tar.add(build_plan, arcname="build_plan.json")
            for key in source_map:
                rel_path = os.path.relpath(key, self.source_root)
                if rel_path.startswith('junk/'):  # XXX: dirt hack to remove junk/sandbox from a tar
                    continue

                if source_map[key]:
                    tar.add(key, arcname=os.path.join('sources', rel_path))
                else:
                    only_files = [f for f in os.listdir(key) if os.path.isfile(os.path.join(key, f))]
                    for f in only_files:
                        tar.add(os.path.join(key, f), arcname=os.path.join('sources', rel_path, f))

        self._create_resource(
            "Project Source Package",
            output_filename,
            resource_types.PROJECT_SOURCES_PACKAGE,
            attrs={'platform': self.ctx['target_platform']}
        )

    def do_execute(self):
        self.fill_system_info()

        options = TaskOptions(self.ctx)
        svn = SvnAdapter(self, self.ctx[consts.ARCADIA_URL_KEY])

        room_path = self.abs_path()
        self.source_root = svn.get_source_root()
        self.ya_path = os.path.join(self.source_root, options.ya_path)

        logging.info('Source root is %s' % self.source_root)
        logging.info('Room path is %s' % room_path)
        logging.info('Source url is %s' % svn.get_source_url())
        logging.info('Revision is %s' % str(svn.revision))
        logging.info('Ya path is %s' % self.ya_path)

        json_graph_file = self.log_path('build_plan.json')
        self.run_ya(
            ['dump', 'build-plan', '--no-ymake-resource', '-O', options.target_platform],
            options.project,
            stdout_path=json_graph_file,
            stderr_path=self.log_path('ya-dump-build-plan.log')
        )

        deps_file = self.log_path('deps.file')
        self.run_ya(
            ['dump', 'json-dep-graph', '-O', options.target_platform],
            options.project,
            stdout_path=deps_file,
            stderr_path=self.log_path('ya-dump-json-dep-graph.log')
        )

        core_checkout = self.import_ya_module()  # XXX Hack-hack-hack! Have to call ya. Fix ASAP

        source_map = core_checkout.get_project_deps_from_graph([options.project], self.source_root, deps_file)
        with open(self.log_path('source_map.file'), 'w') as source_map_file:
            json.dump(source_map, fp=source_map_file, indent=4, sort_keys=True)

        tar_file = os.path.join(room_path, 'project_source_package.tar.gz')
        self.tar_sources(tar_file, source_map, json_graph_file)
