# -*- coding: utf-8 -*-

import os.path

from sandbox.projects import resource_types

from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import svn

import sandbox.projects.common.build.parameters as build_params
from sandbox.projects.common.build.base_arcadia_project_task import BaseArcadiaProjectTask
from sandbox.projects.common.build.arcadia_project_misc import get_arcadia_project_base_target_params


LOG_FILE = 'changelog.txt'


def get_arcadia_project_log_params(group_name='Arcadia project log parameters:'):
    """\
        Параметры генерации лога, специфичные для проектов Аркадии.
        Функцию рекомендуется использовать в задачах-наследниках для составления списка входных параметров.
        :param group_name: наименование группы параметров
    """

    class Params:
        class FromParameter(parameters.SandboxIntegerParameter):
            name = 'from'
            description = 'Revision to start from'
            required = True
            group = group_name

        class LimitParameter(parameters.SandboxIntegerParameter):
            name = 'limit'
            description = 'Log entries limit (0 for unlimited)'
            required = True
            group = group_name

        class StatParameter(parameters.SandboxBoolParameter):
            name = 'stat'
            description = 'Output changed modules and paths'
            group = group_name
            default_value = False

        class DiffParameter(parameters.SandboxBoolParameter):
            name = 'diff'
            description = 'Output diffs'
            group = group_name
            default_value = False

        params = [FromParameter, LimitParameter, StatParameter, DiffParameter]

    return Params


def get_arcadia_project_target_params(group_name='Arcadia project target parameters:'):
    """\
        Параметры целей генерации лога, специфичные для проектов Аркадии.
        Функцию рекомендуется использовать в задачах-наследниках общего плана для составления списка входных параметров.
        В задачах-наследниках, генерирующих логи конкретных проектов, вместо использования данной группы параметров
        рекомендуется переопределить функцию get_targets.
        :param group_name: наименование группы параметров
    """

    return get_arcadia_project_base_target_params(group_name)


class LogArcadiaProjectTask(BaseArcadiaProjectTask):
    """\
        Генерация лога изменений для произвольного проекта или набора проектов в Аркадии.
        Таск легко наследуется и конфигурируется перегрузкой методов pre_log, get_targets, get_log_resource и
        get_log_resource_attrs.
    """

    type = 'LOG_ARCADIA_PROJECT'

    # resources quota + arcadia cache
    execution_space = 1*1024 + 10*1024

    input_parameters = build_params.get_arcadia_params() + \
        get_arcadia_project_log_params().params + \
        get_arcadia_project_target_params().params

    def pre_log(self, source_dir):
        """
            Действия, производимые перед генерацией лога.
            Функция может быть перегружена в наследниках.
            :param source_dir: директория с исходниками 'arcadia'
        """

        pass

    def get_targets(self):
        """
            Получение списка целей для генерации лога.
            Функция может быть перегружена в наследниках.
            :return: список целей
        """

        return self.__parse_list(self.ctx.get('targets', ''))

    def get_log_resource(self):
        """
            Получение описания ресурса лога.
            Поле resource_path будет перезаписано.
            Функция вызывается на стадии Enqueue.
            Функция может быть перегружена в наследниках.
            :return: словарь параметров
        """

        return {}

    def get_log_resource_attrs(self):
        """
            Получение атрибутов для ресурса лога.
            К атрибутам автоматически будет добавлена информация о ревизиях.
            Функция может быть перегружена в наследниках.
            :return: словарь атрибутов
        """

        return {}

    def initCtx(self):
        self.ctx.update(self.init_ctx_for_params(get_arcadia_project_log_params().params))

    def on_enqueue(self):
        params = self.get_log_resource()
        assert type(params) == dict
        params['resource_path'] = LOG_FILE
        params.setdefault('description', 'Arcadia Project Changelog')
        params.setdefault('resource_type', resource_types.ARCADIA_CHANGELOG)
        self.ctx['ap_log_pack'] = self.create_resource(**params).id

    def do_execute(self):
        """
            Генерация лога.
            Callback-функция для ArcadiaTask.
        """

        self.pre_execute()

        # Создаем директорию build и определяем путь к логу.
        build_dir = paths.make_folder('build')
        log_path = os.path.join(self.abs_path(), LOG_FILE)

        # Выкачиваем исходники.
        source_dir = self.get_arcadia_src_dir()

        # Пробуем применить патч.
        svn.Arcadia.apply_patch(source_dir, self.ctx.get('arcadia_patch'), self.abs_path())

        # Пре-хук.
        self.pre_log(source_dir)

        # Вычисляем параметры сборки.
        targets = self.get_targets()

        # Генерируем лог.
        revision_from = self.ctx.get('from')
        limit = self.ctx.get('limit')
        stat = self.ctx.get('stat')
        diff = self.ctx.get('diff')
        self.__ya_log(source_dir, build_dir, log_path, targets, revision_from, limit, stat, diff)

        # Финализируем ресурс лога.
        attrs = self.get_base_resource_attrs()
        attrs['arcadia_revision_to'] = attrs.pop('arcadia_revision')
        attrs['arcadia_revision_from'] = self.ctx.get('from')
        attrs['entries_limit'] = limit if limit else None
        resource_attrs = self.get_log_resource_attrs()
        assert type(resource_attrs) == dict
        attrs.update(resource_attrs)
        resource_id = self.ctx.get('ap_log_pack')
        for attr_name, attr_value in attrs.iteritems():
            channel.channel.sandbox.set_resource_attribute(resource_id, attr_name, attr_value)
        self.mark_resource_ready(resource_id)

    @classmethod
    def __ya_log(cls, source_dir, build_dir, log_path, targets, revision_from, limit, stat, diff):
        cmd = [os.path.join(source_dir, 'ya'), 'log']
        cmd.append('--build-dir=' + build_dir)
        cmd.append('--target=' + ';'.join(targets))
        cmd.append('--descriptions')
        cmd.append('--no-color')
        if revision_from:
            cmd.append('--from=%d' % revision_from)
        if limit:
            cmd.append('--limit=%d' % limit)
        if stat:
            cmd.append('--stat')
        if diff:
            cmd.append('--diff')
        with open(log_path, 'w') as log_file:
            cls.run_cmd(cmd, 'ya_log', source_dir, log_file)

    @staticmethod
    def __parse_list(value):
        elems = [e.strip() for e in value.split(';')]
        return sorted(set([e for e in elems if e]))
