import os
import platform

from sandbox import common
import sandbox.common.types.misc as ctm
from sandbox.common.types.client import Tag

from sandbox.sandboxsdk.task import SandboxTask

from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxStringParameter
from sandbox.sandboxsdk.environments import SvnEnvironment, Xcode
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxTaskUnknownError, SandboxSubprocessError
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects import resource_types
import sandbox.projects.common.constants as consts
import sandbox.projects.common.build.BuildForAllTask as bfa
from sandbox.projects.common.build import script_build_task_2


class UseSandboxGCCParameter(SandboxBoolParameter):
    name = consts.USE_SANDBOX_GCC
    description = 'Use Sandbox GCC'
    default_value = True


class ScriptNameParameter(SandboxStringParameter):
    name = 'script_name'
    description = 'Script name (from default script dir)'
    default_value = None


class ToolNameParameter(SandboxStringParameter):
    name = 'tool_name'
    description = 'Tool name for build'
    default_value = None


class InplaceArchiveParameter(SandboxBoolParameter):
    name = 'inplace_archive'
    description = 'Create archive inplace?'
    default_value = None


class XcodeVersionParameter(SandboxStringParameter):
    name = 'xcode_version'
    description = 'Xcode version to install'
    default_value = None


def get_script_build_params():
    return [
        bfa.DoNotRemoveResourcesParameter,
        UseSandboxGCCParameter,
        XcodeVersionParameter,
    ]


class ScriptBuildTask(SandboxTask):
    type = 'SCRIPT_BUILD_TASK'
    dns = ctm.DnsType.DNS64
    # noinspection PyUnresolvedReferences
    client_tags = Tag.GENERIC | Tag.Group.OSX
    cores = 17
    svn_scripts_dir = 'arcadia:/arc/trunk/arcadia/devtools/tools_build'
    logs_dir_name = 'logs'
    tool_name = None
    inplace_archive = False
    resource_type = resource_types.SCRIPT_BUILD_RESULT
    release_to = 'sandbox-releases@yandex-team.ru'
    execution_space = 10 * 1024  # 10 Gb
    environment = [SvnEnvironment(), ]

    input_parameters = get_script_build_params() + [
        bfa.VersionParameter,
        ToolNameParameter,
        ScriptNameParameter,
        InplaceArchiveParameter,
    ]

    def _get_version(self):
        return self.ctx.get(bfa.VersionParameter.name)

    def _get_do_not_remove(self):
        return self.ctx.get(bfa.DoNotRemoveResourcesParameter.name, False)

    def on_enqueue(self):
        if common.platform.get_arch_from_platform(self.arch) == ctm.OSFamily.OSX:
            self.cores = 4

    def run_script(self, version, result_dir_path, env, extra_params=None):
        if extra_params is None:
            extra_params = []
        if self.ctx[consts.XCODE_VERSION]:
            extra_params.append('--use-local-xcode')

        run_process(
            [
                '/skynet/python/bin/python',
                self.abs_path('scripts/{script_name}'.format(script_name=self.get_script_name())),
                '--version={version}'.format(version=version) if version else '',
                '--logs={logsdir}'.format(logsdir=self.path(self.logs_dir_name))
            ] + extra_params + [result_dir_path],
            shell=True, log_prefix='build', environment=env
        )

    def get_resource_description(self, version, target):
        return '{tool_name} {version} for {target}'.format(
            tool_name=self.get_tool_name(),
            version=version,
            target=target
        )

    def get_result_archive_name(self):
        return '{}.tgz'.format(self.get_tool_name())

    def get_script_name(self):
        script_name = self.ctx.get(ScriptNameParameter.name)
        script_name_by_tool_name = 'build_{tool_name}.py'.format(tool_name=self.get_tool_name())
        return script_name or script_name_by_tool_name

    def get_tool_name(self):
        return self.tool_name or self.ctx.get(ToolNameParameter.name)

    def is_inplace_archive(self):
        return self.ctx.get(InplaceArchiveParameter.name) or self.inplace_archive

    def get_build_dir_name(self):
        return 'build'

    def get_result_dir_name(self):
        return self.get_tool_name()

    def create_resource_(self, result_dir, env):
        result_archive_name = self.get_result_archive_name()

        script_build_task_2.create_resource(
            self.is_inplace_archive(),
            result_archive_name,
            self.abs_path(result_archive_name),
            result_dir,
            env
        )

        return result_archive_name

    @property
    def current_platform(self):
        return platform.platform()

    def export_scripts(self):
        Arcadia.export(self.svn_scripts_dir, 'scripts')
        if not os.path.exists('scripts'):
            raise SandboxTaskUnknownError('Failed to export build script from svn')

    def on_execute(self):
        env = dict(os.environ)
        if self.ctx[UseSandboxGCCParameter.name]:
            self.ctx['compilers'] = script_build_task_2.prepare_compilers(env)
        if self.ctx[consts.XCODE_VERSION]:
            Xcode.prepare_xcode(self.ctx[consts.XCODE_VERSION])

        # Some clients may not have valid temporary directory, so we use custom.
        if not os.path.exists('_tmp'):
            os.mkdir('_tmp')
        env['TMPDIR'] = self.abs_path('_tmp')

        # Drop path set by skynet python binary
        for env_key in ('LD_LIBRARY_PATH', 'LIBRARY_PATH'):
            if env_key in env:
                env[env_key] = script_build_task_2.drop_skynet_paths(env[env_key])

        self.export_scripts()
        result_dir_name = self.get_result_dir_name()
        remove_path(self.path(self.logs_dir_name))
        remove_path(self.abs_path(result_dir_name))

        version = self.ctx[bfa.VersionParameter.name]
        if not self.arch:
            self.arch = platform.system().lower()
        target_platform_alias = self.ctx.get('target_platform_alias', self.arch)

        build_error = None

        build_dir_name = self.get_build_dir_name()
        old_dir = None

        if build_dir_name:
            build_dir_abs = self.abs_path(build_dir_name)
            remove_path(build_dir_abs)
            os.makedirs(build_dir_abs)

            old_dir = os.getcwd()
            os.chdir(build_dir_abs)

        try:
            self.run_script(version, self.abs_path(result_dir_name), env)
        except SandboxSubprocessError as e:
            build_error = e
        finally:
            if old_dir:
                os.chdir(old_dir)

        if os.path.exists(self.logs_dir_name):
            self.create_resource(
                description='Build logs',
                resource_path=self.logs_dir_name,
                resource_type=resource_types.BUILD_LOGS
            )

        if build_error is not None:
            if build_dir_name:
                self.create_resource(
                    description='Build directory',
                    resource_path=build_dir_name,
                    resource_type=resource_types.BUILD_LOGS
                )
            raise SandboxTaskFailureError(build_error)

        if not os.path.exists(result_dir_name):
            raise SandboxTaskUnknownError("Can't find {name} in task directory".format(name=result_dir_name))

        resource_path = self.create_resource_(result_dir_name, env)

        attrs = {'platform': self.current_platform}
        if self._get_version():
            attrs['version'] = self._get_version()
        if self._get_do_not_remove():
            attrs['ttl'] = 'inf'

        self.create_resource(
            description=self.get_resource_description(version, target_platform_alias),
            resource_path=resource_path,
            resource_type=self.resource_type,
            attributes=attrs
        )

    def arcadia_info(self):
        return '', None, self.ctx.get(bfa.VersionParameter.name)
