# -*- coding: utf-8 -*-

import logging
import os
import six
import shutil

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.parameters as parameters
import sandbox.sandboxsdk.environments as environments
import sandbox.sandboxsdk.process as process

from sandbox.sandboxsdk import channel
from sandbox.projects import resource_types

import sandbox.projects.common.build.ArcadiaTask as arcadia_task
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
import sandbox.projects.common.utils as utils
import sandbox.projects.common.build.parameters as common_params
import sandbox.projects.common.constants as consts
from sandbox.projects.common import context_managers
from sandbox.projects.common.build.sdk import sdk_compat as ya_sdk_compat


class ProgramParam(parameters.SandboxStringParameter):
    name = 'program'
    description = 'Program to build and run'
    default_value = ''


class ArgsParam(parameters.SandboxStringParameter):
    name = 'program_args'
    description = 'Args to program'
    default_value = ''


class PostScriptParam(parameters.SandboxStringParameter):
    name = 'post_script'
    description = 'Post Execution Script'
    default_value = ''
    multiline = True


class DNS64Param(parameters.SandboxBoolParameter):
    name = 'dns64'
    description = 'Use DNS64'


class TaskResourceParam(parameters.SandboxStringParameter):
    name = 'resource_dir'
    description = 'Share directory as resource'
    default_value = ''


class YaExec(arcadia_task.ArcadiaTask):
    """
        Собрать и запустить пользовательскую программу
        (PROGRAM, PY2_PROGRAM, etc.), описанную сборкой ya make
    """

    type = "YA_EXEC"

    environment = [environments.SvnEnvironment()]
    client_tags = ctc.Tag.GENERIC
    cores = 17
    input_parameters = [
        parameters.SandboxArcadiaUrlParameter,
        common_params.DefinitionFlags,
        ProgramParam,
        ArgsParam,
        common_params.EnvironmentVarsParam,
        PostScriptParam,
        TaskResourceParam,
        DNS64Param,
    ] + common_params.get_aapi_parameters_as_default()

    def __init__(self, *args, **kwargs):
        arcadia_task.ArcadiaTask.__init__(self, *args, **kwargs)

    @property
    def dns(self):
        return ctm.DnsType.DNS64 if self.ctx.get(DNS64Param.name) else ctm.DnsType.DEFAULT

    def get_build_def_flags(self):
        return ya_sdk_compat.get_build_def_flags(self)

    def save_result_dir_as_resource(self, arcadia_dir, resource_dir):
        origin = os.path.join(arcadia_dir, resource_dir)
        if os.path.exists(origin):
            destination = os.path.join(os.getcwd(), resource_dir)
            if os.path.isdir(origin):
                shutil.copytree(origin, destination)
            else:
                shutil.copy(origin, destination)
            logging.info("Create resource from %s", destination)
            task_res = channel.channel.task.create_resource(
                description="Program resource",
                resource_path=destination,
                resource_type=resource_types.OTHER_RESOURCE,
                attributes={
                    'ttl': 30,
                }
            )
            channel.channel.task.mark_resource_ready(task_res.id)
        else:
            logging.error('Path %s does not exist', origin)
            raise Exception('Expected program output does not exist')

    def get_arcadia_src_dir(self):
        if ya_sdk_compat.is_arc_path(self) or utils.get_or_default(self.ctx, common_params.UseArcadiaApiFuseAsDefault):
            arcadia_path = arcadia_sdk.mount_arc_path(
                self.ctx.get(consts.ARCADIA_URL_KEY), arc_oauth_token=os.getenv('ARC_TOKEN'),
                use_arc_instead_of_aapi=utils.get_or_default(self.ctx, common_params.UseArcInsteadOfArcadiaApiAsDefault)
            )
            return arcadia_path
        else:
            return context_managers.nullcontext(arcadia_task.ArcadiaTask.get_arcadia_src_dir(self))

    def on_execute(self):
        program_path = utils.get_or_default(self.ctx, ProgramParam)
        post_script = utils.get_or_default(self.ctx, PostScriptParam)
        resource_dir = utils.get_or_default(self.ctx, TaskResourceParam)
        env_vars = self.get_env_vars()

        if program_path:
            build_target = os.path.dirname(program_path)

            build_def_flags = self.get_build_def_flags()

            with self.get_arcadia_src_dir() as arcadia_dir:
                logging.debug("Arcadia src dir %s", arcadia_dir)
                logging.info("Target to build %s", build_target)
                build_return_code = arcadia_sdk.do_build(
                    'ya', arcadia_dir, [build_target],
                    clear_build=False,
                    results_dir=arcadia_dir,
                    def_flags=build_def_flags
                )

                logging.info("Build returned %s", build_return_code)

                process.run_process(
                    [os.path.join(arcadia_dir, program_path), ] + utils.get_or_default(self.ctx, ArgsParam).split(),
                    log_prefix='program',
                    outputs_to_one_file=False,
                    shell=True,
                    work_dir=arcadia_dir,
                    environment=env_vars,
                )

                if resource_dir:
                    self.save_result_dir_as_resource(arcadia_dir, resource_dir)

                if post_script:
                    post_script = "import os; os.chdir('{}'); {}".format(arcadia_dir, post_script)
                    logging.info('Script to execute: %s', post_script)
                    six.exec_(post_script, globals(), locals())

        elif post_script:
            logging.info('Script to execute: %s', post_script)
            six.exec_(post_script, globals(), locals())


__Task__ = YaExec
