# -*- coding: utf-8 -*-

import errno
import itertools as it
import json
import logging
import operator as op
import os
import os.path
import platform
import six
import sys
import tempfile
import traceback

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.common.mds import compression

import sandbox.projects.common.build.YaMake2 as ya_make_2
import sandbox.projects.devtools.resources as devtools_resources

from sandbox import common
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.environments import Xcode

import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.projects import resource_types
from sandbox.projects.common import context_managers
from sandbox.projects.common import file_utils
from sandbox.projects.common.build import resource_types as build_resource_types
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.vcs import arc
from sandbox.projects.common.build import dependency_banner
from sandbox.projects.common.build.base_arcadia_project_task import BaseArcadiaProjectTask
from sandbox.projects.common.build.sdk import sdk_compat as ya_sdk_compat


re_target_android = ya_sdk_compat.re_target_android

BUILD_DIR = ya_sdk_compat.BUILD_DIR
PACK_DIR = ya_sdk_compat.PACK_DIR
DEFAULT_RESULT_DESCRIPTION = ya_sdk_compat.DEFAULT_RESULT_DESCRIPTION

STRICT_TAGS = [ctc.Tag.PORTOD]


def get_arcadia_project_build_params():
    """
        Параметры сборки, специфичные для проектов Аркадии.
        Функцию рекомендуется использовать в задачах-наследниках для составления списка входных параметров.
    """

    return build_params.get_sdk1_parameters(ya_make_2.ArcadiaProjectBuildParameters)


ya_make_build_params = get_arcadia_project_build_params


def _sandbox_task_params():
    return build_params.get_sdk1_parameters(ya_make_2.SandboxTaskParameters)


class YaMakeTask(BaseArcadiaProjectTask):
    """\
        Сборка произвольного проекта или набора проектов в Аркадии.
        Таск считается проваленным, если какой-либо из этапов провалился. Опционально запускается тестирование.
        Таск легко наследуется и конфигурируется перегрузкой методов pre_build, get_targets, get_arts,
        get_arts_source, get_resources и get_resources_attrs. Для получения определенных объектов сборки в качестве
        отдельного ресурса необходимо в Build artifacts указать пути к файлам, которые будут получены отдельным
        ресурсом.
    """

    type = 'YA_MAKE'

    execution_space = ya_make_2.YA_MAKE_DISK_SPACE
    input_parameters = build_params.get_sdk1_parameters(ya_make_2.YaMakeParameters)
    client_tags = ya_make_2.YA_MAKE_CLIENT_TAGS

    build_output_ttl = 4
    build_output_html_ttl = 4

    MULTISLOT_TASKS = {"YA_MAKE"}
    MULTISLOT_OWNERS = {
        ("SANDBOX", "OTHER", "guest", "aripinen"): {"ram": 16384, "cores": 4},
        ("SEARCH-RELEASERS",): {"ram": 32768, "cores": 8}
    }
    MULTISLOT_OWNERS_DICT = {
        owner: requirements
        for owner, requirements in
        it.chain(*(tuple(map(lambda name: (name, value), names)) for names, value in six.iteritems(MULTISLOT_OWNERS)))
    }

    @property
    def dns(self):
        return ctm.DnsType.DNS64 if self.ctx.get(consts.DNS64) else ctm.DnsType.DEFAULT

    @property
    def privileged(self):
        return self.ctx.get(consts.PRIVILEGED, False)

    def pre_build(self, source_dir):
        """
            Действия, производимые перед сборкой.
            Функция может быть перегружена в наследниках.
            :param source_dir: директория с исходниками 'arcadia'
        """
        pass

    def post_build(self, source_dir, output_dir, pack_dir):
        """
            Действия, производимые после сборки и тестирования.
            Функция может быть перегружена в наследниках.
            :param source_dir: директория с исходниками 'arcadia'
            :param output_dir: директория с результатами
            :param pack_dir: директория, куда были сложены артефакты сборки
        """

        ya_sdk_compat.post_build(self, source_dir, output_dir)

        logs_ttl = self.ctx.get(consts.BUILD_LOGS_TTL, None)
        if logs_ttl:
            try:
                common.rest.Client().resource[self._log_resource.id].attribute.create({"name": "ttl", "value": logs_ttl})
            except common.rest.Client().HTTPError as ex:
                if ex.status == six.moves.http_client.CONFLICT:
                    common.rest.Client().resource[self._log_resource.id].attribute['ttl'].update(logs_ttl)
                else:
                    raise

        # TODO move to the ya_sdk_compat module (see copy-paste in YaMake2 task)
        if self.ctx.get(consts.TESTS_REQUESTED, False) and self.ctx.get(consts.CANONIZE_TESTS):
            _, mode = ya_sdk_compat.check_parameters_compatibility(self, self.ctx.get(consts.MAKE_CONTEXT_ON_DISTBUILD))
            resource = self._create_resource(
                "Zipatch for Canonization",
                "canonization.zipatch",
                build_resource_types.ZIPATCH_OUTPUT
            )
            if mode in (ya_sdk_compat.GET_ARCADIA_USE_ARC_VCS, ya_sdk_compat.GET_ARCADIA_USE_ARC_FUSE):
                arc_repo = arc.Arc()
                status = arc_repo.status(source_dir, True)
                logging.info("arc status --json output: {}".format(status))
                if status['status']:
                    branch = "canonize-{}".format(str(uuid.uuid4()))
                    arc_repo.checkout(source_dir, branch, create_branch=True)
                    arc_repo.commit(source_dir, "canondata")
                    info = arc_repo.info(source_dir)
                    commit_hash = info["hash"]
                    logging.info("commit hash: {}".format(commit_hash))
                    arc_repo.push(source_dir, refspecs=[commit_hash])

                    arc_repo.create_zipatch(source_dir, commit_hash, str(resource.abs_path()), no_copy=True)

                    if getattr(self.Parameters, consts.CANONIZE_PR, False):
                        logging.info("creating PR with canonization data")
                        pr_result = arc_repo.pr_create(
                            source_dir,
                            message="Canonization from sandbox task {}".format(self.id),
                            publish=True,
                            auto=True,
                            no_commits=True,
                            as_dict=True,
                        )
                        logging.info("arc pr create result: %s", pr_result)
                        self.set_info("created PR with changes in canondata: {}".format(pr_result["url"]))
                    else:
                        self.set_info("you can apply changes in canondata to your branch by calling\n"
                                      "arc cherry-pick {}".format(commit_hash))
            else:
                logging.info("Make zipatch for test canonization")
                res = sdk.do_create_zipatch(source_dir, resource.abs_path())
                if res == 0:
                    self.set_info("Zipatch created, can be loaded from {}".format(resource.proxy_url))

                    base_rev = self.ctx.get('ap_arcadia_revision')
                    if base_rev:
                        self.set_info("You can apply changes to your working copy by calling\n   "
                                      "ya unshelve -u={} -r {}".format(resource.proxy_url, base_rev))
                else:
                    self.set_info("Failed to create zipatch, RetCode: {}".format(res))

    def get_build_type(self):
        """
            Получение типа сборки
            Функция может быть перегружена в наследниках
            :return: тип сборки
        """

        return ya_sdk_compat.get_build_type(self)

    def get_binaries_for_ya_bloat(self):
        """
            Получить список бинарных артефактов сборки для анализа их с помощью ya bloat
            Наследники могут определить список путей к артефактам сборки.
            :return: список бинарных артефактов сборки
        """
        return ya_sdk_compat.get_binaries_for_ya_bloat(self)

    def get_build_def_flags(self):
        """
            Получение дополнительных аргументов для сборки
            Наследники могут добавлять дополнительные аргументы к сборочным утилитам
            :return: список дополнительных аргументов
        """

        return ya_sdk_compat.get_build_def_flags(self)

    def get_target_platform(self):
        """
            Получение значения параметра сборки 'Target platform'
        """

        return ya_sdk_compat.get_target_platform(self)

    def get_target_platform_flags(self):
        """
            Получение значения параметра сборки 'Target platform flags'
        """

        return ya_sdk_compat.get_target_platform_flags(self)

    def get_host_platform_flags(self):
        """
            Получение значения параметра сборки 'Host platform flags'
        """

        return ya_sdk_compat.get_host_platform_flags(self)

    def get_targets(self):
        """
            Получение списка целей для сборки.
            Функция может быть перегружена в наследниках.
            :return: список целей
        """

        return ya_sdk_compat.get_targets(self)

    def get_arts(self):
        """
            Получение списка артефактов для упаковки.
            Функция может быть перегружена в наследниках.
            :return: список артефактов
        """

        return ya_sdk_compat.get_arts(self)

    def get_arts_source(self):
        """
            Получение списка артефактов из исходников для упаковки.
            Функция может быть перегружена в наследниках.
            :return: список артефактов
        """

        return ya_sdk_compat.get_arts_source(self)

    def get_resources(self):
        """
            Получение описаний ресурсов.
            Параметр resource_path должен быть аналогичен dest-путям артефактов сборки.
            Функция вызывается на стадии Enqueue.
            Функция может быть перегружена в наследниках.
            :return: словарь именованных словарей параметров
        """

        return ya_sdk_compat.get_resources(self)

    def get_resources_attrs(self):
        """
            Получение атрибутов для ресурсов.
            К атрибутам автоматически будут добавлены базовые атрибуты ревизии,
            времени и платформы сборки.
            Функция может быть перегружена в наследниках.
            :return: словарь именованных словарей атрибутов
        """

        return ya_sdk_compat.get_resources_attrs(self)

    def get_check_deps(self):
        """
            Dump dependency list and validate it
        """
        return ya_sdk_compat.get_check_deps(self)

    def get_output_dir(self):
        return ya_sdk_compat.get_output_dir(self)

    def initCtx(self):
        self.ctx.update(
            self.init_ctx_for_params(build_params.get_sdk1_parameters(ya_make_2.ArcadiaProjectBuildParameters))
        )

    def on_enqueue(self):
        # Redefine max_restarts which might be specified by caller
        if self.ctx.get('max_restarts'):
            self.max_restarts = int(self.ctx['max_restarts'])

        if self.ctx.get('requested_arch'):
            self.arch = self.ctx.get('requested_arch')

        try:
            require_multislot = ya_sdk_compat.is_required_multislot(self.ctx.get(consts.SANDBOX_TAGS, "GENERIC"))
            if require_multislot:
                if not self.cores:
                    self.cores = 1
                if not self.required_ram or self.required_ram < 8192:
                    self.required_ram = 8192
        except Exception:
            logging.exception("YA_MAKE.on_enqueue():")
        if self.type in self.MULTISLOT_TASKS and self.owner in self.MULTISLOT_OWNERS_DICT:
            try:
                self.cores = self.cores or self.MULTISLOT_OWNERS_DICT[self.owner]["cores"]
                self.required_ram = self.required_ram or self.MULTISLOT_OWNERS_DICT[self.owner]["ram"]
            except:
                logging.exception("Can't add MULTISLOT to sandbox_tags for task %s", self.id)
        # elif (
        #     requested_platform.startswith("linux") and
        #     not self.cores and
        #     self.required_ram <= 8192 and
        #     self.ctx[consts.SANDBOX_TAGS] != "GENERIC"
        # ):
        #     patch = self.ctx.get(consts.ARCADIA_PATCH_KEY)
        #     make_context_on_distbuild = self.ctx.get(consts.MAKE_CONTEXT_ON_DISTBUILD)
        #     use_aapi_fuse = self.ctx.get(consts.USE_AAPI_FUSE)
        #
        #     # Патч не работает с селективным чекаутом
        #     checkout = False if patch or make_context_on_distbuild or use_aapi_fuse else self.should_checkout()
        #
        #     self.cores = (
        #         1
        #         if use_aapi_fuse or make_context_on_distbuild or checkout or require_multislot else
        #         24
        #     )

        if self.ctx.get(consts.DNS64_REQUIRED):
            self.dns = ctm.DnsType.DNS64

        if self.is_build_bundle_task():
            return

        self.ctx['ap_packs'] = {}
        resources = self.get_resources()
        ya_sdk_compat.process_resources(self, resources)

        build_system = self.ctx.get(consts.BUILD_SYSTEM_KEY)
        if self.ctx.get(consts.RAM_DRIVE_SIZE) and build_system != consts.DISTBUILD_BUILD_SYSTEM:
            self.ramdrive = self.RamDrive(
                ctm.RamDriveType.TMPFS,
                self.ctx.get(consts.RAM_DRIVE_SIZE),
                None
            )
        else:
            self.ramdrive = None

        if self.get_check_deps():
            dependency_banner.create_resources(self)

        # We use Clang from the Android NDK for Android targets and
        # it requires at least Ubuntu 14.04 as a Linux host.
        if re_target_android.search(self.get_target_platform_flags() or ''):
            self.arch = 'linux_ubuntu_14.04_trusty'

        if common.config.Registry().common.installation == ctm.Installation.LOCAL:
            return

        try:
            requested_platform = self.ctx.get('requested_arch') or self.arch
            if not requested_platform or requested_platform == ctm.OSFamily.ANY:
                requested_platform = 'linux'

            requested_platform = common.platform.get_platform_alias(requested_platform)
            # save for debug purpose - hosts_match_score is processed on sandbox server,
            # but ctx will be transfered to the task instance and will be available in logs
            self.ctx['requested_platform'] = requested_platform
            self.arch = requested_platform
        except:
            logging.exception("Error in setting requested platform on_enqueue of task #%s type %s", self.id, self.type)

        try:
            tags = self.ctx.get(consts.SANDBOX_TAGS, 'GENERIC').strip().upper()
            # no tags specified - return scores obtained after applying platform filter
            if not tags:
                return

            predicates = ctc.Tag.Query.predicates(tags)
            tags = ctc.Tag.Query(tags)
            unwanted_tags = set(STRICT_TAGS)

            for tag in STRICT_TAGS:
                for requested, _ in predicates:
                    if tag in requested and tag in unwanted_tags:
                        unwanted_tags.remove(tag)

            if unwanted_tags:
                tags &= reduce(op.and_, map(lambda _: ~ctc.Tag.Query(_), unwanted_tags))
            self.client_tags = self.__class__.client_tags & tags
        except:
            logging.exception("Error in setting client tags on_enqueue of task #%s type %s", self.id, self.type)

    def clone(self):
        return ya_sdk_compat.clone(self)

    def check_aapi_available(self):
        return ya_sdk_compat.check_aapi_available(self)

    def get_source_dirs(self, mode):
        try:
            return ya_sdk_compat.get_source_dirs(self, mode)
        except OSError as e:
            if e.errno in [errno.EACCES]:
                logging.exception("Exception while ya_sdk_compat.get_source_dirs")
                raise common.errors.TemporaryError("Unable to get arcadia due to infrastructure error (code {})".format(e.errno))
            raise

    def do_execute(self):
        """
            Сборка.
            Callback-функция для ArcadiaTask.
        """

        self.pre_execute()

        xcode_version = self.ctx.get(consts.XCODE_VERSION, None)
        if xcode_version:
            logging.debug('Xcode version: {}'.format(xcode_version))
            Xcode.prepare_xcode(xcode_version)

        build_path, output_dir = ya_sdk_compat.get_build_path(self)
        self.ctx[consts.OUTPUT_DIR_PATH] = output_dir

        local_output_dir = ya_sdk_compat.get_local_output_dir(self)

        results_dir = output_dir if local_output_dir else None

        html_results_resource = None
        if results_dir and self.ctx.get(consts.CREATE_RESULTS_RESOURCE, True):
            results_resource = self._create_results_resource()
            if self.ctx.get(consts.CREATE_HTML_RESULTS_RESOURCE, False):
                html_results_resource = self._create_html_results_resource()
        else:
            results_resource = None

        if self.ctx.get('hosts_match_error'):
            error_message = self.ctx.get('hosts_match_error')
            # Генерируем results.json для фат теста, запущенного из TE, для того,
            # чтобы уведомить пользователя о неправильно описанных требованиях/тегах в ya.make
            if self.ctx.get(consts.AUTOCHECK_FAT_TEST):
                filename = os.path.join(output_dir, 'results.json')
                self._generate_fake_results(error_message, filename)
                self.__publish_results(results_resource, filename, html_results_resource=html_results_resource)
                streaming_link = self.ctx.get(consts.STREAMING_REPORT_URL, None)
                streaming_check_id = self.ctx.get(consts.STREAMING_REPORT_ID, None)
                if streaming_link is not None and streaming_check_id is not None:
                    self._report_streaming_results(streaming_link, streaming_check_id, filename)
                return
            raise errors.SandboxTaskFailureError(error_message)
        logging.info(
            "Hosts matched task requirements: %s/%s for parsed predicated: %s",
            self.ctx.get('hosts_matched'), self.ctx.get('hosts_total'), self.ctx.get('client_tags_predicates')
        )

        # Запоминаем платформу сборки.
        self.ctx['ap_build_platform'] = platform.platform()

        # Создаем директории build (для сборки)
        # и room (для эмуляции корня arc).
        room_dir = paths.make_folder('room')
        pack_dir = paths.make_folder(PACK_DIR)
        source_dir = os.path.join(room_dir, 'arcadia')

        if self.privileged:
            build_dir = paths.make_folder('local_build_cache')
        else:
            build_dir = None

        # Вычисляем дополнительные опции сборки.
        build_type = self.get_build_type()

        patch = self.ctx.get(consts.ARCADIA_PATCH_KEY)

        build_system = self.ctx.get(consts.BUILD_SYSTEM_KEY)
        make_context_on_distbuild = self.ctx.get(consts.MAKE_CONTEXT_ON_DISTBUILD)

        ya_sdk_compat.check_dist_build_parameters(self, build_system, make_context_on_distbuild)
        checkout, mode = ya_sdk_compat.check_parameters_compatibility(self, make_context_on_distbuild)

        arcadia_ctx, arcadia_tests_data_ctx = self.get_source_dirs(mode)

        with arcadia_ctx as arcadia_path, arcadia_tests_data_ctx as arcadia_tests_data:
            logging.info("Arcadia source dir: %s", arcadia_path)
            logging.info("Arcadia tests data source dir: %s", arcadia_tests_data)
            self.__ensure_link(arcadia_path, source_dir)

            frepkage_root = self.ctx.get(consts.ARCADIA_FREPKAGE_ROOT)

            if patch:
                sdk.apply_patch(self, source_dir, patch, self.abs_path(), checkout)

            # Пре-хук.
            self.pre_build(source_dir)

            # Вычисляем параметры сборки.
            targets = self.get_targets()
            arts = self.get_arts()
            arts_source = self.get_arts_source()

            logging.debug('Targets are %s, arts are %s, arts source is %s', targets, arts, arts_source)

            # Проверяем осмысленность конфигурации.
            if self.ctx.get('ap_packs') and not (arts or arts_source):
                raise errors.SandboxTaskFailureError('Resources are registered, but no artifacts are set.')

            # Собираем.
            build_def_flags = self.get_build_def_flags()
            if self.ctx.get(consts.USE_SYSTEM_PYTHON):
                build_def_flags += ' -DUSE_ARCADIA_PYTHON=no'
            if '-DNO_SRCLINK' not in build_def_flags:
                build_def_flags += ' -DNO_SRCLINK=yes'

            if targets:
                split_entries = lambda name: filter(None, [e.strip() for e in (self.ctx.get(name, '') or '').split(',')])

                if self.get_check_deps() and not frepkage_root:
                    dependency_banner.check_dependencies(self, source_dir, [os.path.dirname(x) for x in targets])

                timeout = int(self.ctx.get(consts.YA_TIMEOUT, 10800))
                use_dev_version = self.ctx.get(consts.USE_DEV_VERSION, False)
                junit_report_path = self.log_path("junit_report.xml") if self.ctx.get(consts.JUNIT_REPORT) else None
                coverage = self.ctx.get(consts.COVERAGE, False)
                coverage_prefix_filter = self.ctx.get(consts.COVERAGE_PREFIX_FILTER) or None
                coverage_exclude_regexp = self.ctx.get(consts.COVERAGE_EXCLUDE_REGEXP) or ''
                coverage_unified_agent = self.ctx.get(consts.COVERAGE_UNIFIED_AGENT) or False
                coverage_unified_agent_sid = self.ctx.get(consts.COVERAGE_UNIFIED_AGENT_SID) or ''
                coverage_unified_agent_strict = self.ctx.get(consts.COVERAGE_UNIFIED_AGENT_STRICT) or False
                sanitize = self.ctx.get(consts.SANITIZE, False)
                lto = self.ctx.get(consts.LTO, False)
                thinlto = self.ctx.get(consts.THINLTO, False)
                pgo_add = self.ctx.get(consts.PGO_ADD, False)
                pgo_use = self.ctx.get(consts.PGO_USE, None)
                pgo_merge_timeout = self.ctx.get(consts.PGO_MERGE_TIMEOUT, 600)
                vault_owner = self.ctx.get(consts.VAULT_OWNER, self.author)
                vault_key_name = self.ctx.get(consts.VAULT_KEY_NAME)
                test_params = self.ctx.get(consts.TEST_PARAMS_KEY) or {}
                clear_build = self.ctx.get(consts.CLEAR_BUILD_KEY)
                strip_binaries = self.ctx.get(consts.STRIP_BINARIES, False)
                keep_on = self.ctx.get(consts.KEEP_ON)
                check_rc = self.ctx.get(consts.CHECK_RETURN_CODE)
                tests_requested = self.ctx.get(consts.TESTS_REQUESTED, False)
                allure_report = self.ctx.get(consts.ALLURE_REPORT, False)
                allure_report_ttl = self.ctx.get(consts.ALLURE_REPORT_TTL)
                test_type_filter = split_entries(consts.TEST_TYPE_FILTER)
                fuzz_opts = self.ctx.get(consts.FUZZ_OPTS) or []
                # TODO add parameter to the task
                sanitize_coverage = self.ctx.get(consts.SANITIZE_COVERAGE)
                cache_namespace = self.ctx.get(consts.CACHE_NAMESPACE)
                sandbox_token = self.ctx.get(consts.SANDBOX_TOKEN)
                musl = self.ctx.get(consts.MUSL, False)
                java_coverage = self.ctx.get(consts.JAVA_COVERAGE, False)
                test_size_filter = split_entries(consts.TEST_SIZE_FILTER)
                save_links_for_files = split_entries(consts.SAVE_LINKS_FOR_FILES)

                yt_token = ya_sdk_compat.get_yt_store_vault_data(self)

                yt_store_params = sdk.YtStoreParams(
                    self.ctx.get(consts.YA_YT_STORE, True),
                    yt_token,
                    self.ctx.get(consts.YA_YT_PROXY, None),
                    self.ctx.get(consts.YA_YT_DIR, None),
                    self.ctx.get(consts.YA_YT_PUT, False),
                    self.ctx.get(consts.YA_YT_STORE_CODEC, None),
                    self.ctx.get(consts.YA_YT_REPLACE_RESULT, False),
                    self.ctx.get(consts.YA_YT_REPLACE_RESULT_ADD_OBJECTS, False),
                    self.ctx.get(consts.YA_YT_REPLACE_RESULT_RM_BINARIES, False),
                    self.ctx.get(consts.YA_YT_MAX_CACHE_SIZE, None),
                    self.ctx.get(consts.YA_YT_STORE_EXCLUSIVE, False),
                    self.ctx.get(consts.YA_YT_STORE_THREADS, None),
                    self.ctx.get(consts.YA_YT_STORE_REFRESH_ON_READ, False),
                )
                yt_store_params.report_status(self)

                if self.ctx.get(consts.RUN_TAGGED_TESTS_ON_YT, False):
                    vanilla_execute_yt_token_path = tempfile.NamedTemporaryFile().name
                    file_utils.write_file(vanilla_execute_yt_token_path, yt_token)
                else:
                    vanilla_execute_yt_token_path = None

                # Выкачиваем и обновляем нужное поддерево ATD
                if (
                    tests_requested
                    and not make_context_on_distbuild
                    and arcadia_tests_data is None
                    and not frepkage_root
                ):
                    arcadia_tests_data = ya_sdk_compat.update_atd(
                        self, checkout, source_dir, targets, clear_build, build_system, use_dev_version
                    )

                # Получаем профили PGO
                if pgo_use:
                    pgo_use = [self.sync_resource(res_id) for res_id in pgo_use]
                    logging.debug('PGO profiles: %s' % pgo_use)

                if fuzz_opts:
                    fuzz_opts = fuzz_opts.split(" ")
                if self.ctx.get(consts.BUILD_OUTPUT_TTL) == 0:
                    attrs = {'ttl': 1}
                else:
                    attrs = {'ttl': self.ctx.get(consts.BUILD_OUTPUT_TTL) or self.build_output_ttl}

                if self.ctx.get(consts.PACK_RESOURCE):
                    attrs['pack_tar'] = compression.base.CompressionType.TAR

                build_output_res = self.create_resource(
                    description='Build output',
                    resource_path=output_dir,
                    resource_type=resource_types.BUILD_OUTPUT,
                    attributes=attrs
                )
                self.__create_default_file(os.path.join(output_dir, "default.json"))

                build_exception_str = ''
                returncode = -1
                try:
                    if vault_owner and vault_key_name:
                        ssh_manager = ssh.Key(self, vault_owner, vault_key_name)
                    else:
                        ssh_manager = context_managers.nullcontext()
                    with ssh_manager:
                        env = self.get_env_vars()
                        env["YA_DUMP_RAW_RESULTS"] = "1"
                        returncode = sdk.do_build(
                            build_system, source_dir, targets, build_type,
                            clear_build=clear_build,
                            build_dir=build_dir,
                            strip_binaries=strip_binaries,
                            test=tests_requested, test_filters=self.ctx.get(consts.TEST_FILTERS),
                            output_only_tests=self.ctx.get(consts.OUTPUT_ONLY_TESTS, False),
                            junit_report_path=junit_report_path,
                            results_dir=results_dir, def_flags=build_def_flags,
                            use_dev_version=use_dev_version, timeout=timeout, coverage=coverage,
                            coverage_prefix_filter=coverage_prefix_filter,
                            coverage_exclude_regexp=coverage_exclude_regexp, sanitize=sanitize,
                            coverage_unified_agent=coverage_unified_agent,
                            coverage_unified_agent_sid=coverage_unified_agent_sid,
                            coverage_unified_agent_strict=coverage_unified_agent_strict,
                            coverage_unified_agent_uids_res_id=self.ctx.get(consts.COVERAGE_UNIFIED_AGENT_UIDS_RESOURCE_ID),
                            coverage_unified_agent_uids_file_path=self.ctx.get(consts.COVERAGE_UNIFIED_AGENT_UIDS_FILE_PATH),
                            coverage_unified_agent_failed_uids_file_path=self.ctx.get(consts.COVERAGE_UNIFIED_AGENT_FAILED_UIDS_FILE_PATH),
                            lto=lto, thinlto=thinlto,
                            build_resource_id=build_output_res.id,
                            test_params=test_params,
                            ram_drive_path=ya_sdk_compat.get_ram_drive_path(self),
                            keep_on=keep_on,
                            check_rc=check_rc,
                            target_platform=self.get_target_platform(),
                            target_platform_flags=self.get_target_platform_flags(),
                            host_platform_flags=self.get_host_platform_flags(),
                            report_tests_only=self.ctx.get(consts.REPORT_TESTS_ONLY),
                            test_log_level=self.ctx.get(consts.TEST_LOG_LEVEL),
                            test_tag=self.ctx.get(consts.TEST_TAG),
                            canonize_tests=self.ctx.get(consts.CANONIZE_TESTS),
                            allure_report=allure_report,
                            allure_report_ttl=allure_report_ttl,
                            disable_test_timeout=self.ctx.get(consts.DISABLE_TEST_TIMEOUT),
                            force_build_depends=self.ctx.get(consts.FORCE_BUILD_DEPENDS),
                            force_vcs_info_update=self.ctx.get(consts.FORCE_VCS_INFO_UPDATE, False),
                            ignore_recurses=self.ctx.get(consts.IGNORE_RECURSES),
                            patch=patch,
                            test_size_filter=test_size_filter,
                            test_threads=self.ctx.get(consts.TEST_THREADS),
                            arcadia_tests_data=arcadia_tests_data,
                            cache_test_results=self.ctx.get(consts.CACHE_TEST_RESULTS),
                            tests_retries=self.ctx.get(consts.TESTS_RETRIES),
                            pgo_add=pgo_add, pgo_use=pgo_use, pgo_merge_timeout=pgo_merge_timeout,
                            test_type_filter=test_type_filter,
                            checkout=checkout,
                            fuzzing=self.ctx.get(consts.FUZZING),
                            fuzz_opts=fuzz_opts,
                            sanitize_coverage=sanitize_coverage,
                            sanitizer_flags=self.ctx.get(consts.SANITIZER_FLAGS),
                            cache_namespace=cache_namespace,
                            sandbox_token=sandbox_token,
                            ssh_user=vault_owner,
                            musl=musl,
                            no_src_changes=self.ctx.get(consts.NO_SRC_CHANGES, True),
                            resource_owner=self.ctx.get(consts.RESOURCE_OWNER, None),
                            streaming_link=self.ctx.get(consts.STREAMING_REPORT_URL, None),
                            streaming_check_id=self.ctx.get(consts.STREAMING_REPORT_ID, None),
                            collect_test_cores=self.ctx.get(consts.COLLECT_TEST_CORES, True),
                            sonar=self.ctx.get(consts.SONAR, False),
                            sonar_options=self.ctx.get(consts.SONAR_OPTIONS, None),
                            sonar_project_filter=self.ctx.get(consts.SONAR_PROJECT_FILTER, None),
                            sonar_default_project_filter=self.ctx.get(consts.SONAR_DEFAULT_PROJECT_FILTER, False),
                            java_coverage=java_coverage,
                            sandbox_uploaded_resource_ttl=self.ctx.get(consts.SANDBOX_UPLOADED_RESOURCE_TTL, None),
                            multiplex_ssh=self.ctx.get(consts.MULTIPLEX_SSH, False),
                            build_execution_time=self.ctx.get(consts.BUILD_EXECUTION_TIME, None),
                            coverage_yt_token_path=self.ctx.get(consts.COVERAGE_YT_TOKEN_PATH, None),
                            cpp_coverage_type=self.ctx.get(consts.CPP_COVERAGE_TYPE, None),
                            python_coverage=self.ctx.get(consts.PYTHON_COVERAGE, False),
                            go_coverage=self.ctx.get(consts.GO_COVERAGE, False),
                            nlg_coverage=self.ctx.get(consts.NLG_COVERAGE, False),
                            upload_coverage=self.ctx.get(consts.UPLOAD_COVERAGE, False),
                            merge_coverage=self.ctx.get(consts.MERGE_COVERAGE, False),
                            graph_timestamp=self.ctx.get(consts.GRAPH_TIMESTAMP, None),
                            download_artifacts=self.ctx.get(consts.DOWNLOAD_ARTIFACTS_FROM_DISTBUILD, True),
                            drop_graph_result_before_tests=self.ctx.get(consts.DROP_GRAPH_RESULT_BEFORE_TESTS, False),
                            save_links_for_files=save_links_for_files,
                            javac_options=self.ctx.get(consts.JAVAC_OPTIONS, None),
                            jvm_args=self.ctx.get(consts.JVM_ARGS_KEY, None),
                            strip_skipped_test_deps=self.ctx.get(consts.STRIP_SKIPPED_TEST_DEPS, False),
                            dist_priority=self.ctx.get(consts.DISTBUILD_PRIORITY, None),
                            coordinators_filter=self.ctx.get(consts.COORDINATORS_FILTER, None),
                            make_context_on_distbuild=make_context_on_distbuild,
                            separate_result_dirs=self.ctx.get(consts.SEPARATE_RESULT_DIRS, False),
                            fast_clang_coverage_merge=self.ctx.get(consts.FAST_CLANG_COVERAGE_MERGE, False),
                            new_dist_mode=self.ctx.get(consts.NEW_DIST_MODE, False),
                            skip_test_console_report=self.ctx.get(consts.SKIP_TEST_CONSOLE_REPORT, False),
                            yt_store_params=yt_store_params,
                            build_output_html_ttl=self.ctx.get(consts.BUILD_OUTPUT_HTML_TTL) or self.build_output_html_ttl,
                            keep_alive_all_streams=self.ctx.get(consts.KEEP_ALIVE_ALL_STREAMS, False),
                            frepkage_root=frepkage_root,
                            frepkage_target_uid=self.ctx.get(consts.ARCADIA_FREPKAGE_TARGET_UID),
                            run_tagged_tests_on_sandbox=self.ctx.get(consts.RUN_TAGGED_TESTS_ON_SANDBOX),
                            run_tagged_tests_on_yt=self.ctx.get(consts.RUN_TAGGED_TESTS_ON_YT),
                            merge_split_tests=self.ctx.get(consts.MERGE_SPLIT_TESTS, True),
                            vanilla_execute_yt_token_path=vanilla_execute_yt_token_path,
                            trace_ya_output=self.ctx.get(consts.TRACE_YA_OUTPUT, False),
                            failed_tests_cause_error=self.ctx.get(consts.FAILED_TESTS_CAUSE_ERROR, True),
                            retriable_displaced_build=self.ctx.get(consts.RETRIABLE_DISPLACED_BUILD, False),
                            dir_outputs=self.ctx.get(consts.DIR_OUTPUTS, False),
                            env=env,
                            use_prebuilt_tools=self.ctx.get(consts.USE_PREBUILT_TOOLS, None),
                            distbuild_pool=self.ctx.get(consts.DISTBUILD_POOL, None),
                            fuzz_minimize=self.ctx.get(consts.FUZZ_FORCE_MINIMIZATION, False),
                        )
                        self.ctx['build_returncode'] = returncode
                except (common.errors.TemporaryError, errors.SandboxSubprocessTimeoutError):
                    logging.debug(traceback.format_exc())
                    # Отключаем выброс исключения при отсутствии results.json
                    # https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/common/build/sdk/sdk_compat.py?blame=true&rev=5985354#L191
                    check_rc = True
                    raise
                except common.errors.SandboxException as e:
                    build_exception_str = str(e) or str(type(e))
                    logging.debug(traceback.format_exc())
                    raise
                except Exception as e:
                    build_exception_str = str(e) or str(type(e))
                    logging.debug(traceback.format_exc())
                    if check_rc:
                        raise
                finally:
                    # Публикуем результаты, если они есть
                    try:
                        from sandbox.sandboxsdk import process
                        p = process.run_process(["chmod", "--recursive", "755", output_dir], log_prefix='chmod_output_dir')
                        p.communicate()
                    except Exception as e:
                        logging.error("Error while chmod: %s", e)

                    # Упаковываем артефакты до того как ресурс помечается как ready
                    self.__pack_arts(arts, pack_dir, output_dir)
                    self.__pack_arts(arts_source, pack_dir, source_dir)

                    self.mark_resource_ready(build_output_res.id)
                    try:
                        self.__publish_results(
                            results_resource,
                            os.path.join(output_dir, 'results.json'),
                            check_rc,
                            html_results_resource,
                        )
                    except Exception as e:
                        logging.debug(traceback.format_exc())
                        if build_exception_str:
                            message = "{}\n\n{}".format(build_exception_str, e.message)
                        else:
                            message = e.message
                        six.reraise(type(e), type(e)(message), sys.exc_info()[2])
            else:  # not targets
                # Упаковываем артефакты
                self.__pack_arts(arts, pack_dir, output_dir)
                self.__pack_arts(arts_source, pack_dir, source_dir)

            # Пост-хук.
            self.post_build(source_dir, output_dir, pack_dir)

            # Финализируем ресурсы.
            ya_sdk_compat.finalize_resources(self)

    def _create_results_resource(self):
        resource = self._create_resource(
            "TestEnv Json",
            "results.json",
            resource_types.TEST_ENVIRONMENT_JSON_V2
        )
        ya_sdk_compat.create_default_file(resource.abs_path())
        return resource

    def _create_html_results_resource(self):
        resource = self._create_resource(
            "Ya make html results",
            "results.html",
            devtools_resources.YA_MAKE_HTML_RESULTS,
        )
        open(resource.abs_path(), 'w').close()
        return resource

    def _generate_fake_results(self, snippet, filename):
        ya_sdk_compat.generate_fake_results(self, snippet, filename)

    def _report_streaming_results(self, streaming_link, streaming_check_id, filename,
                                  check_type='HEAVY', stream_partition=0):
        with open(filename) as fp:
            results = json.load(fp)

        ya_sdk_compat.report_streaming_results(streaming_link, streaming_check_id, results, stream_partition)

    def __create_default_file(self, abs_path):
        ya_sdk_compat.create_default_file(abs_path)

    @staticmethod
    def __parse_arts(value):
        return ya_sdk_compat.parse_arts(value)

    @staticmethod
    def __ensure_link(path, link):
        ya_sdk_compat.ensure_link(path, link)

    @staticmethod
    def __pack_arts(arts, pack_dir, base_dir):
        ya_sdk_compat.pack_arts(arts, pack_dir, base_dir)

    def __publish_results(self, resource, filename, check_rc=True, html_results_resource=None):
        return ya_sdk_compat.publish_results(self, resource, filename, check_rc, html_results_resource)
