# -*- coding: utf-8 -*-
import logging
import os
import platform
import re
import six
import shlex
import sys
import tempfile
import uuid

from sandbox import common
from sandbox import sdk2
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.environments import Xcode
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.common.mds import compression

from sandbox.projects import resource_types
from sandbox.projects.common import binary_task
from sandbox.projects.common import context_managers
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils2
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import dependency_banner
from sandbox.projects.common.build import resource_types as build_resource_types
from sandbox.projects.common.build import ya_make_package
from sandbox.projects.common.sdk_compat import task_helper
from sandbox.projects.common.build.sdk import sdk_compat as ya_sdk_compat
import sandbox.projects.common.build.parameters as build_parameters
import sandbox.projects.common.constants as consts
from sandbox.projects.common.vcs import arc


# Common methods and constants for YaMake and YaMake2

MULTISLOT_REQUIRED_CORES = 1
MULTISLOT_REQUIRED_RAM = 8192
PACK_DIR = ya_sdk_compat.PACK_DIR

ENV_VAR_PATTERN = r'^\s*(\b[a-zA-Z_]\w*=((\'[^\']*\')|(\"[^\"]*\")|([^\s\'\"]+))(\s+|$))+$'
VAULT_PATTERN = r'\$\(vault:(?P<dst>file|value):(?P<owner>[^:]+):(?P<name>[^:]+)\)'

# build tree + resources quota + arcadia and partial arcadia_tests_data cache
YA_MAKE_DISK_SPACE = 10 * 1024 + 10 * 1024 + (10 + 10) * 1024
YA_MAKE_CLIENT_TAGS = (
    (ctc.Tag.GENERIC | ctc.Tag.Group.OSX | ctc.Tag.PORTOD) &
    (
        ~ctc.Tag.Group.LINUX | ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL |
        ctc.Tag.LINUX_BIONIC | ctc.Tag.LINUX_FOCAL
    )
)

re_target_android = re.compile(
    r'--target-platform=\w+-android-\w+'
)


def get_resource_ttl_parameter(name, description, default_value="14"):

    class ResultResourceTtlParameter(sdk2.parameters.String):
        name = "resource_ttl"
        description = "Resource TTL"
        default_value = '14'

        @classmethod
        def cast(cls, value):
            if not value:
                return cls.default_value

            value = value.lower()

            if value != 'inf' and not value.isdigit():
                raise ValueError("Resource ttl must be 'inf' or integer")

            return value

    ResultResourceTtlParameter.name = name
    ResultResourceTtlParameter.description = description
    ResultResourceTtlParameter.default_value = default_value

    return ResultResourceTtlParameter()


class ResultResourceTypeParameter(sdk2.parameters.String):
    name = "result_rt"
    description = "Result resource type"

    @classmethod
    def get_custom_parameters(cls):
        return {"values": sorted(((rt.name, rt.name) for rt in resource_types.AbstractResource))}

    default_value = resource_types.ARCADIA_PROJECT.name

    @classmethod
    def cast(cls, value):
        if not value:
            return None

        for rt in resource_types.AbstractResource:
            if value == rt.name:
                break
        else:
            raise ValueError("Resource type %s does not exist" % value)

        return value


class ResultResourcesTypesParameter(sdk2.parameters.String):
    name = "result_resources_types"
    description = "Multiple result resources types, semicolon-separated"


class ArcadiaProjectBuildParameters(sdk2.Task.Parameters):
    """
    Параметры сборки, специфичные для проектов Аркадии.
    """
    build_output_html_ttl = build_parameters.BuildOutputHtmlTtl()
    build_output_ttl = build_parameters.BuildOutputTtl()
    do_not_remove_resources = build_parameters.DoNotRemoveResources2()
    build_system = build_parameters.BuildSystem()
    distbuild_pool = build_parameters.DistbuildPool()
    build_type = build_parameters.BuildType()
    check_return_code = build_parameters.CheckReturnCode()
    failed_tests_cause_error = build_parameters.FailedTestsCauseError()
    checkout = build_parameters.CheckoutParameter()
    checkout_mode = build_parameters.CheckoutModeParameter()
    force_disable_arcadia_tests_data = build_parameters.ForceDisableArcadiaTestsData()
    clear_build = build_parameters.ClearBuild()
    definition_flags = build_parameters.DefinitionFlags()
    env_vars = build_parameters.EnvironmentVarsParam()
    env_vars_secret = build_parameters.EnvironmentVarsParamSecret()
    force_build_depends = build_parameters.ForceBuildDepends()
    force_vcs_info_update = build_parameters.ForceVCSInfoUpdate()
    ignore_recurses = build_parameters.IgnoreRecurses()
    keep_on = build_parameters.KeepOn()
    lto = build_parameters.LTO()
    make_context_on_distbuild = build_parameters.MakeContextOnDistbuild()
    musl = build_parameters.Musl()
    use_prebuilt_tools = build_parameters.UsePrebuiltTools()
    pgo_add = build_parameters.PGOAdd()
    pgo_merge_timeout = build_parameters.PGOMergeTimeout()
    pgo_use = build_parameters.PGOUse()
    strip_binaries = build_parameters.StripBinaries()
    target_platform = build_parameters.TargetPlatform()
    target_platform_flags = build_parameters.TargetPlatformFlags()
    host_platform_flags = build_parameters.HostPlatformFlags()
    thinlto = build_parameters.ThinLTO()
    use_dev_version = build_parameters.UseDevVersion()
    use_system_python = build_parameters.UseSystemPythonParameter()
    use_ya_bloat = build_parameters.UseYaBloat()
    vault_key_name = build_parameters.VaultKeyName()
    vault_owner = build_parameters.VaultOwner()
    ya_timeout = build_parameters.YaTimeout()
    ya_add_result = build_parameters.AddResult(label=build_parameters.AddResult.description)

    check_deps_parameters = build_parameters.CheckDepsParameters()
    java_specific_parameters = build_parameters.JavaSpecificParameters()
    test_specific_parameters = build_parameters.TestSpecificParameters()
    yt_store_parameters = build_parameters.YtStoreParameters()
    yt_store_exclusive = build_parameters.YtStoreExclusive()
    debug_specific_parameters = build_parameters.DebugSpecificParameters()


class YaMakeProjectParameters(sdk2.Task.Parameters):
    """
        Параметры, предназначенные для настройки дефолтной реализации YA_MAKE в рамках этой задачи.
        В задачах-наследниках, собирающих конкретные проекты, вместо использования данной группы параметров
        рекомендуется переопределить функции get_targets, get_arts и get_arts_source и так далее.
    """
    with sdk2.parameters.Group("Project configuration parameters:") as ya_make_project_params:
        arts = sdk2.parameters.String("Build artifacts (semicolon separated pairs path[=destdir])")
        arts_source = sdk2.parameters.String("Source artifacts (semicolon separated pairs path[=destdir])")
        result_rd = sdk2.parameters.String("Result resource description", default="Arcadia Project")
        result_rt = ResultResourceTypeParameter()
        result_resources_types = ResultResourcesTypesParameter()
        result_ttl = get_resource_ttl_parameter("result_ttl", "Result resource ttl")
        result_released_ttl = get_resource_ttl_parameter(
            "result_released_ttl",
            "Result resource ttl after release",
            default_value="inf",
        )
        result_attrs = sdk2.parameters.Dict("Result resource attributes")
        result_single_file = sdk2.parameters.Bool("Result is a single file", default=False)
        targets = sdk2.parameters.String("Targets (semicolon separated)", required=True)
        xcode_version = sdk2.parameters.String("Required version of Xcode")


class ContainerParameter(sdk2.parameters.Container):
    name = "sandbox_container"
    description = 'Container the task should execute in'
    default_value = None
    required = False


class SandboxTaskParameters(sdk2.Task.Parameters):
    # This parameter will be available in "Advanced" section of the task parameters
    sandbox_container = ContainerParameter()

    with sdk2.parameters.Group("Sandbox parameters:"):
        sandbox_tags = sdk2.parameters.String("Sandbox task tags. Doc: https://nda.ya.ru/3RhRAc", default="GENERIC")
        force_sandbox_tags = sdk2.parameters.Bool("Whether client tags should be treated as given", default=False, required=False)
        privileged = sdk2.parameters.Bool("Should run tasks under root privileges", default=False, required=False)
        pack_resource = sdk2.parameters.Bool("Pack created resource to tar archive", default=False)


class StorageIntegrationParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Storage parameters:"):
        report_to_ci = sdk2.parameters.Bool('report_to_ci')
        ci_endpoints = sdk2.parameters.String('ci_endpoints', default='')
        ci_check_id = sdk2.parameters.String('ci_check_id')
        ci_iteration_number = sdk2.parameters.Integer('ci_iteration_number', default=1)
        ci_task_id = sdk2.parameters.String('ci_task_id')
        ci_type = sdk2.parameters.String('ci_type')


class YaMakeParameters(sdk2.Task.Parameters):
    arcadia_parameters = build_parameters.ArcadiaParameters()

    use_aapi_fuse = build_parameters.UseArcadiaApiFuse()
    use_arc_instead_of_aapi = build_parameters.UseArcInsteadOfArcadiaApi()
    aapi_fallback = build_parameters.AllowArcadiaApiFallback()
    yav_token = build_parameters.YavToken()
    minimize_arc_mount_path = build_parameters.MinimizeArcMountPath()

    arcadia_project_build_parameters = ArcadiaProjectBuildParameters()
    sandbox_task_parameters = SandboxTaskParameters()
    ya_make_project_parameters = YaMakeProjectParameters()
    ya_make_extra_parameters = sdk2.parameters.List("Extra parameters for yamake", default=None)

    storage_integration_parameters = StorageIntegrationParameters()


class YaMake2(ya_make_package.BinaryMixin, binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Сборка произвольного проекта или набора проектов в Аркадии.

    Задача переходит в `FAILURE`, если какой-либо из этапов сборки провалился.
    Опционально запускается тестирование.

    Задача легко наследуется и конфигурируется перегрузкой методов
    `pre_build`, `get_targets`, `get_arts`, `get_arts_source`, `get_resources` и `get_resources_attrs`.

    Для получения определенных объектов сборки в качестве отдельного ресурса необходимо
    в Build artifacts указать пути к файлам, которые будут получены отдельным ресурсом.

    Для типовых кастомизаций (собирать определённый набор целей или исходных файлов/каталогов)
    рекомендуется использование YaMakeTemplate:
    https://docs.yandex-team.ru/ya-make/usage/sandbox/ya_make#yamaketemplate
    """

    build_output_html_ttl = 4
    build_output_ttl = 4

    class Requirements(sdk2.Task.Requirements):
        disk_space = YA_MAKE_DISK_SPACE
        client_tags = YA_MAKE_CLIENT_TAGS & ctc.Tag.GENERIC

    class Parameters(YaMakeParameters):
        ext_params = binary_task.binary_release_parameters(custom=True)

    class Context(sdk2.Context):
        create_results_resource = True

    def abs_path(self, *args):
        settings = common.config.Registry()
        local_path = os.path.join(*(ctt.relpath(self.id) + list(six.moves.map(str, args))))
        return os.path.join(settings.client.tasks.data_dir, local_path)

    def pre_execute(self):
        checkout_arcadia_from_url = self.Parameters.checkout_arcadia_from_url
        try:
            checkout_arcadia_from_url = sdk2.svn.Arcadia.freeze_url_revision(checkout_arcadia_from_url)
        except sdk2.svn.SvnError as e:
            eh.fail('Arcadia URL {0} does not exist. Error: {1}'.format(checkout_arcadia_from_url, e))

        parsed_url = sdk2.svn.Arcadia.parse_url(checkout_arcadia_from_url)
        self.Context.ap_arcadia_revision = parsed_url.revision
        self.Context.ap_arcadia_trunk = parsed_url.trunk
        self.Context.ap_arcadia_branch = parsed_url.branch
        self.Context.ap_arcadia_tag = parsed_url.tag
        self.Context.checkout_arcadia_from_url = checkout_arcadia_from_url
        xcode_version = getattr(self.Parameters, consts.XCODE_VERSION, None)
        logging.debug('Xcode version: {}'.format(xcode_version))
        if xcode_version:
            Xcode.prepare_xcode(xcode_version)

    def get_base_resource_attrs(self):
        return {
            'arcadia_revision': self.Context.ap_arcadia_revision,
            'arcadia_trunk': self.Context.ap_arcadia_trunk,
            'arcadia_branch': self.Context.ap_arcadia_branch,
            'arcadia_tag': self.Context.ap_arcadia_tag,
        }

    def pre_build(self, source_dir):
        """
            Действия, производимые перед сборкой.
            Функция может быть перегружена в наследниках.
            :param source_dir: директория с исходниками 'arcadia'
        """
        pass

    def post_build(self, source_dir, output_dir, pack_dir):
        """
            Действия, производимые после сборки и тестирования.
            Функция может быть перегружена в наследниках.
            :param source_dir: директория с исходниками 'arcadia'
            :param output_dir: директория с результатами
            :param pack_dir: директория, куда были сложены артефакты сборки
        """
        ya_sdk_compat.post_build(self, source_dir, output_dir)

        logs_ttl = getattr(self.Parameters, consts.BUILD_LOGS_TTL, None)
        if logs_ttl:
            try:
                common.rest.Client().resource[self._log_resource.id].attribute.create({"name": "ttl", "value": logs_ttl})
            except common.rest.Client().HTTPError as ex:
                if ex.status == six.moves.http_client.CONFLICT:
                    common.rest.Client().resource[self._log_resource.id].attribute['ttl'].update(logs_ttl)
                else:
                    raise

        # TODO move to the ya_sdk_compat module (see copy-paste in YaMake task)
        if (
            getattr(self.Parameters, consts.TESTS_REQUESTED, False)
            and getattr(self.Parameters, consts.CANONIZE_TESTS, False)
        ):
            _, mode = ya_sdk_compat.check_parameters_compatibility(
                self, getattr(self.Parameters, consts.MAKE_CONTEXT_ON_DISTBUILD, False)
            )
            resource = self.create_resource(
                "Zipatch for Canonization",
                "canonization.zipatch",
                build_resource_types.ZIPATCH_OUTPUT
            )
            if mode in (ya_sdk_compat.GET_ARCADIA_USE_ARC_VCS, ya_sdk_compat.GET_ARCADIA_USE_ARC_FUSE):
                arc_repo = arc.Arc()
                status = arc_repo.status(source_dir, True)
                logging.info("arc status --json output: {}".format(status))
                if status['status']:
                    branch = "canonize-{}".format(str(uuid.uuid4()))
                    arc_repo.checkout(source_dir, branch, create_branch=True)
                    arc_repo.commit(source_dir, "canondata")
                    info = arc_repo.info(source_dir)
                    commit_hash = info["hash"]
                    logging.info("commit hash: {}".format(commit_hash))
                    arc_repo.push(source_dir, refspecs=[commit_hash])

                    arc_repo.create_zipatch(source_dir, commit_hash, str(resource.path.resolve()), no_copy=True)

                    if getattr(self.Parameters, consts.CANONIZE_PR, False):
                        logging.info("creating PR with canonization data")
                        pr_result = arc_repo.pr_create(
                            source_dir,
                            message="Canonization from sandbox task {}".format(self.id),
                            publish=True,
                            auto=True,
                            no_commits=True,
                            as_dict=True,
                        )
                        logging.info("arc pr create result: %s", pr_result)
                        self.set_info("created PR with changes in canondata: {}".format(pr_result["url"]))
                    else:
                        self.set_info("you can apply changes in canondata to your branch by calling\n"
                                      "arc cherry-pick {}".format(commit_hash))
            else:
                logging.info("Make zipatch for test canonization")
                res = sdk.do_create_zipatch(source_dir, resource.path.resolve())
                if res == 0:
                    self.set_info("Zipatch created, can be loaded from {}".format(resource.proxy_url))

                    base_rev = getattr(self.Parameters, "ap_arcadia_revision", False)
                    if base_rev:
                        self.set_info("You can apply changes to your working copy by calling\n   "
                                      "ya unshelve -u={} -r {}".format(resource.proxy_url, base_rev))
                else:
                    self.set_info("Failed to create zipatch, RetCode: {}".format(res))

    def get_arcadia_src_dir(self, copy_trunk=False):
        return sdk2.svn.Arcadia.get_arcadia_src_dir(
            self.Context.checkout_arcadia_from_url,
            copy_trunk=copy_trunk,
        )

    def deref(self, s):
        def deref_vault(match):
            secret = sdk2.Vault.data(match.group('owner'), match.group('name'))

            if match.group('dst') == 'file':
                deref_path = tempfile.NamedTemporaryFile().name
                fu.write_file(deref_path, secret)
                return deref_path

            return secret

        s = re.sub(VAULT_PATTERN, deref_vault, s)

        return s

    def get_env_vars(self):
        env_vars = self.Parameters.env_vars
        if env_vars and not re.match(ENV_VAR_PATTERN, env_vars):
            eh.check_failed("Incorrect 'Environment variables' parameter '{}'".format(env_vars))

        env_vars = {k: self.deref(v) for k, v in (x.split('=', 1) for x in shlex.split(env_vars))}
        env_vars_secret = task_helper.input_field(self, consts.ENV_VARS_SECRET)
        if env_vars_secret:
            env_vars.update(env_vars_secret.data())

        for key in ('GSID', 'PATH_TO_SOY_BATCH'):
            if key not in env_vars.keys() and key in os.environ.keys():
                env_vars[key] = os.getenv(key)

        return env_vars

    def create_resource(self, description, resource_path, resource_type, arch=None, attributes=None, owner=None):
        if arch is None and self.Parameters.target_platform is not None:
            arch = ctm.OSFamily.from_system_name(self.Parameters.target_platform)

        resource = sdk2.Resource[resource_type](
            task=self,
            description=description,
            path=resource_path,
            arch=arch,
        )

        return resource

    def get_resources(self):
        return ya_sdk_compat.get_resources(self)

    def get_resources_attrs(self):
        return ya_sdk_compat.get_resources_attrs(self)

    def get_check_deps(self):
        return ya_sdk_compat.get_check_deps(self)

    def get_target_platform_flags(self):
        return ya_sdk_compat.get_target_platform_flags(self)

    def get_host_platform_flags(self):
        return ya_sdk_compat.get_host_platform_flags(self)

    def get_additional_build_def_flags(self):
        additional_build_def_flags = {
            'NO_SRCLINK': 'yes',
            'SANDBOX_TASK_ID': self.id,  # probably, it's a good idea to save build task in flags by default
        }
        if self.Parameters.use_system_python:
            additional_build_def_flags['USE_ARCADIA_PYTHON'] = 'no'
        return additional_build_def_flags

    def get_all_build_def_flags(self):
        all_build_def_flags = sdk.parse_flags(self.Parameters.definition_flags)
        all_build_def_flags.update(self.get_additional_build_def_flags())
        return all_build_def_flags

    def get_build_path(self):
        return ya_sdk_compat.get_build_path(self)

    def get_local_output_dir(self):
        return ya_sdk_compat.get_local_output_dir(self)

    def get_source_dirs(self, mode, arc_oauth_token=None):
        return ya_sdk_compat.get_source_dirs(self, mode, arc_oauth_token=arc_oauth_token)

    def get_targets(self):
        all_targets = ya_sdk_compat.get_targets(self)
        return [target for target in all_targets if len(target) != 0]

    def get_arts(self):
        return ya_sdk_compat.get_arts(self)

    def get_arts_source(self):
        return ya_sdk_compat.get_arts_source(self)

    def get_dist_priority(self):
        return getattr(self.Parameters, consts.DISTBUILD_PRIORITY, None)

    def get_heater_mode(self):
        return False

    def get_binaries_for_ya_bloat(self):
        return ya_sdk_compat.get_binaries_for_ya_bloat(self)

    def check_aapi_available(self):
        return ya_sdk_compat.check_aapi_available(self)

    def prepare_output_resource(self, output_dir):
        if self.Parameters.do_not_remove_resources:
            attrs = {"ttl": "inf"}
        else:
            if self.Parameters.build_output_ttl == 0:
                attrs = {"ttl": 1}
            else:
                attrs = {"ttl": self.Parameters.build_output_ttl or self.build_output_ttl}
        if getattr(self.Parameters, consts.PACK_RESOURCE, None):
            attrs["pack_tar"] = compression.base.CompressionType.TAR

        return sdk2.Resource['BUILD_OUTPUT'](
            self,
            description='Build output',
            path=output_dir,
            **attrs
        )

    def build_output_resource(self, build_output_res):
        if build_output_res:
            sdk2.ResourceData(build_output_res).ready()

    # Derived class may delay this exception
    def on_exception(self, exception):
        logging.debug('Raising exception `on_exception`: %s', exception)
        raise exception

    def on_enqueue(self):
        try:
            require_multislot = ya_sdk_compat.is_required_multislot(self.Parameters.sandbox_tags)
            if require_multislot:
                if not self.Requirements.cores:
                    self.Requirements.cores = MULTISLOT_REQUIRED_CORES
                if not self.Requirements.ram or self.Requirements.ram < MULTISLOT_REQUIRED_RAM:
                    self.Requirements.ram = MULTISLOT_REQUIRED_RAM
        except Exception:
            logging.exception("YA_MAKE.on_enqueue():")

        if ya_sdk_compat.is_build_bundle_task(self):
            return

        self.Context.ap_packs = {}
        resources = self.get_resources()
        ya_sdk_compat.process_resources(self, resources)

        build_system = self.Parameters.build_system
        if self.Parameters.ram_drive_size and build_system != consts.DISTBUILD_BUILD_SYSTEM:
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                self.Parameters.ram_drive_size,
                None
            )

        if self.get_check_deps():
            dependency_banner.create_resources(self)

        # We should not add any TAGS if force_sandbox_tags flag is set
        if self.Parameters.force_sandbox_tags:
            self.Requirements.client_tags = ctc.Tag.Query.cast(self.Parameters.sandbox_tags)
            return

        # We should not add any TAGS when user requests WINDOWS
        for tag_set in self.Requirements.client_tags:
            if ctc.Tag.WINDOWS in tag_set:
                return

        # We use Clang from the Android NDK for Android targets and
        # it requires at least Ubuntu 14.04 as a Linux host.
        if re_target_android.search(self.get_target_platform_flags() or ''):
            android_compatible = ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL | ctc.Tag.LINUX_BIONIC
            self.Requirements.client_tags &= android_compatible

        try:
            self.Requirements.client_tags &= ctc.Tag.Query.cast(self.Parameters.sandbox_tags)
        except Exception:  # FIXME
            pass

    @sdk2.report(title="results html")
    def build_report(self):
        if self.Context.html_results_resource_id is not ctm.NotExists:
            return (
                "<iframe src='https://proxy.sandbox.yandex-team.ru/{}' style='height:2000px;width:100%;'>".format(
                    self.Context.html_results_resource_id,
                )
            )
        elif not self.Parameters.create_html_results_resource:
            return (
                "<h3>HTML results resource wasn't requested - parameter "
                "'Create html resource from ya make results'/create_html_results_resource is disabled</h3>"
            )
        else:
            return "<h3>No html results resource was yet generated</h3>"

    def _generate_links_report(self, param, title, resource_type):
        if not getattr(self.Parameters, param, None):
            return (
                "<h3>{} report wasn't requested - parameter "
                "'Create {} report from ya make results'/{} is disabled</h3>".format(title, title, param)
            )
        links = [
            utils2.resource_redirect_url(r.id, 'index.html')
            for r in sdk2.Resource.find(task=self, type=resource_type, state=ctr.State.READY).limit(0)
        ]
        if len(links) == 0:
            return "<h3>No {} reports was generated yet</h3>".format(title)
        elif len(links) == 1:
            return "<iframe src='{}' style='height:2000px;width:100%;'>".format(links[0])
        else:
            return ' '.join([
                '<b><font color="red">{} report</font></b>: <a href="{}" target="_blank">index.html</a>'.
                    format(title, link)
                for link in links
            ])

    @sdk2.report(title="allure")
    def allure_report(self):
        return self._generate_links_report(consts.ALLURE_REPORT, 'Allure', resource_types.ALLURE_REPORT)

    @sdk2.report(title="coverage")
    def coverage_report(self):
        return self._generate_links_report(consts.COVERAGE, 'Coverage', resource_types.COVERAGE_REPORT)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        self.pre_execute()

        build_path, output_dir = self.get_build_path()
        self.Context.output_dir_path = output_dir

        local_output_dir = self.get_local_output_dir()

        logging.debug('local_output_dir: %s', local_output_dir)

        results_dir = output_dir if local_output_dir else None

        logging.debug('results_dir: %s', results_dir)

        html_results_resource = None
        if results_dir and self.Context.create_results_resource:
            results_resource = sdk2.Resource['TEST_ENVIRONMENT_JSON_V2'](self, 'results.json', 'TestEnv Json')
            results_resource = sdk2.ResourceData(results_resource)
            ya_sdk_compat.create_default_file(str(results_resource.path))
            if self.Parameters.create_html_results_resource:
                html_results_resource = sdk2.Resource['YA_MAKE_HTML_RESULTS'](self, 'results.html', 'results.html')
                self.Context.html_results_resource_id = html_results_resource.id
                html_results_resource = sdk2.ResourceData(html_results_resource)

                with open(str(html_results_resource.path), 'w') as afile:
                    afile.write("<h3>No html results resource was yet generated</h3>")
        else:
            results_resource = None

        # Запоминаем платформу сборки.
        self.Context.ap_build_platform = platform.platform()

        # Создаем директории build (для сборки) и room (для эмуляции корня arc).
        room_dir = sdk2.paths.make_folder('room')
        pack_dir = sdk2.paths.make_folder(PACK_DIR)
        source_dir = os.path.join(room_dir, 'arcadia')
        self.Context.source_dir = source_dir

        # Вычисляем дополнительные опции сборки.
        build_type = self.Parameters.build_type

        patch = self.Parameters.arcadia_patch

        build_system = self.Parameters.build_system
        make_context_on_distbuild = self.Parameters.make_context_on_distbuild

        ya_sdk_compat.check_dist_build_parameters(self, build_system, make_context_on_distbuild)
        checkout, mode = ya_sdk_compat.check_parameters_compatibility(self, make_context_on_distbuild)
        self.Context.checkout = checkout
        self.Context.mode = mode

        arc_oauth_token = ya_make_package.prepare_arc_token(self, mode)
        arcadia_ctx, arcadia_tests_data_ctx = self.get_source_dirs(mode, arc_oauth_token=arc_oauth_token)

        with arcadia_ctx as arcadia_path, arcadia_tests_data_ctx as arcadia_tests_data:
            logging.info("Arcadia source dir: %s", arcadia_path)
            logging.info("Arcadia tests data source dir: %s", arcadia_tests_data)

            # XXX For more info see https://st.yandex-team.ru/DEVTOOLSSUPPORT-13736
            enable_direct_arcadia_mode = self.get_env_vars().get("ENABLE_DEVTOOLSSUPPORT_13736") == "yes"

            if enable_direct_arcadia_mode:
                source_dir = arcadia_path
            else:
                ya_sdk_compat.ensure_link(arcadia_path, source_dir)

            # Set YA_SOURCE_ROOT env to avoid ya respawn after symlink creation
            os.environ['YA_SOURCE_ROOT'] = source_dir

            if patch:
                sdk.apply_patch(self, source_dir, patch, self.abs_path(), checkout)

            self.pre_build(source_dir)

            targets = self.get_targets()
            arts = self.get_arts()
            arts_source = self.get_arts_source()

            logging.debug('Targets are %s, arts are %s, arts source is %s', targets, arts, arts_source)

            # Проверяем осмысленность конфигурации.
            if self.Context.ap_packs and not (arts or arts_source):
                eh.check_failed('Resources are registered, but no artifacts are set.')

            self.Context.build_def_flags = self.get_all_build_def_flags()

            if targets:
                split_entries = (
                    lambda name: filter(None, [
                        e.strip() for e in (getattr(self.Parameters, name, '') or '').split(',')
                    ])
                )

                if self.Parameters.check_dependencies:
                    dependency_banner.check_dependencies(self, source_dir, [os.path.dirname(x) for x in targets])

                timeout = int(getattr(self.Parameters, consts.YA_TIMEOUT, 10800))
                use_dev_version = self.Parameters.use_dev_version
                junit_report_path = (
                    self.log_path("junit_report.xml")
                    if getattr(self.Parameters, consts.JUNIT_REPORT) else None
                )
                coverage = getattr(self.Parameters, consts.COVERAGE, False)
                coverage_prefix_filter = getattr(self.Parameters, consts.COVERAGE_PREFIX_FILTER) or None
                coverage_exclude_regexp = getattr(self.Parameters, consts.COVERAGE_EXCLUDE_REGEXP) or ''
                coverage_unified_agent = getattr(self.Parameters, consts.COVERAGE_UNIFIED_AGENT) or False
                coverage_unified_agent_sid = getattr(self.Parameters, consts.COVERAGE_UNIFIED_AGENT_SID) or ''
                coverage_unified_agent_strict = getattr(self.Parameters, consts.COVERAGE_UNIFIED_AGENT_STRICT) or False
                sanitize = getattr(self.Parameters, consts.SANITIZE, False)
                lto = getattr(self.Parameters, consts.LTO, False)
                thinlto = getattr(self.Parameters, consts.THINLTO, False)
                pgo_add = getattr(self.Parameters, consts.PGO_ADD, False)
                pgo_use = getattr(self.Parameters, consts.PGO_USE, None)
                pgo_merge_timeout = getattr(self.Parameters, consts.PGO_MERGE_TIMEOUT, 600)
                vault_owner = getattr(self.Parameters, consts.VAULT_OWNER, self.author)
                vault_key_name = getattr(self.Parameters, consts.VAULT_KEY_NAME)
                test_params = getattr(self.Parameters, consts.TEST_PARAMS_KEY) or {}
                clear_build = getattr(self.Parameters, consts.CLEAR_BUILD_KEY)
                strip_binaries = getattr(self.Parameters, consts.STRIP_BINARIES, False)
                keep_on = getattr(self.Parameters, consts.KEEP_ON)
                check_rc = getattr(self.Parameters, consts.CHECK_RETURN_CODE)
                tests_requested = getattr(self.Parameters, consts.TESTS_REQUESTED, False)
                allure_report = getattr(self.Parameters, consts.ALLURE_REPORT, False)
                allure_report_ttl = getattr(self.Parameters, consts.ALLURE_REPORT_TTL)
                test_type_filter = split_entries(consts.TEST_TYPE_FILTER)
                fuzz_opts = getattr(self.Parameters, consts.FUZZ_OPTS, [])
                # TODO add parameter to the task
                sanitize_coverage = getattr(self.Parameters, consts.SANITIZE_COVERAGE, None)
                cache_namespace = getattr(self.Parameters, consts.CACHE_NAMESPACE, None)
                sandbox_token = getattr(self.Parameters, consts.SANDBOX_TOKEN, None)
                musl = getattr(self.Parameters, consts.MUSL, False)
                java_coverage = getattr(self.Parameters, consts.JAVA_COVERAGE, False)
                test_size_filter = split_entries(consts.TEST_SIZE_FILTER)
                save_links_for_files = split_entries(consts.SAVE_LINKS_FOR_FILES)
                distbuild_pool = getattr(self.Parameters, consts.DISTBUILD_POOL)

                yt_store_params = ya_make_package.prepare_yt_store_params(self)

                force_disable_arcadia_tests_data = getattr(self.Parameters, consts.FORCE_DISABLE_ARCADIA_TESTS_DATA)

                # Выкачиваем и обновляем нужное поддерево ATD
                if (not force_disable_arcadia_tests_data
                    and tests_requested
                    and not make_context_on_distbuild
                    and arcadia_tests_data is None):
                    logging.debug(
                        'update_atd %s, %s, %s, %s, %s, %s',
                        checkout, source_dir, targets, clear_build, build_system, use_dev_version,
                    )
                    arcadia_tests_data = ya_sdk_compat.update_atd(
                        self, checkout, source_dir, targets, clear_build, build_system, use_dev_version
                    )
                    logging.debug('arcadia_tests_data = {}'.format(arcadia_tests_data))

                if pgo_use:
                    pgo_use = [sdk2.Resource[res_id] for res_id in pgo_use]
                    logging.debug('PGO profiles: %s' % pgo_use)

                build_output_res = self.prepare_output_resource(output_dir)

                ya_sdk_compat.create_default_file(os.path.join(output_dir, "default.json"))

                returncode = -1
                build_exception_str = ''
                try:
                    if vault_owner and vault_key_name:
                        ssh_manager = ssh.Key(self, vault_owner, vault_key_name)
                    else:
                        ssh_manager = context_managers.nullcontext()
                    with ssh_manager:
                        env = self.get_env_vars()
                        env['YA_DUMP_RAW_RESULTS'] = '1'
                        logging.debug('Start do_build')
                        returncode = sdk.do_build(
                            build_system,
                            source_dir,
                            targets,
                            build_resource_id=build_output_res.id if build_output_res else None,
                            build_type=build_type,
                            clear_build=clear_build,
                            def_flags=self.Context.build_def_flags,
                            results_dir=results_dir,
                            target_platform=self.Parameters.target_platform,
                            yt_store_params=yt_store_params,

                            add_result=[  # TODO: remove filter after SANDBOX-6065
                                suffix for suffix in self.Parameters.ya_add_result if suffix
                            ],
                            strip_binaries=strip_binaries,
                            test=tests_requested,
                            test_filters=getattr(self.Parameters, consts.TEST_FILTERS),
                            output_only_tests=getattr(self.Parameters, consts.OUTPUT_ONLY_TESTS, False),
                            canonize_tests=getattr(self.Parameters, consts.CANONIZE_TESTS, False),
                            junit_report_path=junit_report_path,
                            use_dev_version=use_dev_version, timeout=timeout, coverage=coverage,
                            coverage_prefix_filter=coverage_prefix_filter,
                            coverage_exclude_regexp=coverage_exclude_regexp, sanitize=sanitize,
                            coverage_unified_agent=coverage_unified_agent,
                            coverage_unified_agent_sid=coverage_unified_agent_sid,
                            coverage_unified_agent_strict=coverage_unified_agent_strict,
                            coverage_unified_agent_uids_res_id=getattr(
                                self.Parameters, consts.COVERAGE_UNIFIED_AGENT_UIDS_RESOURCE_ID, None
                            ),
                            coverage_unified_agent_uids_file_path=getattr(
                                self.Parameters, consts.COVERAGE_UNIFIED_AGENT_UIDS_FILE_PATH, None
                            ),
                            coverage_unified_agent_failed_uids_file_path=getattr(
                                self.Parameters, consts.COVERAGE_UNIFIED_AGENT_FAILED_UIDS_FILE_PATH, None
                            ),
                            lto=lto, thinlto=thinlto,
                            test_params=test_params,
                            ram_drive_path=ya_sdk_compat.get_ram_drive_path(self),
                            keep_on=keep_on, check_rc=check_rc,
                            target_platform_flags=self.get_target_platform_flags(),
                            host_platform_flags=self.get_host_platform_flags(),
                            report_tests_only=getattr(self.Parameters, consts.REPORT_TESTS_ONLY),
                            test_log_level=getattr(self.Parameters, consts.TEST_LOG_LEVEL),
                            test_tag=getattr(self.Parameters, consts.TEST_TAG),
                            allure_report=allure_report,
                            allure_report_ttl=allure_report_ttl,
                            disable_test_timeout=getattr(self.Parameters, consts.DISABLE_TEST_TIMEOUT),
                            force_build_depends=getattr(self.Parameters, consts.FORCE_BUILD_DEPENDS),
                            force_vcs_info_update=getattr(self.Parameters, consts.FORCE_VCS_INFO_UPDATE, False),
                            ignore_recurses=getattr(self.Parameters, consts.IGNORE_RECURSES),
                            patch=patch,
                            test_size_filter=test_size_filter,
                            test_threads=getattr(self.Parameters, consts.TEST_THREADS),
                            arcadia_tests_data=arcadia_tests_data,
                            cache_test_results=getattr(self.Parameters, consts.CACHE_TEST_RESULTS),
                            tests_retries=getattr(self.Parameters, consts.TESTS_RETRIES),
                            pgo_add=pgo_add, pgo_use=pgo_use, pgo_merge_timeout=pgo_merge_timeout,
                            test_type_filter=test_type_filter,
                            checkout=checkout,
                            fuzzing=getattr(self.Parameters, consts.FUZZING, None),
                            fuzz_opts=fuzz_opts,
                            sanitize_coverage=sanitize_coverage,
                            cache_namespace=cache_namespace,
                            sandbox_token=sandbox_token,
                            ssh_user=vault_owner,
                            musl=musl,
                            no_src_changes=getattr(self.Parameters, consts.NO_SRC_CHANGES, True),
                            resource_owner=getattr(self.Parameters, consts.RESOURCE_OWNER, None),
                            streaming_link=getattr(self.Parameters, consts.STREAMING_REPORT_URL, None),
                            streaming_check_id=getattr(self.Parameters, consts.STREAMING_REPORT_ID, None),
                            collect_test_cores=getattr(self.Parameters, consts.COLLECT_TEST_CORES, True),
                            sonar=getattr(self.Parameters, consts.SONAR, False),
                            sonar_options=getattr(self.Parameters, consts.SONAR_OPTIONS, None),
                            sonar_project_filter=getattr(self.Parameters, consts.SONAR_PROJECT_FILTER, None),
                            sonar_default_project_filter=getattr(
                                self.Parameters, consts.SONAR_DEFAULT_PROJECT_FILTER, False
                            ),
                            java_coverage=java_coverage,
                            sandbox_uploaded_resource_ttl=getattr(
                                self.Parameters, consts.SANDBOX_UPLOADED_RESOURCE_TTL, None
                            ),
                            multiplex_ssh=getattr(self.Parameters, consts.MULTIPLEX_SSH, False),
                            build_execution_time=getattr(self.Parameters, consts.BUILD_EXECUTION_TIME, None),
                            coverage_yt_token_path=getattr(self.Parameters, consts.COVERAGE_YT_TOKEN_PATH, None),
                            cpp_coverage_type=getattr(self.Parameters, consts.CPP_COVERAGE_TYPE, None),
                            python_coverage=getattr(self.Parameters, consts.PYTHON_COVERAGE, False),
                            go_coverage=getattr(self.Parameters, consts.GO_COVERAGE, False),
                            upload_coverage=getattr(self.Parameters, consts.UPLOAD_COVERAGE, False),
                            merge_coverage=getattr(self.Parameters, consts.MERGE_COVERAGE, False),
                            graph_timestamp=getattr(self.Parameters, consts.GRAPH_TIMESTAMP, None),
                            download_artifacts=getattr(self.Parameters, consts.DOWNLOAD_ARTIFACTS_FROM_DISTBUILD, True),
                            drop_graph_result_before_tests=getattr(
                                self.Parameters, consts.DROP_GRAPH_RESULT_BEFORE_TESTS, False
                            ),
                            save_links_for_files=save_links_for_files,
                            javac_options=getattr(self.Parameters, consts.JAVAC_OPTIONS, None),
                            jvm_args=getattr(self.Parameters, consts.JVM_ARGS_KEY, None),
                            strip_skipped_test_deps=getattr(self.Parameters, consts.STRIP_SKIPPED_TEST_DEPS, False),
                            dist_priority=self.get_dist_priority(),
                            coordinators_filter=getattr(self.Parameters, consts.COORDINATORS_FILTER, None),
                            make_context_on_distbuild=make_context_on_distbuild,
                            separate_result_dirs=getattr(self.Parameters, consts.SEPARATE_RESULT_DIRS, False),
                            fast_clang_coverage_merge=getattr(self.Parameters, consts.FAST_CLANG_COVERAGE_MERGE, False),
                            new_dist_mode=getattr(self.Parameters, consts.NEW_DIST_MODE, False),
                            skip_test_console_report=getattr(self.Parameters, consts.SKIP_TEST_CONSOLE_REPORT, False),
                            build_output_html_ttl=getattr(
                                self.Parameters, consts.BUILD_OUTPUT_HTML_TTL
                            ) or self.build_output_html_ttl,
                            keep_alive_all_streams=getattr(self.Parameters, consts.KEEP_ALIVE_ALL_STREAMS, False),
                            failed_tests_cause_error=getattr(self.Parameters, consts.FAILED_TESTS_CAUSE_ERROR, True),
                            trace_ya_output=getattr(self.Parameters, consts.TRACE_YA_OUTPUT, False),
                            env=env,
                            heater_mode=self.get_heater_mode(),
                            dir_outputs=getattr(self.Parameters, consts.DIR_OUTPUTS, False),
                            use_prebuilt_tools=getattr(self.Parameters, consts.USE_PREBUILT_TOOLS, None),
                            ya_make_extra_parameters=getattr(self.Parameters, consts.YA_MAKE_EXTRA_PARAMETERS, None),
                            distbuild_pool=distbuild_pool,
                            run_tagged_tests_on_yt=getattr(self.Parameters, consts.RUN_TAGGED_TESTS_ON_YT, False),
                        )
                        self.Context.build_returncode = returncode
                except (common.errors.TemporaryError, errors.SandboxSubprocessTimeoutError) as err:
                    logging.exception("TemporaryError or SandboxSubprocessTimeoutError: %s", err)
                    self.on_exception(err)
                except common.errors.SandboxException as e:
                    build_exception_str = str(e) or str(type(e))
                    logging.exception("Exception: %s", e)
                    self.set_info(
                        "`ya make` command failed: {}. "
                        "See colored build errors in html format.".format(build_exception_str)
                    )
                    raise
                except Exception as e:
                    build_exception_str = str(e) or str(type(e))
                    logging.exception("Exception: %s", e)

                    # Make failures explicitly visible, at least in task info.
                    # Please do not remove this failure report. Consult with mvel@ if in doubt.
                    # See RMDEV-2339 for details.
                    self.set_info(
                        "`ya make` command failed: {}. "
                        "See colored build errors in html format.".format(build_exception_str)
                    )

                    if check_rc:
                        self.on_exception(e)  # re-raise by default

                finally:
                    # First of all, report results and handle possible build errors obtained
                    try:
                        ya_sdk_compat.publish_results(
                            self,
                            results_resource,
                            os.path.join(output_dir, 'results.json'),
                            check_rc,
                            html_results_resource
                        )
                    except Exception as e:
                        logging.exception("Exception while publishing results: %s", e)
                        raise

                    if check_rc and sys.exc_info()[0] is not None:
                        # re-raise error in case of non-empty build exception context
                        logging.info("Re-raising exception: %s", sys.exc_info())
                        self.on_exception(sys.exc_info()[1])  # re-raise by default

                    try:
                        from sandbox.sandboxsdk import process
                        p = process.run_process(
                            ["chmod", "--recursive", "755", output_dir], log_prefix='chmod_output_dir'
                        )
                        p.communicate()
                    except Exception as e:
                        logging.error("Error while chmod: %s", e)

                    # Pack artifacts before marking resource as ready
                    try:
                        logging.info('pack arts: arts %s, pack_dir %s, output_dir %s', arts, pack_dir, output_dir)
                        ya_sdk_compat.pack_arts(arts, pack_dir, output_dir)
                        ya_sdk_compat.pack_arts(arts_source, pack_dir, source_dir)
                        self.build_output_resource(build_output_res)
                    except common.errors.TaskFailure as err:
                        self.on_exception(err)  # re-raise by default
            else:
                ya_sdk_compat.pack_arts(arts, pack_dir, output_dir)
                ya_sdk_compat.pack_arts(arts_source, pack_dir, source_dir)

            self.post_build(source_dir, output_dir, pack_dir)

            ya_sdk_compat.finalize_resources(self)

            if self.Context.exceptions:
                eh.check_failed('FORCE_FAILURE: Delayed exceptions found, see logs above')

    def on_release(self, additional_parameters):
        super(YaMake2, self).on_release(additional_parameters)
        self.mark_released_resources(
            additional_parameters["release_status"],
            ttl=self.Parameters.result_released_ttl,
        )
