# -*- coding: utf-8 -*-

import json
import logging
import os
import shutil
import itertools
import six
import tempfile
import glob
import platform
import base64
from contextlib import contextmanager

from six.moves.urllib import parse as urlparse

from sandbox import common, sdk2
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox import sandboxsdk
from sandbox.sandboxsdk.environments import Xcode
from sandbox.sandboxsdk.channel import channel

import sandbox.projects.common.constants as consts
import sandbox.projects.common.nanny.nanny as nanny
from sandbox.projects.common.nanny import const as nannyconsts
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import ArcadiaTask
from sandbox.projects.common.build import parameters
from sandbox.projects.common.build.sdk import sdk_compat as ya_sdk_compat
from sandbox.projects.common.build.ya_package_config import consts as ya_package_consts
from sandbox.projects.common.vcs import aapi
from sandbox.projects.common import context_managers
from sandbox.projects.common import gnupg
from sandbox.projects.common import debpkg
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import string
from sandbox.projects import resource_types

logger = logging.getLogger(__name__)

TARBALL = 'tarball'
DEBIAN = 'debian'
DOCKER = 'docker'
RPM = 'rpm'
WHEEL = 'wheel'
AAR = 'aar'
NPM = 'npm'

DEBUG = 'debug'
RELEASE = 'release'

PACKAGES_FILE = 'packages.json'
# this file is an output of 'ya package' run

PACKAGE_RESOURCES = '_package_resources'
PUBLISH_TO_LIST = '_publish_to_list'

OUTPUT_RESOURCES = "output_resources"


class PackagesBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Packages'


class PackagesParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'packages'
    description = 'Package paths, related to arcadia \';\'-separated'
    required = False
    multiline = True


class ForceDuploadParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'force_dupload'
    description = 'Force dupload'
    required = False
    default_value = False


class DuploadMaxAttemptsParameter(sandboxsdk.parameters.SandboxIntegerParameter):
    name = 'dupload_max_attempts'
    description = 'How many times try to run dupload if it fails'
    required = False
    default_value = 1


class DebianArchParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'debian_arch'
    description = 'Debian arch (--debian-arch)'
    required = False
    default_value = None


class DebianCompressionParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'debian_compression_level'
    description = 'Debian compression (the lower compression, the  faster package build)'
    required = False
    default_value = None
    choices = [
        ('Default', ''),
        ('No', '0'),
        ('Low', '3'),
        ('Medium', '6'),
        ('High', '9'),
    ]


class DebianCompressionTypeParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'debian_compression_type'
    description = 'Debian compression type'
    required = False
    default_value = None
    choices = [
        ('Default', ''),
        ('gzip', 'gzip'),
        ('xz', 'xz'),
        ('bzip2', 'bzip2'),
        ('lzma', 'lzma'),
        ('none', 'none'),
    ]


class DontStoreDebianPackageParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'debian_dont_store_package'
    description = "Don't save debian packages in resources"
    required = False
    default_value = False


class BuildDebianScriptsParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'build_debian_scripts'
    description = 'Build debian scripts'
    required = False
    default_value = False


class SloppyDebianParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'sloppy_debian'
    description = 'Fast debian packages build'
    required = False
    default_value = False
    sub_fields = {
        "false": [DebianCompressionParameter.name],
    }


class DebianDistributionParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'debian_distribution'
    description = 'Debian distribution'
    required = False
    default_value = 'unstable'


class DockerBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Docker'


class DockerImageRepositoryParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_image_repository'
    description = 'Image repository'
    required = False
    default_value = ""


class DockerSaveImageParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'docker_save_image'
    description = 'Save docker image in resource'
    required = False
    default_value = False


class DockerPushImageParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'docker_push_image'
    description = 'Push docker image'
    required = False
    default_value = False


class DockerRegistryParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_registry'
    description = 'Docker registry'
    required = False
    default_value = "registry.yandex.net"


class DockerUserParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_user'
    description = 'Docker user'
    required = False
    default_value = ""


class DockerTokenVaultName(sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_token_vault_name'
    description = 'Docker token vault name'
    required = False
    default_value = ""


class DockerBuildNetwork(sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_build_network'
    description = 'Docker build network'
    required = False
    default_value = "host"


class DockerBuildArg(sandboxsdk.parameters.DictRepeater, sandboxsdk.parameters.SandboxStringParameter):
    name = 'docker_build_arg'
    description = 'Docker build arg'
    default_value = None


class RawPackageParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'raw_package'
    description = 'Raw package (used with "tar" package type to get package content without tarring)'
    required = False
    default_value = False


class HostPlatformParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'host_platform'
    description = 'Host platform'
    required = False
    default_value = ''


class TargetPlatformParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'target_platform'
    description = 'Target platform'
    required = False
    default_value = ''


class FullStripBinariesParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'full_strip_binaries'
    description = 'Strip all (like strip without parameters)'
    required = False
    default_value = False


class StripBinariesParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'strip_binaries'
    description = 'Strip debug information (i.e. strip -g)'
    required = False
    default_value = False
    sub_fields = {
        'true': [
            FullStripBinariesParameter.name,
        ]
    }


class CreateDebugPackagesParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'create_debug_packages'
    description = 'Create separate packages with debug info'
    required = False
    default_value = False


class CompressPackageArchiveParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'compress_package_archive'
    description = 'Compress package archive'
    required = False
    default_value = True


class OverwriteReadOnlyFilesParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'overwrite_read_only_files'
    description = 'Overwrite read-only files in package'
    required = False
    default_value = False


class EnsurePackagePublishedParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'ensure_package_published'
    description = 'fail ya package if publishing was required but failed'
    required = False
    default_value = False


class SemiClearBuildParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'semi_clear_build'
    description = 'Use temporary build directory (can be used only with clear build)'
    required = False
    default_value = False


class ClearBuildParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'clear_build'
    description = 'Clear build'
    required = False
    default_value = False
    sub_fields = {
        'true': [
            SemiClearBuildParameter.name,
        ]
    }


class ArchitectureAllParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'architecture_all'
    description = 'Architecture: all'
    required = False
    default_value = False


class TestsBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Tests'


class RunMediumTestsParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'run_medium_tests'
    description = 'Run medium tests'
    required = False
    default_value = False


class RunLongTestsParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'run_long_tests'
    description = 'Run long tests'
    required = False
    default_value = False


class IgnoreFailTestsParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'ignore_fail_tests'
    description = 'Ignore fail tests'
    required = False
    default_value = False


class RunTestsParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'run_tests'
    description = 'Run tests after build'
    required = True
    default_value = False
    sub_fields = {
        'true': [
            RunMediumTestsParameter.name,
            RunLongTestsParameter.name,
            IgnoreFailTestsParameter.name
        ]
    }


class ResourcesBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Resources'


class ResourceTypeParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'resource_type'
    description = 'Created package resource type, or \';\'-separated list'
    required = True
    default_value = resource_types.YA_PACKAGE.name


class ResourceIdParameter(sandboxsdk.parameters.SandboxStringParameter):
    """If specified, task will use existing resources instead of creating new ones
    """
    name = 'resource_id'
    description = 'Previously created package resource id, or \';\'-separated list.'
    required = False
    default_value = ''


class SaveBuildOutputParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'save_build_output'
    description = 'Save build output in a separate resource'
    required = False
    default_value = True


class PublishingBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Publishing'


class ChangelogParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'changelog'
    description = 'Changelog (message text or path to changelog file arcadia relative)'
    required = False
    default_value = ''
    multiline = True


class KeyUserParameter(sandboxsdk.parameters.SandboxSelectParameter):
    name = 'key_user'
    description = 'Name of the user to sign the package'
    required = False
    choices = [(v, k) for k, v in six.iteritems(ya_package_consts.VAULT_OWNERS)]
    default_value = None


class PublishToParameter(sandboxsdk.parameters.SandboxSelectParameter):
    name = 'publish_to'
    description = 'Publish to'
    required = False
    choices = [(item, item) for item in sorted(set(ya_package_consts.DUPLOAD_CONF.keys()))]
    default_value = None


class MultiplePublishToParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'multiple_publish_to'
    description = 'Publish to'
    required = False
    default_value = ''


class MultiplePublishMappingParameter(sandboxsdk.parameters.DictRepeater, sandboxsdk.parameters.SandboxStringParameter):
    name = 'publish_to_mapping'
    description = "Package file -> repos ';' separated (has higher priority if filled)"
    default_value = None


class MultiplePublishParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'multiple_publish'
    description = 'Multiple publish'
    required = False
    default_value = False
    sub_fields = {
        'false': [
            PublishToParameter.name
        ],
        'true': [
            MultiplePublishToParameter.name
        ]
    }


class ArtifactoryUploadParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'artifactory'
    description = 'Upload package to artifactory'
    required = False
    default_value = False


class PublishPackageParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'publish_package'
    description = 'Publish package'
    required = True
    default_value = True


class WheelUploadRepo(sandboxsdk.parameters.SandboxStringParameter):
    name = 'wheel_upload_repo'
    description = 'Publish to'
    required = False
    default_value = "https://pypi.yandex-team.ru/simple"


class WheelPython3Parameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'wheel_python3'
    description = 'build python 3 package'
    required = False
    default_value = False


class WheelAccessKeyTokenVaultName(sandboxsdk.parameters.SandboxStringParameter):
    name = 'wheel_access_key_token'
    description = 'Wheel access key token vault name'
    required = False
    default_value = ""


class WheelSecretKeyTokenVaultName(sandboxsdk.parameters.SandboxStringParameter):
    name = 'wheel_secret_key_token'
    description = 'Wheel secret key token vault name'
    required = False
    default_value = ""


class NpmRegistryParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'npm_registry'
    description = 'Publish to'
    required = False
    default_value = "https://npm.yandex-team.ru"


class NpmLoginParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'npm_login'
    description = 'npm user login'
    required = False
    default_value = ""


class NpmPasswordVaultName(sandboxsdk.parameters.SandboxStringParameter):
    name = 'npm_password_vault_name'
    description = 'npm password vault name'
    required = False
    default_value = ""


class AdvancedBlock(sandboxsdk.parameters.SandboxInfoParameter):
    description = 'Advanced'


class UseYaDevParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'use_ya_dev'
    description = 'Use ya-dev to build'
    required = False
    default_value = False


class BeVerboseParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'be_verbose'
    description = 'Verbose'
    required = False
    default_value = False


class UseNewFormatParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'use_new_format'
    description = 'Use new ya package json format'
    required = False
    default_value = False


class CustomVersionParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'custom_version'
    description = 'Custom version'
    required = False
    default_value = None


class PackageResourceDescriptionParameter(
    sandboxsdk.parameters.DictRepeater, sandboxsdk.parameters.SandboxStringParameter
):
    name = 'package_resource_description'
    description = "Package file -> created package description"
    required = False
    default_value = None


class PackageResourceAttrsParameter(sandboxsdk.parameters.DictRepeater, sandboxsdk.parameters.SandboxStringParameter):
    name = 'package_resource_attrs'
    description = "Package resource attributes"
    default_value = None


class ContainerParameter(sandboxsdk.parameters.Container):
    name = consts.SANDBOX_CONTAINER
    description = 'Container the task should execute in'
    default_value = None
    required = False


class AdhocPackagesParameter(sandboxsdk.parameters.ListRepeater, sandboxsdk.parameters.SandboxStringParameter):
    name = 'adhoc_packages'
    description = "Adhoc packages (pure json)"
    required = False
    default_value = None
    multiline = True


class CreateBuildOutputResourceParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'create_build_output_resource'
    description = 'Create BUILD_OUTPUT resource'
    required = False
    default_value = False


class XcodeVersionParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = consts.XCODE_VERSION
    description = 'Required version of Xcode'
    required = False
    default_value = None


class PackageTypeParameter(sandboxsdk.parameters.SandboxSelectParameter):
    name = 'package_type'
    description = 'Package type: debian or tarball'
    required = True
    choices = [
        (AAR, AAR),
        (DEBIAN, DEBIAN),
        (TARBALL, TARBALL),
        (DOCKER, DOCKER),
        (RPM, RPM),
        (WHEEL, WHEEL),
        (NPM, NPM)
    ]
    default_value = DEBIAN
    sub_fields = {
        DEBIAN: [
            SloppyDebianParameter.name,
            DebianCompressionParameter.name,
            DebianCompressionTypeParameter.name,
            ForceDuploadParameter.name,
            DuploadMaxAttemptsParameter.name,
            BuildDebianScriptsParameter.name,
            DontStoreDebianPackageParameter.name,
            DebianDistributionParameter.name,
            DebianArchParameter.name,

            PublishingBlock.name,
            PublishPackageParameter.name,
            MultiplePublishParameter.name,
            MultiplePublishMappingParameter.name,
            MultiplePublishToParameter.name,
            PublishToParameter.name,
        ],
        DOCKER: [
            DockerBlock.name,
            DockerImageRepositoryParameter.name,
            DockerSaveImageParameter.name,
            DockerPushImageParameter.name,
            DockerUserParameter.name,
            DockerTokenVaultName.name,
            DockerBuildNetwork.name,
            DockerBuildArg.name,
            DockerRegistryParameter.name,
        ],
        RPM: [
            PublishingBlock.name,
            PublishPackageParameter.name,
            MultiplePublishParameter.name,
            MultiplePublishToParameter.name,
            PublishToParameter.name,
        ],
        AAR: [
            PublishingBlock.name,
            PublishPackageParameter.name,
            MultiplePublishParameter.name,
            MultiplePublishToParameter.name,
            PublishToParameter.name,
        ],
        WHEEL: [
            WheelPython3Parameter.name,
            PublishingBlock.name,
            PublishPackageParameter.name,
            WheelUploadRepo.name,
            WheelAccessKeyTokenVaultName.name,
            WheelSecretKeyTokenVaultName.name,
        ],
        NPM: [
            PublishingBlock.name,
            PublishPackageParameter.name,
            NpmRegistryParameter.name,
            NpmLoginParameter.name,
            NpmPasswordVaultName.name,
        ],
        TARBALL: [
            PublishingBlock.name,
            ArtifactoryUploadParameter.name,
            parameters.ArtifactoryPassword.name,
            PublishPackageParameter.name,
            PublishToParameter.name,
        ],
    }


def extract_test_data_paths(packages, arcadia_path, test_data_prefix):
    test_data = []

    for package in packages:
        if not os.path.isabs(package):
            eh.ensure(
                not package.startswith('arcadia/'),
                'Package path {} is not relative to arcadia/'.format(package)
            )

            package_file_path = os.path.join(arcadia_path, package)
        else:
            package_file_path = package

        eh.ensure(
            os.path.exists(package_file_path),
            'Package path {} does not exist in arcadia'.format(package),
        )

        try:
            parsed_package = json.load(open(package_file_path))  # XXX: need to support includes
        except Exception as e:
            raise Exception("Failed to load package {}: {}".format(package_file_path, e))
        for element in parsed_package.get('data', []):
            source = element.get('source', None) or element.get('src', None)
            if source and source.get('type', '').lower() in ['data', 'test_data']:
                path = source.get('path', None)
                if path and path.startswith(test_data_prefix.rstrip('/') + '/'):
                    test_data.append(path)

    return test_data


def checkout_test_data(test_data, svn_root, arcadia_url, task):
    """
        Checkout subdirectories from arcadia_tests_data and data.

        :param packages: list of json files in ya package format
        :param arcadia_url: url to arcadia svn(possibly in a branch)
        :param svn_root: root directory for arcadia, arcadia_tests_data and data
        :param task: current BUILD_PACKAGE task
    """
    created_data_directories = set()
    for data_directory, path in test_data:
        svn_path = os.path.normpath(sandboxsdk.svn.Arcadia.append(arcadia_url, os.path.join('..', path)))
        sandboxsdk.svn.ArcadiaTestData.get_arcadia_test_data(task, svn_path)

        if data_directory not in created_data_directories:
            data_directory_location = sandboxsdk.svn.ArcadiaTestData.test_data_location(svn_path)[0]
            logger.info('New data directory location %s', data_directory_location)

            symlink_path = os.path.join(svn_root, data_directory)
            os.symlink(data_directory_location, symlink_path)
            logger.info('Symlink created %s -> %s', symlink_path, data_directory_location)

            created_data_directories.add(data_directory)


def _parse_multi_parameter(p):
    return [s for s in itertools.chain.from_iterable(x.split(';') for x in p.split()) if s]


# This function is needed for short period of time after release
# of dbg_resource_id functionality, to allow correct work of task, created
# before release.
def _get_resources_pair(descr):
    return descr if isinstance(descr, tuple) \
        else (descr, None)


class YaPackage(ArcadiaTask.ArcadiaTask, nanny.ReleaseToNannyTask, release_integration.ReleaseToYaDeployTask):
    """
        This task type is DEPRECATED.
        Please consider migrating to compatible KOSHER_YA_PACKAGE task based on sdk2.
        See announce at https://mvel.at.yandex-team.ru/8271/

        Packaging: https://docs.yandex-team.ru/devtools/package/
    """

    type = 'YA_PACKAGE'
    client_tags = ctc.Tag.LXC | ctc.Tag.ARCADIA_HG

    input_parameters = [
        parameters.ArcadiaUrl,
        parameters.DoNotRemoveResources,
        parameters.ArcadiaPatch,
        parameters.UseArcadiaApiFuseAsDefault,
        parameters.UseArcInsteadOfArcadiaApiAsDefault,
        parameters.AllowArcadiaApiFallback,
        parameters.ArcSecret,
        parameters.YavToken,
        parameters.MinimizeArcMountPath,

        parameters.BuildType,
        parameters.UsePrebuiltTools,
        HostPlatformParameter,
        TargetPlatformParameter,
        ClearBuildParameter,
        SemiClearBuildParameter,
        parameters.ForceBuildDepends,
        parameters.ForceVCSInfoUpdate,
        parameters.IgnoreRecurses,
        parameters.Sanitize,
        parameters.Musl,
        parameters.LTO,
        parameters.ThinLTO,
        parameters.EnvironmentVarsParam,
        XcodeVersionParameter,

        PackagesBlock,
        # Packages
        PackagesParameter,
        PackageTypeParameter,
        WheelPython3Parameter,
        RawPackageParameter,
        StripBinariesParameter,
        FullStripBinariesParameter,
        CreateDebugPackagesParameter,
        CompressPackageArchiveParameter,
        parameters.CompressionFilter,
        parameters.CompressionLevel,
        OverwriteReadOnlyFilesParameter,
        ArchitectureAllParameter,
        ForceDuploadParameter,
        DuploadMaxAttemptsParameter,
        BuildDebianScriptsParameter,
        DebianCompressionParameter,
        DebianCompressionTypeParameter,
        SloppyDebianParameter,
        DontStoreDebianPackageParameter,
        DebianDistributionParameter,
        DebianArchParameter,

        DockerBlock,
        DockerImageRepositoryParameter,
        DockerSaveImageParameter,
        DockerPushImageParameter,
        DockerRegistryParameter,
        DockerUserParameter,
        DockerTokenVaultName,
        DockerBuildNetwork,
        DockerBuildArg,

        TestsBlock,
        # Tests
        RunTestsParameter,
        RunMediumTestsParameter,
        RunLongTestsParameter,
        IgnoreFailTestsParameter,
        CreateBuildOutputResourceParameter,
        parameters.CacheTests,

        ResourcesBlock,
        # Resources
        ResourceTypeParameter,
        ResourceIdParameter,
        SaveBuildOutputParameter,

        PublishingBlock,
        # Publishing
        ArtifactoryUploadParameter,
        parameters.ArtifactoryPassword,
        ChangelogParameter,
        PublishPackageParameter,
        KeyUserParameter,
        MultiplePublishParameter,
        PublishToParameter,
        MultiplePublishToParameter,
        MultiplePublishMappingParameter,
        EnsurePackagePublishedParameter,
        # Wheel repo
        WheelUploadRepo,
        WheelAccessKeyTokenVaultName,
        WheelSecretKeyTokenVaultName,
        # npm
        NpmRegistryParameter,
        NpmLoginParameter,
        NpmPasswordVaultName,
        # Nanny release
        nanny.ReleaseToNannyParameter,
        nanny.ReleaseToAdmNannyParameter,
        # Ya.Deploy release integration: https://wiki.yandex-team.ru/deploy/release-integration/
        release_integration.YpTokenVaultParameter,
        release_integration.ReleaseToYaDeployParameter,
        release_integration.ReleasePerResourceParameter,

        AdvancedBlock,
        # Advanced
        AdhocPackagesParameter,
        UseYaDevParameter,
        BeVerboseParameter,
        UseNewFormatParameter,
        parameters.CheckoutModeParameter,
        parameters.CheckoutParameter,
        CustomVersionParameter,
        PackageResourceDescriptionParameter,
        PackageResourceAttrsParameter,
        parameters.BuildSystem,
        parameters.DistbuildPool,
        parameters.YaTimeout,
        ContainerParameter,
    ] + parameters.get_yt_store_params()

    execution_space = 100 * 1024

    arcadia_src_dir = None

    def resource_types_iter(self):
        rt = _parse_multi_parameter(self.ctx.get(ResourceTypeParameter.name))
        logger.info('$$ Resource types {}'.format(rt))
        return itertools.chain(rt, itertools.repeat(rt[-1]))

    def resource_ids_list(self):
        return [
            int(resource_id_str)
            for resource_id_str in _parse_multi_parameter(self.ctx.get(ResourceIdParameter.name, ""))
            if resource_id_str.isdigit()
        ]

    def resource_ids_iter(self):
        resource_ids = self.resource_ids_list()
        logger.info('$$ Resource ids {}'.format(resource_ids))
        return itertools.chain(resource_ids, itertools.repeat(None))

    def package_type_map(self):
        packages = _parse_multi_parameter(self.ctx.get(PackagesParameter.name))
        logger.info('$$ Packages {}'.format(packages))
        types = self.resource_types_iter()
        ids = self.resource_ids_iter()
        return zip(packages, types, ids)

    # Used by MAPS_DOCKER task
    def prebuild_hook(self):
        pass

    def on_enqueue(self):
        channel.task = self

        if PACKAGE_RESOURCES not in self.ctx:
            if self.ctx.get(PackageTypeParameter.name) in [DEBIAN, RPM, WHEEL, AAR]:
                logger.info("self.platform = %s", self.platform)

            if self.ctx.get(PackageTypeParameter.name) == DOCKER:
                self.dns = ctm.DnsType.DNS64
                if not self.ctx.get(ContainerParameter.name):
                    self.ctx[ContainerParameter.name] = 773239891  # default docker container id for YA_PACKAGE
            elif self.ctx.get(PackageTypeParameter.name) == AAR:
                self.dns = ctm.DnsType.DNS64
            elif self.ctx.get(PackageTypeParameter.name) == DEBIAN:
                if self.platform == 'linux_ubuntu_18.04_bionic' and not self.ctx.get(ContainerParameter.name):
                    self.ctx[ContainerParameter.name] = 855378432  # default docker container id bionic with dev tools
            elif self.ctx.get(PackageTypeParameter.name) == RPM:
                if not self.ctx.get(ContainerParameter.name):
                    self.ctx[ContainerParameter.name] = 1116457063  # default docker container id xenial with rpmbuild
            elif self.ctx.get(PackageTypeParameter.name) == WHEEL:
                if not self.ctx.get(ContainerParameter.name):
                    # default docker container id bionic with setuptools wheel
                    self.ctx[ContainerParameter.name] = 2287800509
            elif self.ctx.get(PackageTypeParameter.name) == NPM:
                self.dns = ctm.DnsType.DNS64
                if not self.ctx.get(ContainerParameter.name):
                    self.ctx[ContainerParameter.name] = 1724812895  # default docker container id bionic with npm & node

            strip_binaries = self.ctx.get(StripBinariesParameter.name)
            if not strip_binaries:
                self.ctx[CreateDebugPackagesParameter.name] = False

            create_dbg = self.ctx.get(CreateDebugPackagesParameter.name)
            is_tarball = self.ctx.get(PackageTypeParameter.name) == TARBALL
            raw_package = self.ctx.get(RawPackageParameter.name)

            if raw_package and not is_tarball:
                raise sandboxsdk.errors.SandboxTaskFailureError("You can get raw package only using 'tar' package type")

            packages = []
            resources = []

            for path, resource_type, resource_id in self.package_type_map():
                packages.append(path)
                if not self.ctx.get(DontStoreDebianPackageParameter.name):
                    if resource_id is not None:
                        resource = sdk2.Resource[resource_id]
                        self.update_foreign_package_resource(path, resource, False)
                    else:
                        resource = self.init_package_resource(path, resource_type, False)
                    if create_dbg and is_tarball:
                        debug_resource = self.init_package_resource(path, resource_types.ARCADIA_PROJECT_SYMBOLS, True)
                    else:
                        debug_resource = None
                    resources.append((resource.id, debug_resource and debug_resource.id))

            self.ctx[PACKAGE_RESOURCES] = {'packages': packages, 'resources': resources}
            if self.ctx.get(MultiplePublishMappingParameter.name):
                self.ctx[PUBLISH_TO_LIST] = self.ctx.get(MultiplePublishMappingParameter.name)
            elif self.ctx.get(MultiplePublishParameter.name):
                self.ctx[PUBLISH_TO_LIST] = self.ctx.get(MultiplePublishToParameter.name).split(';')
            elif self.ctx.get(PublishToParameter.name):
                self.ctx[PUBLISH_TO_LIST] = [self.ctx.get(PublishToParameter.name)]
            elif self.ctx.get(WheelUploadRepo.name) and self.ctx.get(PackageTypeParameter.name) == WHEEL:
                self.ctx[PUBLISH_TO_LIST] = self.ctx.get(WheelUploadRepo.name).split(';')
            elif self.ctx.get(NpmRegistryParameter.name) and self.ctx.get(PackageTypeParameter.name) == NPM:
                self.ctx[PUBLISH_TO_LIST] = [self.ctx.get(NpmRegistryParameter.name)]
            else:
                self.ctx[PUBLISH_TO_LIST] = []

        deb_package = self.ctx.get(PackageTypeParameter.name) == DEBIAN

        if deb_package:
            # deb package management is done via "debchange" binary, which is available in "devscripts" package,
            # which is only guaranteed to have been installed in Sandbox's standard LXC container images
            self.client_tags = self.__class__.client_tags & ctc.Tag.LXC

    def on_execute(self):
        logging.info("Process on_execute")
        xcode_version = self.ctx.get(consts.XCODE_VERSION)
        if xcode_version:
            Xcode.prepare_xcode(xcode_version)
        parsed_url = sandboxsdk.svn.Arcadia.parse_url(self.ctx.get(parameters.ArcadiaUrl.name))
        is_hg_repo = (
            urlparse.urlparse(self.ctx.get(parameters.ArcadiaUrl.name)).scheme ==
            sandboxsdk.svn.Arcadia.ARCADIA_HG_SCHEME
        )
        is_arc_repo = (
            urlparse.urlparse(self.ctx.get(parameters.ArcadiaUrl.name)).scheme ==
            sandboxsdk.svn.Arcadia.ARCADIA_ARC_SCHEME
        )
        if not is_hg_repo and not is_arc_repo and not parsed_url.trunk and not parsed_url.branch and not parsed_url.tag:
            logging.error("Cannot determine branch from %s: %s", self.ctx.get(parameters.ArcadiaUrl.name), parsed_url)
            eh.check_failed("Cannot determine branch from {}, make sure it ends with '/arcadia'".format(
                self.ctx.get(parameters.ArcadiaUrl.name)))

        if filter(None, self.ctx.get(AdhocPackagesParameter.name) or []):
            self._init_adhoc_packages()

        use_aapi_fuse = self.ctx.get(consts.USE_AAPI_FUSE)
        use_arc_instead_of_aapi = self.ctx.get(consts.USE_ARC_INSTEAD_OF_AAPI)
        aapi_fallback = self.ctx.get(consts.ALLOW_AAPI_FALLBACK)
        copy_trunk = self.ctx.get(consts.COPY_TRUNK)
        use_arc_vcs = is_arc_repo
        use_fuse = use_aapi_fuse or use_arc_vcs
        distbuild_pool = self.ctx.get(consts.DISTBUILD_POOL)

        checkout = False if use_fuse else self.should_checkout()
        if (
            not use_fuse and not checkout and self.owner in ya_sdk_compat.GROUPS_WITH_FORCE_ARC and
            self.type in ya_sdk_compat.TASKS_WITH_FORCE_ARC
        ):
            use_arc_vcs = True
            use_fuse = True
            use_arc_instead_of_aapi = True

        if use_aapi_fuse and aapi_fallback and not use_arc_vcs and not use_arc_instead_of_aapi:
            if not self.check_aapi_available():
                use_aapi_fuse = False
                logger.info('Disable AAPI, because service is temporary unavailable or inapplicable')

        if use_fuse and not sdk.fuse_available():
            raise sandboxsdk.errors.SandboxTaskFailureError(
                'Fuse is not available on {} yet'.format(platform.platform())
            )

        if self.ctx.get(PackageTypeParameter.name) == DOCKER and not self.ctx.get(DockerPushImageParameter.name):
            self.set_info("<strong>Docker image will not be pushed due to 'Push docker image' is set to False</strong>")

        logger.info('Mount = %s (using %s)', use_fuse, 'Arc VCS' if use_arc_vcs else 'AAPI')

        packages = self.ctx[PACKAGE_RESOURCES]['packages']
        eh.ensure(packages, "Packages are not specified")
        logger.info('Packages: %s', packages)

        wheel_access_key_tf = None
        wheel_secret_key_tf = None
        if self.ctx.get(PackageTypeParameter.name) == WHEEL and self.ctx.get(PublishPackageParameter.name):
            wheel_access_key_tf, wheel_secret_key_tf = self._setup_wheel_repo_keys()

        if self.ctx.get(PackageTypeParameter.name) == DOCKER:
            self._docker_login()

        if self.ctx.get(PackageTypeParameter.name) == NPM:
            self._npm_login()

        with self.get_arcadia(use_fuse, checkout, copy_trunk, use_arc_instead_of_aapi) as arcadia_src_dir:
            self.arcadia_src_dir = arcadia_src_dir
            self.prebuild_hook()
            changelog = self.ctx.get(ChangelogParameter.name, "")
            if changelog and os.path.exists(os.path.join(self.arcadia_src_dir, changelog)):
                changelog = os.path.join(self.arcadia_src_dir, changelog)
                logger.info("Change log: %s", changelog)
                changelog_is_path = True
            else:
                changelog_is_path = False

            if self.ctx.get(parameters.ArcadiaPatch.name):
                if checkout:
                    eh.check_failed(
                        'Patch can be applied only on full arcadia for now. Disable parameter "{}"'.format(
                            parameters.CheckoutParameter.description))
                patch_path = sdk.apply_patch(
                    self,
                    self.arcadia_src_dir,
                    self.ctx.get(parameters.ArcadiaPatch.name),
                    self.abs_path()
                )
                if patch_path and not changelog_is_path:
                    changelog += "Package was created with patch {}".format(
                        string.to_utf8(self.ctx.get(parameters.ArcadiaPatch.name))
                    )

            with self.get_test_data(
                arcadia_src_dir, packages, 'arcadia_tests_data',
                use_arc_vcs, use_aapi_fuse, checkout, use_arc_instead_of_aapi,
            ) as arcadia_tests_data_dir, \
                self.get_test_data(
                    arcadia_src_dir, packages, 'data', use_arc_vcs,
                    use_aapi_fuse, checkout, use_arc_instead_of_aapi) as data_dir:

                if use_arc_vcs:
                    revision = str(sdk.mounted_path_svnversion(self.arcadia_src_dir, True)['revision'])

                elif use_aapi_fuse:
                    if is_hg_repo:
                        revision = str(sdk.mounted_path_svnversion(self.arcadia_src_dir)['hash'])
                    else:
                        revision = str(sdk.mounted_path_svnversion(self.arcadia_src_dir)['revision'])

                else:
                    if is_hg_repo:
                        revision = sandboxsdk.svn.Arcadia.get_revision(self.ctx.get(parameters.ArcadiaUrl.name))
                    else:
                        revision = sandboxsdk.svn.Arcadia.get_revision(self.arcadia_src_dir)

                logger.info('Arcadia path: {}, at rev {}'.format(self.arcadia_src_dir, revision))
                self.ctx['revision'] = revision
                common.rest.Client().task.current.context.value = {'key': 'revision', 'value': self.ctx['revision']}

                resources = iter(self.ctx[PACKAGE_RESOURCES]['resources'])

                fake_svn_root = os.path.join(os.getcwd(), 'fake-svn-root')
                os.mkdir(fake_svn_root)
                logger.info('Fake svn root: %s', fake_svn_root)

                # TODO: Remove it when ya package will not take --source-root as a directory containing 'arcadia'
                arcadia_src_symlink_path = os.path.join(fake_svn_root, 'arcadia')
                os.symlink(self.arcadia_src_dir, arcadia_src_symlink_path)
                logger.info('Symlink created %s -> %s', arcadia_src_symlink_path, self.arcadia_src_dir)

                # TODO: Remove arcadia_tests_data and data from packages and remove this checkout
                if arcadia_tests_data_dir:
                    atd_symlink_path = os.path.join(fake_svn_root, 'arcadia_tests_data')
                    os.symlink(arcadia_tests_data_dir, atd_symlink_path)
                    logger.info('Symlink created %s -> %s', atd_symlink_path, arcadia_tests_data_dir)
                if data_dir:
                    data_symlink_path = os.path.join(fake_svn_root, 'data')
                    os.symlink(data_dir, data_symlink_path)
                    logger.info('Symlink created %s -> %s', data_symlink_path, data_dir)

                key_user = self._get_key_user()

                publish_to = self.ctx.get(PUBLISH_TO_LIST, [])
                if publish_to and isinstance(publish_to, list):
                    # otherwise it is a publish mapping and should be transferred as is
                    if self.ctx.get(PackageTypeParameter.name) == AAR:
                        publish_to = [
                            os.path.join(arcadia_src_symlink_path, settings_xml)
                            for settings_xml in publish_to
                        ]
                    publish_to = ";".join(publish_to)

                build_system = self.ctx.get(consts.BUILD_SYSTEM_KEY)

                debian_compression_level = None
                if self.ctx.get(DebianCompressionParameter.name):
                    debian_compression_level = self.ctx.get(DebianCompressionParameter.name)

                debian_compression_type = None
                if self.ctx.get(DebianCompressionTypeParameter.name):
                    debian_compression_type = self.ctx.get(DebianCompressionTypeParameter.name)

                debian_store_package = not self.ctx.get(DontStoreDebianPackageParameter.name)

                yt_store_token = ya_sdk_compat.get_yt_store_vault_data(self)

                yt_store_params = sdk.YtStoreParams(
                    self.ctx.get(consts.YA_YT_STORE, True),
                    yt_store_token,
                    self.ctx.get(consts.YA_YT_PROXY, None),
                    self.ctx.get(consts.YA_YT_DIR, None),
                    self.ctx.get(consts.YA_YT_PUT, False),
                    self.ctx.get(consts.YA_YT_STORE_CODEC, None),
                    self.ctx.get(consts.YA_YT_REPLACE_RESULT, False),
                    self.ctx.get(consts.YA_YT_REPLACE_RESULT_ADD_OBJECTS, False),
                    self.ctx.get(consts.YA_YT_REPLACE_RESULT_RM_BINARIES, False),
                    self.ctx.get(consts.YA_YT_MAX_CACHE_SIZE, None),
                    self.ctx.get(consts.YA_YT_STORE_EXCLUSIVE, False),
                    self.ctx.get(consts.YA_YT_STORE_THREADS, None),
                    self.ctx.get(consts.YA_YT_STORE_REFRESH_ON_READ, False),
                )
                yt_store_params.report_status(self)

                def _do_package():
                    env = self.get_env_vars()
                    if env:
                        with custom_env(env):
                            return _do_sdk_package()
                    else:
                        return _do_sdk_package()

                def _do_sdk_package():
                    sdk.do_package(
                        fake_svn_root,
                        packages,
                        use_ya_dev=self.ctx.get(UseYaDevParameter.name),
                        be_verbose=self.ctx.get(BeVerboseParameter.name),
                        artifactory=self.ctx.get(ArtifactoryUploadParameter.name),
                        artifactory_password=self.ctx.get(parameters.ArtifactoryPassword.name),
                        publish=self.ctx.get(PublishPackageParameter.name),
                        sloppy_deb=self.ctx.get(SloppyDebianParameter.name),
                        publish_to=publish_to,
                        key=key_user,
                        changelog=changelog,
                        run_tests=1 if self.ctx.get(RunTestsParameter.name, False) else 0,
                        run_long_tests=self.ctx.get(RunLongTestsParameter.name),
                        run_medium_tests=self.ctx.get(RunMediumTestsParameter.name),
                        ignore_fail_tests=self.ctx.get(IgnoreFailTestsParameter.name),
                        sandbox_task_id=self.id,
                        debian=(self.ctx.get(PackageTypeParameter.name) == DEBIAN),
                        build_type=self.ctx.get(parameters.BuildType.name),
                        strip_binaries=self.ctx.get(StripBinariesParameter.name),
                        full_strip_binaries=self.ctx.get(FullStripBinariesParameter.name),
                        create_dbg=self.ctx.get(CreateDebugPackagesParameter.name),
                        compress_archive=self.ctx.get(CompressPackageArchiveParameter.name),
                        compression_filter=self.ctx.get(parameters.CompressionFilter.name),
                        compression_level=self.ctx.get(parameters.CompressionLevel.name),
                        clear_build=self.ctx.get(ClearBuildParameter.name),
                        arch_all=self.ctx.get(ArchitectureAllParameter.name),
                        force_dupload=self.ctx.get(ForceDuploadParameter.name),
                        dupload_max_attempts=int(self.ctx.get(
                            DuploadMaxAttemptsParameter.name,
                            DuploadMaxAttemptsParameter.default_value,
                        )),
                        build_debian_scripts=self.ctx.get(BuildDebianScriptsParameter.name),
                        debian_distribution=self.ctx.get(DebianDistributionParameter.name),
                        convert=not self.ctx.get(UseNewFormatParameter.name),
                        host_platform=self.ctx.get(HostPlatformParameter.name),
                        target_platform=self.ctx.get(TargetPlatformParameter.name),
                        sanitize=self.ctx.get(parameters.Sanitize.name),
                        checkout=checkout,
                        html_display=os.path.join(self.log_path(), 'output.html'),
                        custom_version=self.ctx.get(CustomVersionParameter.name),
                        build_system=build_system,
                        distbuild_pool=distbuild_pool,
                        debian_compression_level=debian_compression_level,
                        debian_store_package=debian_store_package,
                        musl=self.ctx.get(parameters.Musl.name),
                        debian_compression_type=debian_compression_type,
                        lto=self.ctx.get(parameters.LTO.name),
                        thinlto=self.ctx.get(parameters.ThinLTO.name),
                        timeout=int(self.ctx.get(consts.YA_TIMEOUT, 10800) or 10800),
                        semi_clear_build=self.ctx.get(SemiClearBuildParameter.name),
                        raw_package=self.ctx.get(RawPackageParameter.name),
                        yt_store_params=yt_store_params,
                        force_build_depends=self.ctx.get(parameters.ForceBuildDepends.name),
                        force_vcs_info_update=self.ctx.get(parameters.ForceVCSInfoUpdate.name, False),
                        ignore_recurses=self.ctx.get(parameters.IgnoreRecurses.name),
                        build_docker=(self.ctx.get(PackageTypeParameter.name) == DOCKER),
                        docker_image_repository=self.ctx.get(DockerImageRepositoryParameter.name),
                        docker_save_image=self.ctx.get(DockerSaveImageParameter.name),
                        docker_push_image=self.ctx.get(DockerPushImageParameter.name),
                        docker_registry=self.ctx.get(DockerRegistryParameter.name),
                        docker_build_network=self.ctx.get(DockerBuildNetwork.name),
                        docker_build_arg=self.ctx.get(DockerBuildArg.name),
                        rpm=(self.ctx.get(PackageTypeParameter.name) == RPM),
                        wheel=(self.ctx.get(PackageTypeParameter.name) == WHEEL),
                        wheel_access_key_path=wheel_access_key_tf.name if wheel_access_key_tf else None,
                        wheel_secret_key_path=wheel_secret_key_tf.name if wheel_secret_key_tf else None,
                        overwrite_read_only_files=self.ctx.get(OverwriteReadOnlyFilesParameter.name, False),
                        ensure_package_published=self.ctx.get(EnsurePackagePublishedParameter.name, False),
                        aar=(self.ctx.get(PackageTypeParameter.name) == AAR),
                        use_prebuilt_tools=self.ctx.get(parameters.UsePrebuiltTools.name),
                        wheel_python3=self.ctx.get(WheelPython3Parameter.name, False),
                        npm=(self.ctx.get(PackageTypeParameter.name) == NPM),
                        create_build_output_resource=self.ctx.get(CreateBuildOutputResourceParameter.name, False),
                        debian_arch=self.ctx.get(DebianArchParameter.name),
                        cache_tests=self.ctx.get(parameters.CacheTests.name, False),
                    )

                try:
                    if key_user:
                        self._run_with_gpg_and_ssh(_do_package, key_user)
                    else:
                        eh.ensure(
                            not self.ctx.get(PublishToParameter.name) and
                            not self.ctx.get(MultiplePublishToParameter.name) and
                            not self.ctx.get(MultiplePublishMappingParameter.name),
                            'You can not publish package without user key selected'
                        )
                        _do_package()
                except Exception:
                    output_html = os.path.join(self.log_path(), "output.html")
                    if os.path.exists(output_html):
                        self.set_info(
                            "See human-readable build errors here: "
                            "<a href='{}/output.html'>output.html</a>".format(self._log_resource.proxy_url),
                            do_escape=False
                        )
                    raise
                finally:
                    junit_report_file_mask = 'junit*.xml'
                    for junit_report_file in glob.glob(junit_report_file_mask):
                        if os.path.exists(junit_report_file):
                            shutil.copy(
                                junit_report_file, os.path.join(self.log_path(), os.path.basename(junit_report_file))
                            )
                            self.create_resource('JUnit report', junit_report_file, resource_types.JUNIT_REPORT.name)

                    if self.ctx.get(SaveBuildOutputParameter.name) and os.path.exists('build') and os.listdir('build'):
                        self.create_resource(
                            description='Build output',
                            resource_path='build',
                            resource_type=resource_types.BUILD_OUTPUT
                        )

                    for fname in glob.glob("*.json"):
                        if fname != "packages.json":
                            shutil.copy(fname, os.path.join(self.log_path(), fname))

                if os.path.exists(PACKAGES_FILE):
                    with open(PACKAGES_FILE) as packages_file:
                        built_packages = json.load(packages_file)

                        logging.debug("packages.json: %s", json.dumps(built_packages, indent=4, sort_keys=True))

                        def populate(resource_id, filename):
                            if "docker_image" in package:
                                version = package["docker_image"]
                            else:
                                version = package['version']
                            self.populate_package_resource(
                                package['name'],
                                resource_id,
                                filename,
                                version,
                                revision,
                            )

                        if isinstance(built_packages, list):
                            package_jsons = iter(packages)
                            self.ctx[OUTPUT_RESOURCES] = {}
                            for package in built_packages:
                                resource_id, dbg_resource_id = _get_resources_pair(next(resources))
                                logging.debug("Got resource_id %s, %s", repr(resource_id), type(resource_id))
                                populate(resource_id, package['path'])

                                if dbg_resource_id:
                                    populate(dbg_resource_id, package['debug_path'])

                                package_json = next(package_jsons)
                                PLATFORM_RUN = "platform.run"
                                platform_run_path = os.path.join(
                                    self.arcadia_src_dir, os.path.dirname(package_json), PLATFORM_RUN
                                )
                                logging.debug(
                                    "os.path.exists(%s): %s",
                                    platform_run_path, os.path.exists(platform_run_path),
                                )
                                if os.path.exists(platform_run_path):
                                    # copy to an unique local dir so that it can be shared as a resource
                                    package_json_local_dir = os.path.dirname(package_json)
                                    os.makedirs(package_json_local_dir)
                                    shutil.copy(platform_run_path, package_json_local_dir)
                                    self._create_resource(
                                        'platform.run for {}'.format(package_json),
                                        os.path.join(package_json_local_dir, PLATFORM_RUN),
                                        resource_types.PLATFORM_RUN,
                                        attrs=self._get_common_package_resource_attrs(),
                                    )

                        else:
                            logging.info('Unordered version of packages.json, cannot match to pre-created resources')

                            for name, info in packages.iteritems():
                                if '.' in name:
                                    name = name.split('.')[0]

                                resource = next(resources)
                                if isinstance(resource, tuple):
                                    resource = resource[0]

                                self.populate_package_resource(
                                    name,
                                    resource,
                                    info['package'],
                                    info['version'],
                                    revision
                                )

    def _init_adhoc_packages(self):
        packages = []
        resources = []
        res_types = self.resource_types_iter()
        for i, package_json in enumerate(self.ctx[AdhocPackagesParameter.name]):
            if not package_json.strip():
                logging.warning("Empty adhoc package json")
                continue
            path = self.log_path("adhoc_package_{}.json".format(i))
            packages.append(path)
            with open(path, "w") as pf:
                json.dump(json.loads(package_json), pf)

            resource_type = next(res_types)

            if not self.ctx.get(DontStoreDebianPackageParameter.name):
                create_dbg = self.ctx.get(CreateDebugPackagesParameter.name)
                is_tarball = self.ctx.get(PackageTypeParameter.name) == TARBALL

                resource = self.init_package_resource(path, resource_type, False)
                if create_dbg and is_tarball:
                    debug_resource = self.init_package_resource(path, resource_types.ARCADIA_PROJECT_SYMBOLS, True)
                else:
                    debug_resource = None
                resources.append((resource.id, debug_resource and debug_resource.id))

        self.ctx[PACKAGE_RESOURCES]['packages'] = packages
        self.ctx[PACKAGE_RESOURCES]['resources'] = resources
        return packages

    def _get_common_package_resource_attrs(self):
        arcadia_url = self.ctx.get(parameters.ArcadiaUrl.name)
        do_not_remove = self.ctx.get(parameters.DoNotRemoveResources.name)
        parsed_url = sandboxsdk.svn.Arcadia.parse_url(arcadia_url)
        revision, branch, tag = parsed_url[1], parsed_url[2], parsed_url[3]
        build_type = self.ctx.get(parameters.BuildType.name)

        return {
            'svn_path': arcadia_url,
            'svn_revision': self.ctx.get('revision') or revision or 'HEAD',
            'build_type': build_type,
            'branch': branch or tag or 'trunk',
            'platform': 'unknown',
            'ttl': 'inf' if do_not_remove else 30,
        }

    def update_foreign_package_resource(self, path, resource, is_debug):
        attrs = self._get_common_package_resource_attrs()
        attrs.update(self.ctx.get(PackageResourceAttrsParameter.name) or {})
        for attr_key, attr_value in attrs.items():
            setattr(resource, attr_key, attr_value)

    def init_package_resource(self, path, resource_type, is_debug):
        resource_filename = path.replace('/', '.')
        if is_debug:
            resource_filename += '.debug'
        if self.ctx.get(PackageResourceDescriptionParameter.name):
            resource_description = self.ctx.get(PackageResourceDescriptionParameter.name).get(path, path)
        else:
            resource_description = path
        attrs = self.ctx.get(PackageResourceAttrsParameter.name) or {}
        attrs.update(self._get_common_package_resource_attrs())
        return self._create_resource(
            resource_description,
            resource_filename,
            resource_type,
            attrs=attrs,
        )

    def populate_package_resource(self, name, resource_id, path, version, revision):
        prepare_foreign_resource = resource_id in self.resource_ids_list()
        if prepare_foreign_resource:
            logger.warning("Skipping change resource basename because resource #%s is not mine", resource_id)
        else:
            self.change_resource_basename(resource_id, os.path.basename(path))

        self.set_info('{}={}'.format(name, version))
        self.ctx['output_resource_version'] = version
        self.ctx[OUTPUT_RESOURCES][name] = version

        if revision:
            channel.sandbox.set_resource_attribute(resource_id, 'svn_revision', revision)

        channel.sandbox.set_resource_attribute(
            resource_id, 'platform', self.ctx.get(TargetPlatformParameter.name) or self.platform
        )
        channel.sandbox.set_resource_attribute(resource_id, 'resource_name', name)
        channel.sandbox.set_resource_attribute(resource_id, 'resource_version', version)

        build_type = channel.sandbox.get_resource_attribute(resource_id, 'build_type')
        branch = channel.sandbox.get_resource_attribute(resource_id, 'branch')

        description = '.'.join([name, 'linux', build_type, branch, revision])
        channel.sandbox.set_resource_attribute(resource_id, 'description', description)

        if prepare_foreign_resource:
            logger.info("Prepairing data for parent resource #%s", resource_id)
            self.save_parent_task_resource(path, resource_id)
            self.mark_resource_ready(resource_id)

    def check_aapi_available(self):
        try:
            url = self.ctx.get(parameters.ArcadiaUrl.name)
            return sdk.wait_aapi_url(url)
        except aapi.ArcadiaApiCommandFailed as e:
            logger.warning("AAPI availability check failed: %r", e)
            return False

    def get_arcadia(self, use_fuse, checkout, copy_trunk=False, use_arc_instead_of_aapi=False):
        url = self.ctx.get(parameters.ArcadiaUrl.name)

        if use_fuse:
            return sdk.mount_arc_path(url, use_arc_instead_of_aapi=use_arc_instead_of_aapi, fetch_all=False)

        elif checkout:
            return context_managers.nullcontext(sdk.do_clone(self.ctx.get(parameters.ArcadiaUrl.name), self))

        else:
            return context_managers.nullcontext(
                ArcadiaTask.ArcadiaTask.get_arcadia_src_dir(self, copy_trunk=copy_trunk)
            )

    def get_test_data(
        self,
        arc_root,
        packages,
        test_data_prefix,
        use_arc_vcs,
        use_aapi_fuse,
        checkout,
        use_arc_instead_of_aapi=False,
    ):
        arcadia_url = self.ctx.get(parameters.ArcadiaUrl.name)

        if checkout:
            return context_managers.nullcontext(None)

        paths = extract_test_data_paths(packages, arc_root, test_data_prefix)

        if not paths:
            return context_managers.nullcontext(None)

        elif use_aapi_fuse and not use_arc_vcs:
            url = os.path.normpath(sandboxsdk.svn.Arcadia.append(arcadia_url, os.path.join('..', test_data_prefix)))
            if sandboxsdk.svn.Arcadia.check(url):
                return sdk.mount_arc_path(url, use_arc_instead_of_aapi=use_arc_instead_of_aapi, fetch_all=False)
            else:
                return context_managers.nullcontext(None)

        else:
            path_url = None
            for path in paths:
                path_url = os.path.normpath(sandboxsdk.svn.Arcadia.append(arcadia_url, os.path.join('..', path)))
                sandboxsdk.svn.ArcadiaTestData.get_arcadia_test_data(self, path_url)
            return context_managers.nullcontext(sandboxsdk.svn.ArcadiaTestData.test_data_location(path_url)[0])

    def on_release(self, additional_parameters):
        ArcadiaTask.ArcadiaTask.on_release(self, additional_parameters)
        if self.ctx.get(PackageTypeParameter.name) == DOCKER:
            image_name, image_tag = self._parse_docker_version(self.ctx.get('output_resource_version', ''))
            release_payload = {
                'spec': {
                    'type': "DOCKER_RELEASE",
                    'docker_release': {
                        "image_name": image_name,
                        "image_tag": image_tag,
                        "release_type": additional_parameters['release_status'].upper(),
                    }
                }
            }
            logging.info('Sending release of Docker task %s to Nanny', self.id)
            logging.info('Release payload: %s', json.dumps(release_payload, indent=4))
            result = self.nanny_client.create_release2(release_payload)
            logging.info('Release result is %s', result)
            if self.ctx.get(release_integration.ReleaseToYaDeployParameter.name):
                images = []
                for version in self.ctx.get(OUTPUT_RESOURCES).values():
                    image = release_integration.make_docker_image_from_path(version)
                    images.append(image)
                result = release_integration.create_docker_release(
                    task=self,
                    images=images,
                    additional_parameters=additional_parameters,
                )
                logging.info('Release result is %s', result)
                self.store_release_id_in_context(result['meta']['id'])
        elif self.ctx.get(PackageTypeParameter.name) == DEBIAN:
            key_user = self._get_key_user()
            publish_to = self.ctx.get(PUBLISH_TO_LIST)

            if publish_to and isinstance(publish_to, list):
                publish_to = itertools.chain(iter(publish_to), itertools.repeat(publish_to[-1]))

                def _dupload():
                    resources = self.ctx[PACKAGE_RESOURCES]['resources']

                    for resource in resources:
                        resource_id, _ = _get_resources_pair(resource)  # symbols not published
                        temp_dir = tempfile.mkdtemp()
                        with common.fs.WorkDir(temp_dir):
                            archive_path = self.sync_resource(resource_id)
                            self._subprocess('/bin/tar -xzf {}'.format(archive_path), wait=True)
                            self._subprocess('/usr/bin/dupload --to {}'.format(next(publish_to)), wait=True)
                        shutil.rmtree(temp_dir)

                if not self.ctx.get(PublishPackageParameter.name) and self.ctx.get(PackageTypeParameter.name) == DEBIAN:
                    if key_user and self.ctx.get(PUBLISH_TO_LIST):
                        self._run_with_gpg_and_ssh(_dupload, key_user)

            if self.ctx.get(nanny.ReleaseToNannyParameter.name):
                nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        elif self.ctx.get(PackageTypeParameter.name) == RPM:
            raise sandboxsdk.errors.SandboxTaskFailureError("Releasing rpm package supported only for nanny")
        elif self.ctx.get(PackageTypeParameter.name) == NPM:
            raise sandboxsdk.errors.SandboxTaskFailureError("Releasing npm package supported only for nanny")
        else:
            nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
            if self.ctx.get(release_integration.ReleaseToYaDeployParameter.name):
                release_integration.ReleaseToYaDeployTask.on_release(self, additional_parameters)
            if self.ctx.get(nanny.ReleaseToAdmNannyParameter.name):
                data = self.get_nanny_release_info(additional_parameters)
                logging.info('Sending release of task %s to adm-nanny', self.id)
                logging.info('Release payload: %s', json.dumps(data, indent=4))
                adm_nanny_client = nanny.NannyClient(api_url=nannyconsts.NANNY_ADM_API_URL,
                                                     oauth_token=None)
                result = adm_nanny_client.create_release2(data)
                logging.info('Release of task %s has been sent to adm-nanny', self.id)
                release_id = result['value']['id']
                release_link = nannyconsts.RELEASE_REQUEST_TEMPLATE.format(
                    nanny_api_url=nannyconsts.NANNY_ADM_API_URL,
                    release_request_id=release_id,
                )
                self.set_info(
                    'Adm-Nanny release <a href="{}">{}</a> was created.'.format(release_link, release_id),
                    do_escape=False
                )

    def _parse_docker_version(self, version):
        image_name, _, image_tag = version.partition(':')
        _, _, image_name = image_name.partition('/')
        return image_name, image_tag

    def _get_key_user(self):
        key_user = self.ctx.get(KeyUserParameter.name)

        if not key_user and self.ctx.get(PackageTypeParameter.name) == DEBIAN:
            for user, owner in ya_package_consts.VAULT_OWNERS.items():
                if owner == self.owner:
                    logger.info('Task owner was taken as a vault owner: user %s, owner %s', user, owner)
                    key_user = user
                    break

        return key_user

    def _run_with_gpg_and_ssh(self, f, key_user):
        eh.ensure(key_user in ya_package_consts.VAULT_OWNERS, 'Unknown keys owner, key_user = {}'.format(key_user))

        vault_owner = ya_package_consts.VAULT_OWNERS[key_user]
        logger.info('Vault owner: %s', vault_owner)

        secret_key_name = '{}-gpg-private'.format(key_user)
        logger.info('Secret GPG key name: %s', secret_key_name)

        public_key_name = '{}-gpg-public'.format(key_user)
        logger.info('Public GPG key name: %s', public_key_name)

        ssh_key_name = '{}-ssh'.format(key_user)
        logger.info('SSH key name: %s', ssh_key_name)

        os.environ['DEBEMAIL'] = '{}@yandex-team.ru'.format(key_user)

        with gnupg.GpgKey(self, vault_owner, secret_key_name, public_key_name):
            with debpkg.DebRelease(ya_package_consts.DUPLOAD_CONF, login=key_user):
                with sandboxsdk.ssh.Key(self, vault_owner, ssh_key_name):
                    f()

    def _docker_login(self):

        token_vault_name = self.ctx.get(DockerTokenVaultName.name, None)

        try:
            token = self.get_vault_data(self.owner, token_vault_name)

        except common.errors.VaultNotFound:

            if token_vault_name is None:
                raise

            # Last try: get token 'shared with' by name.
            token = self.get_vault_data(token_vault_name)

        self._subprocess('docker login {registry} -u {user} -p $DOCKER_TOKEN'.format(
            registry=self.ctx.get(DockerRegistryParameter.name),
            user=self.ctx.get(DockerUserParameter.name),
        ), wait=True, shell=True, extra_env={'DOCKER_TOKEN': token})

    def _npm_login(self):
        password_vault_name = self.ctx.get(NpmPasswordVaultName.name, None)

        password = self.get_vault_data(self.owner, password_vault_name)
        login = self.ctx.get(NpmLoginParameter.name, None)
        registry = self.ctx.get(NpmRegistryParameter.name)

        self._write_npmrc(registry, login, password)

    def _write_npmrc(self, registry, login, password):
        registry = urlparse.urlparse(registry).netloc or registry
        password = base64.b64encode(password)
        email = '{}@yandex-team.ru'.format(login)

        params = dict(
            registry=registry,
            login=login,
            password=password,
            email=email
        )

        content = '\\n'.join(map(lambda line: line.format(**params), [
            '//{registry}/:_password="{password}"',
            '//{registry}/:username={login}',
            '//{registry}/:email={email}',
            '//{registry}/:always-auth=false',
        ]))

        self._subprocess(
            'echo -e ${NPMRC_CONTENT} > $HOME/.npmrc', wait=True, shell=True, extra_env={'NPMRC_CONTENT': content}
        )

    def _setup_wheel_repo_keys(self):
        wheel_access_key_tf = None
        wheel_secret_key_tf = None

        access_key_vault_name = self.ctx.get(WheelAccessKeyTokenVaultName.name)
        if access_key_vault_name:
            key = self.get_vault_data(self.owner, access_key_vault_name)
            wheel_access_key_tf = tempfile.NamedTemporaryFile(prefix="access_key")
            wheel_access_key_tf.write(key)
            wheel_access_key_tf.flush()

        secret_key_vault_name = self.ctx.get(WheelSecretKeyTokenVaultName.name)
        if secret_key_vault_name:
            key = self.get_vault_data(self.owner, secret_key_vault_name)
            wheel_secret_key_tf = tempfile.NamedTemporaryFile(prefix="secret_key")
            wheel_secret_key_tf.write(key)
            wheel_secret_key_tf.flush()

        return wheel_access_key_tf, wheel_secret_key_tf


class YaPackageResource(sdk2.Resource):
    releasable = True
    branch = sdk2.parameters.String("Branch")
    build_type = sdk2.parameters.String("Build type")
    platform = sdk2.parameters.String("Platform")
    resource_name = sdk2.parameters.String("Resource name")
    resource_version = sdk2.parameters.String("Resource version")
    svn_path = sdk2.parameters.String("Svn path")
    svn_revision = sdk2.parameters.String("Svn revision")
    arc_commit = sdk2.parameters.String("Arc commit hash")
    major_release_num = sdk2.parameters.Integer("Major release number")
    minor_release_num = sdk2.parameters.Integer("Minor release number")


@contextmanager
def custom_env(env):
    orig = os.environ.copy()
    for k, v in env.items():
        os.environ[k] = v
    yield os.environ
    os.environ = orig
