# -*- coding: utf-8 -*-
import base64
import glob
import itertools
import json
import logging
import os
import platform
import shlex
import shutil
import tempfile
from six.moves.urllib import parse as urlparse

from sandbox import common, sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import binary_task
from sandbox.projects.common import context_managers
from sandbox.projects.common import debpkg
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import gnupg
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import string
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import parameters
from sandbox.projects.common.build import ya_package_internal
from sandbox.projects.common.build import ya_make_package
from sandbox.projects.common.build.ya_package_config import consts as ya_package_consts
from sandbox.projects.common.vcs import aapi
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.environments import Xcode
from sandbox.sandboxsdk.svn import Arcadia, ArcadiaTestData
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.projects.common.nanny.const as nanny_const
import sandbox.projects.common.nanny.nanny as nanny
import sandbox.projects.common.build.parameters as build_parameters
import sandbox.projects.common.constants as consts


class PatchPackageVersion(sdk2.parameters.Bool):
    name = 'patch_package_version'
    description = "Update 'meta'.'version' in package files before build"
    required = False
    default_value = False
    group = parameters.ADVANCED_GROUP_NAME


class PatchPackageVersionFiles(sdk2.parameters.List):
    name = 'patch_package_version_files'
    description = "Update 'meta'.'version' in package files only in listed files"
    required = False
    default_value = None
    group = parameters.ADVANCED_GROUP_NAME


class YaPackage2Parameters(sdk2.Task.Parameters):
    checkout_arcadia_from_url = build_parameters.ArcadiaUrl()
    do_not_remove_resources = build_parameters.DoNotRemoveResources2()
    arcadia_patch = build_parameters.ArcadiaPatch()
    use_aapi_fuse = build_parameters.UseArcadiaApiFuseAsDefault()
    use_arc_instead_of_aapi = build_parameters.UseArcInsteadOfArcadiaApiAsDefault()
    aapi_fallback = build_parameters.AllowArcadiaApiFallback()
    arc_secret = build_parameters.ArcSecret()
    yav_token = build_parameters.YavToken()
    copy_trunk = build_parameters.CopyTrunk()
    minimize_arc_mount_path = build_parameters.MinimizeArcMountPath()

    build_type = build_parameters.BuildType()
    use_prebuilt_tools = build_parameters.UsePrebuiltTools()
    host_platform = build_parameters.HostPlatform2()
    target_platform = build_parameters.TargetPlatform2()
    clear_build = build_parameters.ClearBuild2()
    semi_clear_build = build_parameters.SemiClearBuild()
    force_build_depends = build_parameters.ForceBuildDepends()
    force_vcs_info_update = build_parameters.ForceVCSInfoUpdate()
    ignore_recurses = build_parameters.IgnoreRecurses()
    sanitize = build_parameters.Sanitize()
    musl = build_parameters.Musl()
    lto = build_parameters.LTO()
    thinlto = build_parameters.ThinLTO()
    env_vars = build_parameters.EnvironmentVarsParam()
    xcode_version = sdk2.parameters.String("Required version of Xcode (version number, e. g. 13.1)")

    packages_block = sdk2.parameters.Info("Packages")
    packages = build_parameters.Packages()
    package_type = build_parameters.PackageType()
    wheel_python3 = build_parameters.WheelPython3()
    raw_package = build_parameters.RawPackage()
    raw_package_path = build_parameters.RawPackagePath()
    strip_binaries = build_parameters.StripBinaries2()
    full_strip_binaries = build_parameters.FullStripBinaries()
    create_debug_packages = build_parameters.CreateDebugPackages()
    compress_package_archive = build_parameters.CompressPackageArchive()
    compression_filter = build_parameters.CompressionFilter()
    compression_level = build_parameters.CompressionLevel()
    uc_codec = build_parameters.UcCodec()
    overwrite_read_only_files = build_parameters.OverwriteReadOnlyFiles()
    architecture_all = build_parameters.ArchitectureAll()
    force_dupload = build_parameters.ForceDupload()
    dupload_max_attempts = build_parameters.DuploadMaxAttempts()
    build_debian_scripts = build_parameters.BuildDebianScripts()
    debian_compression_level = build_parameters.DebianCompression()
    debian_compression_type = build_parameters.DebianCompressionType()
    sloppy_debian = build_parameters.SloppyDebian()
    debian_dont_store_package = build_parameters.DontStoreDebianPackage()
    debian_distribution = build_parameters.DebianDistribution()
    debian_arch = build_parameters.DebianArch()

    docker_block = sdk2.parameters.Info("Docker")
    docker_image_repository = build_parameters.DockerImageRepository()
    docker_save_image = build_parameters.DockerSaveImage()
    docker_push_image = build_parameters.DockerPushImage()
    docker_registry = build_parameters.DockerRegistry()
    docker_user = build_parameters.DockerUser()
    docker_token_vault_name = build_parameters.DockerTokenVaultName()
    docker_yav_token = build_parameters.DockerYavToken()
    docker_build_network = build_parameters.DockerBuildNetwork()
    docker_build_arg = build_parameters.DockerBuildArg("Docker build args", description="Docker build args (dict)")

    tests_block = sdk2.parameters.Info("Tests")
    run_tests = build_parameters.RunTests()
    run_medium_tests = build_parameters.RunMediumTests()
    run_long_tests = build_parameters.RunLongTests()
    ignore_fail_tests = build_parameters.IgnoreFailTests()
    create_build_output_resource = build_parameters.CreateBuildOutputResource()
    cache_tests = build_parameters.CacheTests()

    test_threads = build_parameters.TestThreads()
    test_params = build_parameters.TestCustomParameters()
    test_filters = build_parameters.TestFilters()
    test_size_filter = build_parameters.TestSizeFilter()
    test_tag = build_parameters.TestTag()
    test_type_filter = build_parameters.TestTypeFilter()
    test_log_level = build_parameters.TestLogLevel()
    tests_retries = build_parameters.TestsRetriesCount()

    resources_block = sdk2.parameters.Info("Resources")
    resource_type = build_parameters.ResourceType()
    resource_id = build_parameters.ParentResourceId()
    package_ttl = build_parameters.PackageResourceTtl(default=30)
    build_logs_ttl = build_parameters.BuildLogsTtl(default=4)
    save_build_output = build_parameters.SaveBuildOutput()
    build_output_ttl = build_parameters.BuildOutputTtl(default=4)

    publishing_block = sdk2.parameters.Info("Publishing")
    artifactory = build_parameters.ArtifactoryUpload()
    artifactory_password = build_parameters.ArtifactoryPassword()
    changelog = build_parameters.Changelog()
    publish_package = build_parameters.PublishPackage()
    key_user = build_parameters.KeyUser()
    multiple_publish = build_parameters.MultiplePublish()
    publish_to = build_parameters.PublishTo()
    multiple_publish_to = build_parameters.MultiplePublishTo()
    publish_to_mapping = build_parameters.MultiplePublishMapping(
        "Multiple publish mapping",
        description=build_parameters.MultiplePublishMapping.description
    )
    ensure_package_published = build_parameters.EnsurePackagePublished()

    # Wheel repo
    wheel_upload_repo = build_parameters.WheelUploadRepo()
    wheel_access_key_token = build_parameters.WheelAccessKeyTokenVaultName()
    wheel_secret_key_token = build_parameters.WheelSecretKeyTokenVaultName()

    # npm
    npm_registry = build_parameters.NpmRegistry()
    npm_login = build_parameters.NpmLogin()
    npm_password_vault_name = build_parameters.NpmPasswordVaultName()

    # Nanny release
    release_to_nanny = build_parameters.ReleaseToNanny()

    # Ya.Deploy release integration: https://wiki.yandex-team.ru/deploy/release-integration/
    yp_token_vault = release_integration.YpTokenVaultParameter2()
    release_to_ya_deploy = release_integration.ReleaseToYaDeployParameter2()
    release_per_resource = release_integration.ReleasePerResourceParameter2()

    advanced_block = sdk2.parameters.Info("Advanced")
    adhoc_packages = build_parameters.AdhocPackages("Adhoc packages", description="Adhoc packages (pure json)")
    use_ya_dev = build_parameters.UseYaDev()
    be_verbose = build_parameters.BeVerbose()
    use_new_format = build_parameters.UseNewFormat()
    checkout_mode = build_parameters.CheckoutModeParameter()
    checkout = build_parameters.CheckoutParameter()

    custom_version = build_parameters.CustomVersion()
    patch_package_version = PatchPackageVersion(label="Patch 'meta.version' in packages",
                                                description="Set 'meta.version' in package files to the value of " +
                                                            "'Custom version' field before build")
    with patch_package_version.value[True]:
        patch_package_version_files = PatchPackageVersionFiles(
            label="Patch 'meta.version' only in",
            description=(
                "Set 'meta.version' to the value of 'Custom version' "
                "field only in listed package files.\n"
                "Empty value causes all built package files to be patched."
            ))

    package_resource_description = build_parameters.PackageResourceDescription(
        "Package resource description",
        description=build_parameters.PackageResourceDescription.description,
    )
    package_resource_attrs = build_parameters.PackageResourceAttrs(
        "Package resource attributes",
        description=build_parameters.PackageResourceAttrs.description,
    )
    build_system = build_parameters.BuildSystem()
    distbuild_pool = build_parameters.DistbuildPool()
    ya_timeout = build_parameters.YaTimeout()
    sandbox_container = build_parameters.Container()

    yt_store_parameters = build_parameters.YtStoreParameters()
    yt_store_exclusive = build_parameters.YtStoreExclusive()


class YaPackage2(
    ya_make_package.BinaryMixin,
    binary_task.LastBinaryTaskRelease,
    nanny.ReleaseToNannyTask2,
    release_integration.ReleaseToYaDeployTask2,
    sdk2.Task,
):
    """
        Task for creating debian and tarball packages using 'ya package', sdk2-compatible.
        Packaging: https://docs.yandex-team.ru/devtools/package/
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 100 * 1024
        client_tags = ctc.Tag.LXC | ctc.Tag.ARCADIA_HG

    class Parameters(YaPackage2Parameters):
        ext_params = binary_task.binary_release_parameters(custom=True)

    class Context(sdk2.Task.Context):
        package_resources = None
        publish_to_list = None
        revision = None
        output_resource_version = None
        output_resources = {}

    def resource_types_iter(self):
        rt = ya_package_internal._parse_multi_parameter(self.Parameters.resource_type)
        logging.info("$$ Resource types {}".format(rt))
        return itertools.chain(rt, itertools.repeat(rt[-1]))

    def resource_ids_list(self):
        return [
            resource.id
            for resource in self.Parameters.resource_id
        ]

    def resource_ids_iter(self):
        resource_ids = self.resource_ids_list()
        logging.info("$$ Resource ids {}".format(resource_ids))
        return itertools.chain(resource_ids, itertools.repeat(None))

    def package_type_map(self):
        packages = ya_package_internal._parse_multi_parameter(self.Parameters.packages)
        logging.info("$$ Packages {}".format(packages))
        types = self.resource_types_iter()
        ids = self.resource_ids_iter()
        return zip(packages, types, ids)

    def _get_common_package_resource_attrs(self):
        arcadia_url = self.Parameters.checkout_arcadia_from_url
        do_not_remove = self.Parameters.do_not_remove_resources
        package_ttl = self.Parameters.package_ttl
        parsed_url = Arcadia.parse_url(arcadia_url)
        revision, branch, tag = parsed_url[1], parsed_url[2], parsed_url[3]
        build_type = self.Parameters.build_type

        return {
            "svn_path": arcadia_url,
            "svn_revision": self.Context.revision or revision or "HEAD",
            "build_type": build_type,
            "branch": branch or tag or "trunk",
            "platform": "unknown",
            "ttl": "inf" if do_not_remove else package_ttl,
        }

    def update_foreign_package_resource(self, path, resource, is_debug):
        attrs = self._get_common_package_resource_attrs()
        attrs.update(self.Parameters.package_resource_attrs or {})
        for attr_key, attr_value in attrs.items():
            setattr(resource, attr_key, attr_value)

    def init_package_resource(self, path, resource_type, is_debug):
        resource_filename = path.replace("/", ".")
        if is_debug:
            resource_filename += ".debug"
        if self.Parameters.package_resource_description:
            resource_description = self.Parameters.package_resource_description.get(path, path)
        else:
            resource_description = path
        attrs = self._get_common_package_resource_attrs()
        attrs.update(self.Parameters.package_resource_attrs or {})
        attrs["package_path"] = path
        return sdk2.Resource[resource_type](
            self,
            resource_description,
            resource_filename,
            **attrs
        )

    def on_enqueue(self):
        if self.Context.package_resources is not None:
            return

        if self.Parameters.package_type in [
            ya_package_consts.PackageType.DEBIAN.value,
            ya_package_consts.PackageType.RPM.value,
            ya_package_consts.PackageType.WHEEL.value,
            ya_package_consts.PackageType.AAR.value,
        ]:
            logging.info("self.platform = %s", platform.platform())

        self.set_sandbox_container()

        strip_binaries = self.Parameters.strip_binaries
        if not strip_binaries:
            self.Parameters.create_debug_packages = False

        create_dbg = self.Parameters.create_debug_packages
        is_tarball = self.Parameters.package_type == ya_package_consts.PackageType.TARBALL.value
        raw_package = self.Parameters.raw_package

        if raw_package and not is_tarball:
            eh.check_failed("You can get raw package only using 'tar' package type")

        self.set_context_package_resources(create_dbg, is_tarball)
        self.set_publish_context()

    def set_context_package_resources(self, create_dbg, is_tarball):
        packages = []
        resources = []

        for path, resource_type, resource_id in self.package_type_map():
            packages.append(path)
            if not self.Parameters.debian_dont_store_package:
                if resource_id is not None:
                    resource = sdk2.Resource[resource_id]
                    self.update_foreign_package_resource(path, resource, False)
                else:
                    resource = self.init_package_resource(path, resource_type, False)
                if create_dbg and is_tarball:
                    debug_resource = self.init_package_resource(path, resource_types.ARCADIA_PROJECT_SYMBOLS, True)
                else:
                    debug_resource = None
                resources.append((resource.id, debug_resource and debug_resource.id))

        self.Context.package_resources = {"packages": packages, "resources": resources}

    def set_publish_context(self):
        if self.Parameters.publish_to_mapping:
            self.Context.publish_to_list = self.Parameters.publish_to_mapping
        elif self.Parameters.multiple_publish:
            self.Context.publish_to_list = self.Parameters.multiple_publish_to.split(";")
        elif self.Parameters.publish_to:
            self.Context.publish_to_list = [self.Parameters.publish_to]
        elif (
            self.Parameters.wheel_upload_repo
            and self.Parameters.package_type == ya_package_consts.PackageType.WHEEL.value
        ):
            self.Context.publish_to_list = self.Parameters.wheel_upload_repo.split(";")
        elif self.Parameters.npm_registry and self.Parameters.package_type == ya_package_consts.PackageType.NPM.value:
            self.Context.publish_to_list = [self.Parameters.npm_registry]
        else:
            self.Context.publish_to_list = []

    def set_sandbox_container(self):
        if self.Parameters.package_type == ya_package_consts.PackageType.DOCKER.value:
            self.Requirements.dns = ctm.DnsType.DNS64
            if not self.Parameters.sandbox_container:
                self.Parameters.sandbox_container = 773239891  # default docker container id for YA_PACKAGE
        elif self.Parameters.package_type == ya_package_consts.PackageType.AAR.value:
            self.Requirements.dns = ctm.DnsType.DNS64
        elif self.Parameters.package_type == ya_package_consts.PackageType.DEBIAN.value:
            if platform.platform() == "linux_ubuntu_18.04_bionic" and not self.Parameters.sandbox_container:
                self.Parameters.sandbox_container = 855378432  # default docker container id bionic with dev tools
        elif self.Parameters.package_type == ya_package_consts.PackageType.RPM.value:
            if not self.Parameters.sandbox_container:
                self.Parameters.sandbox_container = 1116457063  # default docker container id xenial with rpmbuild
        elif self.Parameters.package_type == ya_package_consts.PackageType.WHEEL.value:
            if not self.Parameters.sandbox_container:
                # default docker container id bionic with setuptools wheel
                self.Parameters.sandbox_container = 2287800509
        elif self.Parameters.package_type == ya_package_consts.PackageType.NPM.value:
            self.Requirements.dns = ctm.DnsType.DNS64
            if not self.Parameters.sandbox_container:
                self.Parameters.sandbox_container = 1724812895  # default docker container id bionic with npm & node

    def _init_adhoc_packages(self):
        """
        We can pass package parameters as `adhoc_packages` input parameter
        instead of getting it from package.json file in Arcadia.
        """
        packages = []
        resources = []
        res_types = self.resource_types_iter()
        for i, package_json in enumerate(self.Parameters.adhoc_packages):
            if not package_json.strip():
                logging.warning("Empty adhoc package json")
                continue
            path = str(self.log_path("adhoc_package_{}.json".format(i)))
            packages.append(path)
            with open(path, "w") as pf:
                json.dump(json.loads(package_json), pf)

            resource_type = next(res_types)

            if not self.Parameters.debian_dont_store_package:
                create_dbg = self.Parameters.create_debug_packages
                is_tarball = self.Parameters.package_type == ya_package_consts.PackageType.TARBALL.value

                resource = self.init_package_resource(path, resource_type, False)
                if create_dbg and is_tarball:
                    debug_resource = self.init_package_resource(path, resource_types.ARCADIA_PROJECT_SYMBOLS, True)
                else:
                    debug_resource = None
                resources.append((resource.id, debug_resource and debug_resource.id))

        self.Context.package_resources["packages"] = packages
        self.Context.package_resources["resources"] = resources
        self.Context.save()
        return packages

    def populate_package_resource(self, name, resource_id, path, version, revision):
        prepare_foreign_resource = resource_id in self.resource_ids_list()
        resource = sdk2.Resource[resource_id]
        if prepare_foreign_resource:
            logging.warning("Skipping change resource basename because resource #%s is not mine", resource_id)
        else:
            # Need this in order to change path property, sandbox magic
            sdk2.ResourceData(resource)
            resource.path = os.path.basename(path)

        self.set_info("{}={}".format(name, version))
        self.Context.output_resource_version = version
        self.Context.output_resources[name] = version

        if revision:
            resource.svn_revision = revision

        resource.platform = self.Parameters.target_platform or platform.platform()
        resource.resource_name = name
        resource.resource_version = version
        build_type = resource.build_type
        branch = resource.branch

        description = ".".join([name, "linux", build_type, branch, revision])
        resource.description = description

        if prepare_foreign_resource:
            logging.info("Prepairing data for parent resource #%s", resource_id)
            source_path = os.path.abspath(path)
            target_path = str(sdk2.Path(common.config.Registry().client.tasks.data_dir).joinpath(
                *common.types.task.relpath(self.id)
            ).joinpath(resource.path))
            if source_path != target_path:
                logging.info("Source path is not equal to target path, moving %s to %s", source_path, target_path)
                shutil.move(source_path, target_path)

    def check_aapi_available(self):
        try:
            return sdk.wait_aapi_url(self.Parameters.checkout_arcadia_from_url)
        except aapi.ArcadiaApiCommandFailed as e:
            logging.warning("AAPI availability check failed: %r", e)
            return False

    def get_arcadia(self, use_fuse, checkout, copy_trunk=False, use_arc_instead_of_aapi=False):
        url = self.Parameters.checkout_arcadia_from_url

        if use_fuse:
            return sdk.mount_arc_path(url, use_arc_instead_of_aapi=use_arc_instead_of_aapi, fetch_all=False)
        elif checkout:
            return context_managers.nullcontext(sdk.do_clone(self.Parameters.checkout_arcadia_from_url, self))
        else:
            arcadia_src_dir = sdk2.svn.Arcadia.get_arcadia_src_dir(url, copy_trunk=copy_trunk)
            if not arcadia_src_dir:
                raise common.errors.TaskFailure("Cannot get repo for url {}".format(url))
            return context_managers.nullcontext(arcadia_src_dir)

    def get_test_data(
        self, arc_root, packages, test_data_prefix, use_arc_vcs, use_aapi_fuse, checkout, use_arc_instead_of_aapi=False
    ):
        arcadia_url = self.Parameters.checkout_arcadia_from_url

        if checkout or use_arc_vcs:
            return context_managers.nullcontext(None)

        paths = ya_package_internal.extract_test_data_paths(packages, arc_root, test_data_prefix)

        if not paths:
            return context_managers.nullcontext(None)

        elif use_aapi_fuse:
            url = os.path.normpath(Arcadia.append(arcadia_url, os.path.join("..", test_data_prefix)))
            if Arcadia.check(url):
                return sdk.mount_arc_path(url, use_arc_instead_of_aapi=use_arc_instead_of_aapi, fetch_all=False)
            else:
                return context_managers.nullcontext(None)

        else:
            path_url = None
            for path in paths:
                path_url = os.path.normpath(Arcadia.append(arcadia_url, os.path.join("..", path)))
                ArcadiaTestData.get_arcadia_test_data(self, path_url)
            return context_managers.nullcontext(ArcadiaTestData.test_data_location(path_url)[0])

    def on_release(self, additional_parameters):
        sdk2.Task.on_release(self, additional_parameters)
        if self.Parameters.package_type == ya_package_consts.PackageType.DOCKER.value:
            image_name, image_tag = self._parse_docker_version(self.Context.output_resource_version)
            release_payload = {
                "spec": {
                    "title": additional_parameters.get(nanny_const.RELEASE_SUBJECT_KEY),
                    "desc": additional_parameters.get(nanny_const.RELEASE_COMMENTS),
                    "component": additional_parameters.get(nanny_const.COMPONENT_KEY) or "",
                    "type": "DOCKER_RELEASE",
                    "docker_release": {
                        "image_name": image_name,
                        "image_tag": image_tag,
                        "release_type": additional_parameters["release_status"].upper()
                    }
                }
            }
            logging.info("Sending release of Docker task %s to Nanny", self.id)
            logging.info("Release payload: %s", json.dumps(release_payload, indent=4))
            result = self.nanny_client.create_release2(release_payload)
            logging.info("Release result is %s", result)
            if self.Parameters.release_to_ya_deploy:
                images = []
                for version in self.Context.output_resources.values():
                    image = release_integration.make_docker_image_from_path(version)
                    images.append(image)
                result = release_integration.create_docker_release(
                    task=self,
                    images=images,
                    additional_parameters=additional_parameters,
                )
                logging.info("Release result is %s", result)
                self.store_release_id_in_context(result["meta"]["id"])
        elif self.Parameters.package_type == ya_package_consts.PackageType.DEBIAN.value:
            key_user = self._get_key_user()
            publish_to = self.Context.publish_to_list

            if publish_to and isinstance(publish_to, list):
                publish_to = itertools.chain(iter(publish_to), itertools.repeat(publish_to[-1]))

                def _dupload():
                    resources = self.Context.package_resources["resources"]

                    for resource in resources:
                        resource_id, _ = ya_package_internal._get_resources_pair(resource)  # symbols not published
                        temp_dir = tempfile.mkdtemp()
                        with common.fs.WorkDir(temp_dir):
                            archive_path = sdk2.ResourceData(sdk2.Resource[resource_id]).path
                            sp.Popen("/bin/tar -xzf {}".format(archive_path)).wait()
                            sp.Popen("/usr/bin/dupload --to {}".format(next(publish_to))).wait()
                        shutil.rmtree(temp_dir)

                if (
                    not self.Parameters.publish_package
                    and self.Parameters.package_type == ya_package_consts.PackageType.DEBIAN.value
                ):
                    if key_user and self.Context.publish_to_list:
                        self._run_with_gpg_and_ssh(_dupload, key_user)

            if self.Parameters.release_to_nanny:
                nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        elif self.Parameters.package_type == ya_package_consts.PackageType.RPM.value:
            raise eh.check_failed("Releasing rpm package supported only for nanny")
        elif self.Parameters.package_type == ya_package_consts.PackageType.NPM.value:
            raise eh.check_failed("Releasing npm package supported only for nanny")
        else:
            nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
            if self.Parameters.release_to_ya_deploy:
                release_integration.ReleaseToYaDeployTask2.on_release(self, additional_parameters)

    def _parse_docker_version(self, version):
        image_name, _, image_tag = version.partition(":")
        _, _, image_name = image_name.partition("/")
        return image_name, image_tag

    def _get_key_user(self):
        key_user = self.Parameters.key_user

        if not key_user and self.Parameters.package_type == ya_package_consts.PackageType.DEBIAN.value:
            for user, owner in ya_package_consts.VAULT_OWNERS.items():
                if owner == self.owner:
                    logging.info("Task owner was taken as a vault owner: user %s, owner %s", user, owner)
                    key_user = user
                    break

        return key_user

    def _run_with_gpg_and_ssh(self, f, key_user):
        eh.ensure(key_user in ya_package_consts.VAULT_OWNERS, "Unknown keys owner, key_user = {}".format(key_user))

        vault_owner = ya_package_consts.VAULT_OWNERS[key_user]
        logging.info("Vault owner: %s", vault_owner)

        secret_key_name = "{}-gpg-private".format(key_user)
        logging.info("Secret GPG key name: %s", secret_key_name)

        public_key_name = "{}-gpg-public".format(key_user)
        logging.info("Public GPG key name: %s", public_key_name)

        ssh_key_name = "{}-ssh".format(key_user)
        logging.info("SSH key name: %s", ssh_key_name)

        os.environ["DEBEMAIL"] = "{}@yandex-team.ru".format(key_user)

        with gnupg.GpgKey2(vault_owner, secret_key_name, public_key_name):
            with debpkg.DebRelease(ya_package_consts.DUPLOAD_CONF, login=key_user):
                with ssh.Key(None, vault_owner, ssh_key_name):
                    f()

    def _get_docker_token(self):
        # If yav secret is specified, its value is prevalent
        yav_secret = self._get_parameter_or_context_value(
            build_parameters.DockerYavToken.name,
            None,
        )
        if yav_secret:
            logging.info(
                "Yav secret from param %r, %r",
                yav_secret,
                getattr(yav_secret, 'default_key', 'NO_ATTRIBUTE'),
            )
            return yav_secret.value()

        # Next try owned token from Vault
        token_vault_name = self.Parameters.docker_token_vault_name
        try:
            token = sdk2.Vault.data(self.owner, token_vault_name)
        except common.errors.VaultNotFound:
            if token_vault_name is None:
                raise

            # Last try: get token 'shared with' by name.
            token = sdk2.Vault.data(token_vault_name)

        return token

    def _docker_login(self):
        token = self._get_docker_token()
        sp.Popen(
            "docker login {registry} -u {user} -p $DOCKER_TOKEN".format(
                registry=self.Parameters.docker_registry,
                user=self.Parameters.docker_user,
            ),
            shell=True,
            env={"DOCKER_TOKEN": token},
        ).wait()

    def _npm_login(self):
        password_vault_name = self.Parameters.npm_password_vault_name
        password = sdk2.Vault.data(self.owner, password_vault_name)
        login = self.Parameters.npm_login
        registry = self.Parameters.npm_registry

        self._write_npmrc(registry, login, password)

    def _write_npmrc(self, registry, login, password):
        registry = urlparse.urlparse(registry).netloc or registry
        password = base64.b64encode(password)
        email = "{}@yandex-team.ru".format(login)

        params = dict(
            registry=registry,
            login=login,
            password=password,
            email=email
        )

        content = "\n".join(map(lambda line: line.format(**params), [
            '//{registry}/:_password="{password}"',
            '//{registry}/:username={login}',
            '//{registry}/:email={email}',
            '//{registry}/:always-auth=false',
        ]))
        home = os.environ["HOME"]
        npmrc_path = os.path.join(home, ".npmrc")
        with open(npmrc_path, 'w') as afile:
            afile.write(content)

    def _setup_wheel_repo_keys(self):
        wheel_access_key_tf = None
        wheel_secret_key_tf = None

        access_key_vault_name = self.Parameters.wheel_access_key_token
        if access_key_vault_name:
            key = sdk2.Vault.data(self.owner, access_key_vault_name)
            wheel_access_key_tf = tempfile.NamedTemporaryFile(prefix="access_key")
            wheel_access_key_tf.write(key)
            wheel_access_key_tf.flush()

        secret_key_vault_name = self.Parameters.wheel_secret_key_token
        if secret_key_vault_name:
            key = sdk2.Vault.data(self.owner, secret_key_vault_name)
            wheel_secret_key_tf = tempfile.NamedTemporaryFile(prefix="secret_key")
            wheel_secret_key_tf.write(key)
            wheel_secret_key_tf.flush()

        return wheel_access_key_tf, wheel_secret_key_tf

    # Used by MAPS_DOCKER task
    def prebuild_hook(self):
        pass

    # TODO: вынести методы ниже в отдельную либу (get_env_vars, deref - общие с YaMake2)

    def should_checkout(self):
        checkout_auto = self.Parameters.checkout_mode == consts.CHECKOUT_MODE_AUTO
        if checkout_auto:
            parsed_url = Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url)
            checkout = not parsed_url.trunk and not self.Parameters.arcadia_patch
        else:
            checkout = self.Parameters.checkout
        logging.info("Use selective checkout: %s", checkout)
        return checkout

    def deref(self, s):
        def deref_vault(match):
            secret = sdk2.Vault.data(match.group("owner"), match.group("name"))

            if match.group("dst") == "file":
                deref_path = tempfile.NamedTemporaryFile().name
                fu.write_file(deref_path, secret)
                return deref_path

            return secret

        s = ya_package_consts.VAULT_PATTERN.sub(deref_vault, s)

        return s

    def get_env_vars(self):
        env_vars = self.Parameters.env_vars
        if env_vars and not ya_package_consts.ENV_VAR_PATTERN.match(env_vars):
            eh.check_failed("Incorrect 'Environment variables' parameter '{}'".format(env_vars))

        env_vars = {k: self.deref(v) for k, v in (x.split("=", 1) for x in shlex.split(env_vars))}

        if "GSID" not in env_vars.keys() and "GSID" in os.environ.keys():
            env_vars["GSID"] = os.getenv("GSID")

        return env_vars

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        xcode_version = getattr(self.Parameters, consts.XCODE_VERSION, None)
        if xcode_version:
            logging.debug('Xcode version: {}'.format(xcode_version))
            Xcode.prepare_xcode(xcode_version)

        logging.info("Process on_execute")
        parsed_url = Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url)
        is_hg_repo = (urlparse.urlparse(self.Parameters.checkout_arcadia_from_url).scheme == Arcadia.ARCADIA_HG_SCHEME)
        is_arc_repo = (
            urlparse.urlparse(self.Parameters.checkout_arcadia_from_url).scheme == Arcadia.ARCADIA_ARC_SCHEME
        )

        if not is_hg_repo and not is_arc_repo and not parsed_url.trunk and not parsed_url.branch and not parsed_url.tag:
            logging.error("Cannot determine branch from %s: %s", self.Parameters.checkout_arcadia_from_url, parsed_url)
            eh.check_failed(
                "Cannot determine branch from {}, make sure it ends with '/arcadia'".format(
                    self.Parameters.checkout_arcadia_from_url
                )
            )

        if list(filter(None, self.Parameters.adhoc_packages or [])):
            self._init_adhoc_packages()

        use_aapi_fuse = self.Parameters.use_aapi_fuse
        use_arc_instead_of_aapi = self.Parameters.use_arc_instead_of_aapi
        aapi_fallback = self.Parameters.aapi_fallback
        copy_trunk = self.Parameters.copy_trunk
        use_arc_vcs = is_arc_repo
        use_fuse = use_aapi_fuse or use_arc_vcs

        if use_aapi_fuse and aapi_fallback and not use_arc_vcs and not use_arc_instead_of_aapi:
            if not self.check_aapi_available():
                use_aapi_fuse = False
                logging.info("Disable AAPI, because service is temporary unavailable or inapplicable")

        if use_fuse and not sdk.fuse_available():
            eh.check_failed("Fuse is not available on {} yet".format(platform.platform()))

        if (
            self.Parameters.package_type == ya_package_consts.PackageType.DOCKER.value
            and not self.Parameters.docker_push_image
        ):
            self.set_info(
                "<strong>Docker image will not be pushed due to 'Push docker image' is set to False</strong>"
            )

        logging.info("Mount = %s (using %s)", use_fuse, "Arc VCS" if use_arc_vcs else "AAPI")

        checkout = False if use_fuse else self.should_checkout()

        packages = self.Context.package_resources["packages"]
        eh.ensure(packages, "Packages are not specified")
        logging.info("Packages: %s", packages)

        wheel_access_key_tf = None
        wheel_secret_key_tf = None
        if (
            self.Parameters.package_type == ya_package_consts.PackageType.WHEEL.value
            and self.Parameters.publish_package
        ):
            wheel_access_key_tf, wheel_secret_key_tf = self._setup_wheel_repo_keys()

        if self.Parameters.package_type == ya_package_consts.PackageType.DOCKER.value:
            self._docker_login()

        if self.Parameters.package_type == ya_package_consts.PackageType.NPM.value:
            self._npm_login()

        with self.get_arcadia(use_fuse, checkout, copy_trunk, use_arc_instead_of_aapi) as arcadia_src_dir:
            self.arcadia_src_dir = arcadia_src_dir
            self.prebuild_hook()
            changelog = self.Parameters.changelog
            if changelog and os.path.exists(os.path.join(self.arcadia_src_dir, changelog)):
                changelog = os.path.join(self.arcadia_src_dir, changelog)
                logging.info("Change log: %s", changelog)
                changelog_is_path = True
            else:
                changelog_is_path = False

            if self.Parameters.arcadia_patch:
                if checkout:
                    eh.check_failed(
                        "Patch can be applied only on full arcadia for now. Disable parameter '{}'".format(
                            parameters.CheckoutParameter.description
                        )
                    )
                patch_path = sdk.apply_patch(
                    self,
                    self.arcadia_src_dir,
                    self.Parameters.arcadia_patch,
                    self.path()
                )
                if patch_path and not changelog_is_path:
                    changelog += "Package was created with patch {}".format(
                        string.to_utf8(self.Parameters.arcadia_patch)
                    )

            with self.get_test_data(
                arcadia_src_dir,
                packages,
                "arcadia_tests_data",
                use_arc_vcs,
                use_aapi_fuse,
                checkout,
                use_arc_instead_of_aapi,
            ) as arcadia_tests_data_dir, self.get_test_data(
                arcadia_src_dir,
                packages,
                "data",
                use_arc_vcs,
                use_aapi_fuse,
                checkout,
                use_arc_instead_of_aapi,
            ) as data_dir:
                if use_arc_vcs:
                    revision = str(sdk.mounted_path_svnversion(self.arcadia_src_dir, True)["revision"])

                elif use_aapi_fuse:
                    if is_hg_repo:
                        revision = str(sdk.mounted_path_svnversion(self.arcadia_src_dir)["hash"])
                    else:
                        revision = str(sdk.mounted_path_svnversion(self.arcadia_src_dir)["revision"])

                else:
                    if is_hg_repo:
                        revision = Arcadia.get_revision(self.Parameters.checkout_arcadia_from_url)
                    else:
                        revision = Arcadia.get_revision(self.arcadia_src_dir)

                logging.info("Arcadia path: {}, at rev {}".format(self.arcadia_src_dir, revision))
                self.Context.revision = revision
                self.Context.save()
                resources = iter(self.Context.package_resources["resources"])

                # XXX For more info see https://st.yandex-team.ru/DEVTOOLSSUPPORT-13736
                enable_direct_arcadia_mode = self.get_env_vars().get("ENABLE_DEVTOOLSSUPPORT_13736") == "yes"

                if enable_direct_arcadia_mode:
                    arcadia_root = self.arcadia_src_dir
                else:
                    fake_svn_root = os.path.join(os.getcwd(), "fake-svn-root")
                    os.mkdir(fake_svn_root)
                    logging.info("Fake svn root: %s", fake_svn_root)

                    # TODO: Remove it when ya package will not take --source-root as a directory containing "arcadia"
                    arcadia_src_symlink_path = os.path.join(fake_svn_root, "arcadia")
                    os.symlink(self.arcadia_src_dir, arcadia_src_symlink_path)
                    logging.info("Symlink created %s -> %s", arcadia_src_symlink_path, self.arcadia_src_dir)

                    # TODO: Remove arcadia_tests_data and data from packages and remove this checkout
                    if arcadia_tests_data_dir:
                        atd_symlink_path = os.path.join(fake_svn_root, "arcadia_tests_data")
                        os.symlink(arcadia_tests_data_dir, atd_symlink_path)
                        logging.info("Symlink created %s -> %s", atd_symlink_path, arcadia_tests_data_dir)
                    if data_dir:
                        data_symlink_path = os.path.join(fake_svn_root, "data")
                        os.symlink(data_dir, data_symlink_path)
                        logging.info("Symlink created %s -> %s", data_symlink_path, data_dir)

                    arcadia_root = fake_svn_root

                key_user = self._get_key_user()

                publish_to = self.Context.publish_to_list
                if publish_to and isinstance(publish_to, list):
                    # otherwise it is a publish mapping and should be transferred as is
                    if self.Parameters.package_type == ya_package_consts.PackageType.AAR.value:
                        publish_to = [
                            os.path.join(arcadia_src_symlink_path, settings_xml) for settings_xml in publish_to
                        ]
                    publish_to = ";".join(publish_to)

                build_system = self.Parameters.build_system

                debian_compression_level = None
                if self.Parameters.debian_compression_level:
                    debian_compression_level = self.Parameters.debian_compression_level

                debian_compression_type = None
                if self.Parameters.debian_compression_type:
                    debian_compression_type = self.Parameters.debian_compression_type

                debian_store_package = not self.Parameters.debian_dont_store_package

                yt_store_params = ya_make_package.prepare_yt_store_params(self)

                def _do_package():
                    env = self.get_env_vars()
                    if env:
                        with ya_package_internal.custom_env(env):
                            return _do_sdk_package()
                    else:
                        return _do_sdk_package()

                def _do_sdk_package():
                    sdk.do_package(
                        arcadia_root,
                        packages,
                        use_ya_dev=self.Parameters.use_ya_dev,
                        be_verbose=self.Parameters.be_verbose,
                        artifactory=self.Parameters.artifactory,
                        artifactory_password=self.Parameters.artifactory_password,
                        publish=self.Parameters.publish_package,
                        sloppy_deb=self.Parameters.sloppy_debian,
                        publish_to=publish_to,
                        key=key_user,
                        changelog=changelog,
                        run_tests=1 if self.Parameters.run_tests else 0,
                        run_long_tests=self.Parameters.run_long_tests,
                        run_medium_tests=self.Parameters.run_medium_tests,
                        ignore_fail_tests=self.Parameters.ignore_fail_tests,
                        sandbox_task_id=self.id,
                        debian=(self.Parameters.package_type == ya_package_consts.PackageType.DEBIAN.value),
                        build_type=self.Parameters.build_type,
                        strip_binaries=self.Parameters.strip_binaries,
                        full_strip_binaries=self.Parameters.full_strip_binaries,
                        create_dbg=self.Parameters.create_debug_packages,
                        compress_archive=self.Parameters.compress_package_archive,
                        compression_filter=self.Parameters.compression_filter,
                        compression_level=self.Parameters.compression_level,
                        uc_codec=self.Parameters.uc_codec,
                        clear_build=self.Parameters.clear_build,
                        arch_all=self.Parameters.architecture_all,
                        force_dupload=self.Parameters.force_dupload,
                        dupload_max_attempts=int(self.Parameters.dupload_max_attempts),
                        build_debian_scripts=self.Parameters.build_debian_scripts,
                        debian_distribution=self.Parameters.debian_distribution,
                        convert=not self.Parameters.use_new_format,
                        host_platform=self.Parameters.host_platform,
                        target_platform=self.Parameters.target_platform,
                        sanitize=self.Parameters.sanitize,
                        checkout=checkout,
                        html_display=os.path.join(str(self.log_path()), "output.html"),
                        custom_version=self.Parameters.custom_version,
                        build_system=build_system,
                        distbuild_pool=self.Parameters.distbuild_pool,
                        build_logs_ttl=self.Parameters.build_logs_ttl,
                        build_output_ttl=self.Parameters.build_output_ttl,
                        debian_compression_level=debian_compression_level,
                        debian_store_package=debian_store_package,
                        musl=self.Parameters.musl,
                        debian_compression_type=debian_compression_type,
                        lto=self.Parameters.lto,
                        thinlto=self.Parameters.thinlto,
                        timeout=int(self.Parameters.ya_timeout or 10800),
                        semi_clear_build=self.Parameters.semi_clear_build,
                        raw_package=self.Parameters.raw_package,
                        raw_package_path=self.Parameters.raw_package_path,
                        yt_store_params=yt_store_params,
                        force_build_depends=self.Parameters.force_build_depends,
                        force_vcs_info_update=self.Parameters.force_vcs_info_update,
                        ignore_recurses=self.Parameters.ignore_recurses,
                        build_docker=(self.Parameters.package_type == ya_package_consts.PackageType.DOCKER.value),
                        docker_image_repository=self.Parameters.docker_image_repository,
                        docker_save_image=self.Parameters.docker_save_image,
                        docker_push_image=self.Parameters.docker_push_image,
                        docker_registry=self.Parameters.docker_registry,
                        docker_build_network=self.Parameters.docker_build_network,
                        docker_build_arg=self.Parameters.docker_build_arg,
                        rpm=(self.Parameters.package_type == ya_package_consts.PackageType.RPM.value),
                        wheel=(self.Parameters.package_type == ya_package_consts.PackageType.WHEEL.value),
                        wheel_access_key_path=wheel_access_key_tf.name if wheel_access_key_tf else None,
                        wheel_secret_key_path=wheel_secret_key_tf.name if wheel_secret_key_tf else None,
                        overwrite_read_only_files=self.Parameters.overwrite_read_only_files,
                        ensure_package_published=self.Parameters.ensure_package_published,
                        aar=(self.Parameters.package_type == ya_package_consts.PackageType.AAR.value),
                        use_prebuilt_tools=self.Parameters.use_prebuilt_tools,
                        wheel_python3=self.Parameters.wheel_python3,
                        npm=(self.Parameters.package_type == ya_package_consts.PackageType.NPM.value),
                        create_build_output_resource=self.Parameters.create_build_output_resource,
                        debian_arch=self.Parameters.debian_arch,
                        cache_tests=self.Parameters.cache_tests,
                        test_threads=self.Parameters.test_threads,
                        test_params=self.Parameters.test_params,
                        test_filters=self.Parameters.test_filters,
                        test_size_filter=self.Parameters.test_size_filter,
                        test_tag=self.Parameters.test_tag,
                        test_type_filter=self.Parameters.test_type_filter,
                        test_log_level=self.Parameters.test_log_level,
                        tests_retries=self.Parameters.tests_retries,
                        patch_pkg_json_version=self.Parameters.patch_package_version,
                        patch_pkg_json_version_files=self.Parameters.patch_package_version_files,
                        direct_arcadia_mode=enable_direct_arcadia_mode,
                    )

                try:
                    if key_user:
                        self._run_with_gpg_and_ssh(_do_package, key_user)
                    else:
                        eh.ensure(
                            not self.Parameters.publish_to and
                            not self.Parameters.multiple_publish_to and
                            not self.Parameters.publish_to_mapping,
                            "You can not publish package without user key selected"
                        )
                        _do_package()
                except Exception:
                    output_html = os.path.join(str(self.log_path()), "output.html")
                    if os.path.exists(output_html):
                        self.set_info(
                            "See human-readable build errors here: "
                            "<a href='{}/output.html'>output.html</a>".format(self.log_resource.http_proxy),
                            do_escape=False
                        )
                    raise
                finally:
                    junit_report_file_mask = "junit*.xml"
                    for junit_report_file in glob.glob(junit_report_file_mask):
                        if os.path.exists(junit_report_file):
                            shutil.copy(
                                junit_report_file,
                                os.path.join(str(self.log_path()), os.path.basename(junit_report_file)),
                            )
                            sdk2.Resource[resource_types.JUNIT_REPORT](
                                self,
                                "JUnit report",
                                junit_report_file,
                            )

                    if self.Parameters.save_build_output and os.path.exists("build") and os.listdir("build"):
                        sdk2.Resource[resource_types.BUILD_OUTPUT](
                            self,
                            "Build output",
                            "build",
                        )

                    for fname in glob.glob("*.json"):
                        if fname != "packages.json":
                            shutil.copy(fname, os.path.join(str(self.log_path()), fname))

                if os.path.exists(ya_package_consts.PACKAGES_FILE):
                    with open(ya_package_consts.PACKAGES_FILE) as packages_file:
                        built_packages = json.load(packages_file)

                        logging.debug("packages.json: %s", json.dumps(built_packages, indent=4, sort_keys=True))

                        def populate(resource_id, filename):
                            if "docker_image" in package:
                                is_docker = True
                                version = package["docker_image"]
                            else:
                                is_docker = False
                                version = package["version"]
                            self.populate_package_resource(
                                package["name"],
                                resource_id,
                                filename,
                                version,
                                revision,
                            )
                            if is_docker:
                                dgst = package.get("digest")
                                if dgst:
                                    resource = sdk2.Resource[resource_id]
                                    resource.docker_image_digest = dgst

                        if isinstance(built_packages, list):
                            package_jsons = iter(packages)
                            for package in built_packages:
                                resource_id, dbg_resource_id = ya_package_internal._get_resources_pair(next(resources))
                                populate(resource_id, package["path"])

                                if dbg_resource_id:
                                    populate(dbg_resource_id, package["debug_path"])

                                package_json = next(package_jsons)
                                PLATFORM_RUN = "platform.run"
                                platform_run_path = os.path.join(
                                    self.arcadia_src_dir, os.path.dirname(package_json), PLATFORM_RUN
                                )
                                logging.debug(
                                    "os.path.exists(%s): %s", platform_run_path, os.path.exists(platform_run_path)
                                )
                                if os.path.exists(platform_run_path):
                                    # copy to an unique local dir so that it can be shared as a resource
                                    package_json_local_dir = os.path.dirname(package_json)
                                    os.makedirs(package_json_local_dir)
                                    shutil.copy(platform_run_path, package_json_local_dir)
                                    sdk2.Resource[resource_types.PLATFORM_RUN](
                                        self,
                                        "platform.run for {}".format(package_json),
                                        os.path.join(package_json_local_dir, PLATFORM_RUN),
                                        **self._get_common_package_resource_attrs()
                                    )

                        else:
                            logging.info("Unordered version of packages.json, cannot match to pre-created resources")

                            for name, info in packages.items():
                                if "." in name:
                                    name = name.split(".")[0]

                                resource = next(resources)
                                if isinstance(resource, (tuple, list)):
                                    resource = resource[0]

                                self.populate_package_resource(
                                    name,
                                    resource,
                                    info["package"],
                                    info["version"],
                                    revision
                                )
