# coding: utf-8

import json
import logging
import os
import shutil
import six

from sandbox import common
from sandbox.common import mds
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt

from sandbox import sdk2

from sandbox.projects import resource_types
from sandbox.projects.common import constants


def get_target_platforms(platforms):
    if not platforms:
        return None
    return platforms.split() if isinstance(platforms, (six.text_type, six.binary_type)) else platforms


class BuildForAllTask2(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

    class Parameters(sdk2.Parameters):
        build_version = sdk2.parameters.String("Build version")
        ttl = sdk2.parameters.Integer("TTL for resources", default=30)

        with sdk2.parameters.Group("Platforms") as platforms_block:
            platforms_list = sdk2.parameters.CheckGroup(
                "List of target platforms",
                choices=[("Linux", "linux"), ("Darwin", "darwin")],
                default=None
            )

            linux_platforms_list = sdk2.parameters.String(
                "Choose linux platform",
                choices=[
                    (constants.SANDBOX_LINUX_UBUNTU_10_LUCID, constants.SANDBOX_LINUX_UBUNTU_10_LUCID),
                    (constants.SANDBOX_LINUX_UBUNTU_12_PRECISE, constants.SANDBOX_LINUX_UBUNTU_12_PRECISE),
                    (constants.SANDBOX_LINUX_UBUNTU_14_TRUSTY, constants.SANDBOX_LINUX_UBUNTU_14_TRUSTY),
                    (constants.SANDBOX_LINUX_UBUNTU_16_XENIAL, constants.SANDBOX_LINUX_UBUNTU_16_XENIAL),
                    (constants.SANDBOX_LINUX_ANY, constants.SANDBOX_LINUX_ANY),
                ],
                default=constants.SANDBOX_LINUX_UBUNTU_12_PRECISE
            )

        subtask_arguments = sdk2.parameters.Dict("Additional parameters to pass into the substask")

        backup_to_mds = sdk2.parameters.Bool("Backup to MDS", default=True)
        mds_namespace = sdk2.parameters.String("MDS namespace")
        mds_token_vault = sdk2.parameters.Vault("MDS token vault")
        mds_download_url = sdk2.parameters.String("MDS download url base")

    TARGET_TO_HOST_MAP = {
        "linux": constants.SANDBOX_LINUX_OLD_VERSION,
        "darwin": constants.SANDBOX_DARWIN_OLD_VERSION,
    }
    BIND_OSX_HOST = False

    def get_host_platform_for_build(self, target_platform):
        if target_platform not in self.TARGET_TO_HOST_MAP:
            raise common.errors.TaskFailure("Target platform {!r} is unavailable.".format(target_platform))
        if target_platform == "linux" and self.Parameters.linux_platforms_list:
            return self.Parameters.linux_platforms_list
        return self.TARGET_TO_HOST_MAP[target_platform]

    def run_subtasks(self, task_type, description, target_platforms, build_version, subtask_args):
        logging.debug("target_platforms={}".format(target_platforms))

        common_params = dict(list(self.Parameters))
        common_params.pop("mds_token_vault")

        common_params["build_version"] = build_version
        common_params.update(subtask_args)

        logging.info("Common parameters: %r", common_params)

        subtasks = []

        for target_platform in target_platforms:
            params = dict(common_params)
            host_name = None

            target_platform_alias = self.get_host_platform_for_build(target_platform)
            params["target_platform_alias"] = target_platform_alias
            logging.debug("Task arch is %s", target_platform_alias)

            # TODO: temporary hack for darwin, DEVTOOLS-3708
            if target_platform_alias == constants.SANDBOX_DARWIN_OLD_VERSION and self.BIND_OSX_HOST:
                host_name = "ymake-dev-mac"
                params[constants.USE_SANDBOX_GCC] = False

            task_class = sdk2.Task[task_type.type]
            task = task_class(
                self,
                description=description.format(target=target_platform_alias, version=build_version),
                **params
            )
            task.Requirements.host = host_name

            task.save().enqueue()
            subtasks.append(task)

        raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH, wait_all=True)

    def save_resource_to_mds(self, path, resource):
        mds_namespace = self.Parameters.mds_namespace
        mds_token = sdk2.Vault.data(self.Parameters.mds_token_vault)

        mds_key = mds.MDS().upload(
            path=path,
            mds_name=os.path.basename(path) + "." + str(resource.id),
            token=mds_token,
            namespace=mds_namespace,
        )
        logging.debug("Resource %s uploaded to MDS, key is %s", resource.id, mds_key)

        if not mds_namespace:
            mds_url = common.config.Registry().client.mds.dl.url + "/" + mds_key
        else:
            mds_url = self.Parameters.mds_download_url + "/" + mds_key

        logging.debug("Resource %s backed up to MDS %s", resource.id, mds_url)

        return mds_url

    def save_resources(self, resource_type, build_version, description):
        by_platform = {"data": {}}

        task_class = sdk2.Task[self.SUBTASK_TYPE.type]
        for task in self.find(task_class):
            if task.status == ctt.Status.FAILURE:
                raise common.errors.TaskFailure("Build in worker {id} failed".format(id=task.id))
            if resource_type is None:
                continue

            required_resource = resource_type.find(task=task).first()

            if required_resource is None:
                continue

            alias = task.Parameters.target_platform_alias

            required_resource_data = sdk2.ResourceData(required_resource)
            src_path = str(required_resource_data.path)

            dst_dir = alias
            dst_path = os.path.join(dst_dir, required_resource.path.name)

            if not os.path.exists(dst_dir):
                os.makedirs(dst_dir)
            if os.path.isdir(src_path):
                shutil.copytree(src_path, dst_dir)
            else:
                shutil.copy(src_path, dst_dir)

            attributes = dict(iter(required_resource))
            if build_version:
                attributes["version"] = build_version
            attributes["ttl"] = self.Parameters.ttl

            resulting_resource = resource_type(
                self,
                description.format(target=alias, version=build_version),
                dst_path,
                required_resource.arch,
                **attributes
            )

            sdk2.ResourceData(resulting_resource).ready()

            by_platform["data"][alias] = {
                "url": resulting_resource.http_proxy,
                "md5": resulting_resource.md5,
                "urls": [resulting_resource.http_proxy],
            }

            if self.Parameters.backup_to_mds:
                logging.debug("Going to back up resource %s to MDS", resulting_resource.id)
                mds_url = self.save_resource_to_mds(dst_path, resulting_resource)
                by_platform["data"][alias]["urls"].append(mds_url)

        return by_platform

    def save_platform_mapping(self, by_platform):
        by_platform_file = "by_platform.json"
        with open(by_platform_file, "w") as f:
            json.dump(by_platform, f, indent=4)

        attributes = {"ttl": self.Parameters.ttl}

        mapping_resource = resource_types.PLATFORM_MAPPING(
            self,
            "",
            by_platform_file,
            **attributes
        )

        if self.Parameters.backup_to_mds:
            logging.debug("Going to back up resource %s to MDS", mapping_resource.id)
            mds_url = self.save_resource_to_mds(by_platform_file, mapping_resource)
            mapping_resource.mds_link = mds_url

        sdk2.ResourceData(mapping_resource).ready()

    def build_and_save_resources(self, build_version, subtask_args):
        subtask_description = "Building {}".format(self.RESOURCE_DESCRIPTION) + " {version} for {target}"

        target_platforms = get_target_platforms(self.Parameters.platforms_list)

        if self.Context.build_subtasks_created == ctm.NotExists:
            self.Context.build_subtasks_created = True

            if target_platforms:
                self.run_subtasks(self.SUBTASK_TYPE, subtask_description, target_platforms, build_version, subtask_args)

        resource_description = self.RESOURCE_DESCRIPTION + " {version} for {target}"
        self.save_platform_mapping(self.save_resources(self.RESOURCE_TYPE, build_version, resource_description))

    def on_execute(self):
        self.build_and_save_resources(self.Parameters.build_version, self.Parameters.subtask_arguments)
