# coding: utf-8
import sandbox.sdk2 as sdk2

from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import constants
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.sdk_compat import task_helper as th
from sandbox.projects import resource_types

_DEPS_RESOURCE_NAME = 'deps.txt'
_DEPS_COUNT_KEY = 'deps_count_all'

BANNED_DEPS_KEY = 'banned_dependencies'
WHITELISTED_DEPS_KEY = 'whitelisted_dependencies'
CHECK_DEPS_KEY = 'check_dependencies'
BAN_UT_KEY = 'ban_ut_dependencies'


class BannedDependencies(sdk2.parameters.String):
    name = BANNED_DEPS_KEY
    description = 'Fail build if it requires specified paths (comma-separated)'
    default_value = ''
    group = "Dependency banner"


class WhitelistedDependencies(sdk2.parameters.String):
    name = WHITELISTED_DEPS_KEY
    description = "Ignore banned_dependencies for specified sub-paths (comma-separated)"
    default_value = ''
    group = "Dependency banner"


class CheckDependencies(sdk2.parameters.Bool):
    name = CHECK_DEPS_KEY
    description = 'Check build dependencies'
    default_value = False
    sub_fields = {
        'true': [BANNED_DEPS_KEY]
    }
    group = "Dependency banner"


class BanUtDependencies(sdk2.parameters.Bool):
    name = BAN_UT_KEY
    description = 'Ban unit tests as dependencies (SEARCH-2944)'
    default_value = False
    group = "Dependency banner"


def create_resources(task, output_file_name=_DEPS_RESOURCE_NAME, description=None):
    resource_description = 'Dependencies list'
    if description:
        resource_description += " ({})".format(description)
    return task.create_resource(resource_description, output_file_name, resource_types.DIR_LIST)


def slashify(dep_list):
    """
        Ensure that every dependency ends with slash to prevent prefix clashes. Also ignore empty entries.
        >>> slashify(['/usr', '/bin/', '', '~/data'])
        ['/usr/', '/bin/', '~/data/']
    """
    return [(d + '/').replace('//', '/') for d in filter(None, dep_list)]


def check_dependencies(task, arcadia_src_dir, targets_dirs, output_file_path=_DEPS_RESOURCE_NAME):
    """
        Build a list of dependencies for given targets
        Fail task if some of them are in ban list.

        Details, rationale: SEARCH-1578

        @param arcadia_src_dir: path to arcadia source root where to calculate deps
        @param target_dirs: list of targets' dirs
        @param output_file_path: full output path to put dependencies list in
    """
    fu.write_file(output_file_path, 'No dependencies')
    th.ctx_field_set(task, _DEPS_COUNT_KEY, 0)
    disallowed_deps = []
    try:
        deps = sdk.dump_targets_deps(arcadia_src_dir, targets_dirs)
        fu.write_lines(output_file_path, deps)
        th.ctx_field_set(task, _DEPS_COUNT_KEY, len(deps))
        banned_dependencies = th.input_or_ctx_field(task, BANNED_DEPS_KEY, '').split(',')
        whitelisted_dependencies = th.input_or_ctx_field(task, WHITELISTED_DEPS_KEY, '').split(',')

        ban_ut = th.input_or_ctx_field(task, BanUtDependencies.name, BanUtDependencies.default_value)

        banned_dependencies_slashed = slashify(banned_dependencies)
        whitelisted_dependencies_slashed = slashify(whitelisted_dependencies)
        deps_slashed = slashify(deps)

        if not banned_dependencies_slashed:
            return

        for dep in deps_slashed:

            if ban_ut and dep.endswith('/ut/'):
                # SEARCH-2944
                disallowed_deps.append(dep)
                continue

            for bd in banned_dependencies_slashed:
                if dep.startswith(bd):
                    for wd in whitelisted_dependencies_slashed:
                        if dep.startswith(wd):
                            break
                    else:
                        disallowed_deps.append(dep)

    except Exception as exc:
        eh.log_exception("An error occurred when building dependencies list", exc)
        task.set_info('An error occurred when building dependencies list, see logs for details')

    if disallowed_deps:
        # In the name of pg@
        disallowed_deps = sorted(list(set(disallowed_deps)))
        disallowed_deps = ', '.join(disallowed_deps)
        task.set_info("Disallowed dependencies detected: {}".format(disallowed_deps))
        # SEARCH-2989: Ban only trunk
        arcadia_url = th.ctx_field(task, constants.ARCADIA_URL_KEY, "")
        if 'arc/trunk' in arcadia_url:
            eh.check_failed("Disallowed dependencies detected: {}".format(disallowed_deps))
