# -*- coding: utf-8 -*-

import copy
import json
import six
import textwrap

from sandbox.sdk2 import svn
from sandbox.sdk2 import parameters
import sandbox.sandboxsdk.parameters as sb_params

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common.build import dependency_banner
from sandbox.projects.common.build.ya_package_config import consts as ya_package_consts
import sandbox.projects.common.constants as consts


BASE_BUILD_GROUP_NAME = 'Base build params'
BUILD_SYSTEM_GROUP_NAME = 'Build system params'
DEBUG_GROUP_NAME = 'Debug params'
ENVIRONMENT_GROUP_NAME = 'Environment params'
JAVA_GROUP_NAME = 'Java params'
OPTIONAL_GROUP_NAME = 'Optional params'
TEST_SYSTEM_GROUP_NAME = 'Test system params'
YT_STORE_GROUP_NAME = 'YT store params'

PACKAGES_GROUP_NAME = "Packages"
DOCKER_GROUP_NAME = "Docker"
RESOURCES_GROUP_NAME = "Resources"
PUBLISHING_GROUP_NAME = "Publishing"
TESTS_GROUP_NAME = "Tests"
ADVANCED_GROUP_NAME = "Advanced"


class NullableString(sdk2.parameters.String):
    @classmethod
    def cast(cls, value):
        value = super(NullableString, cls).cast(value)
        if value == "None":
            value = None
        return value


class DictWithNullableValues(sdk2.parameters.Dict):
    default_type = NullableString


class Packages(sdk2.parameters.String):
    name = "packages"
    description = "Package paths, related to arcadia ';'-separated"
    required = False
    group = PACKAGES_GROUP_NAME
    multiline = True


class ForceDupload(sdk2.parameters.Bool):
    name = 'force_dupload'
    description = 'Force dupload'
    required = False
    default_value = False
    group = PACKAGES_GROUP_NAME


class DuploadMaxAttempts(sdk2.parameters.Integer):
    name = 'dupload_max_attempts'
    description = 'How many times try to run dupload if it fails'
    required = False
    default_value = 1
    group = PACKAGES_GROUP_NAME


class DebianArch(sdk2.parameters.String):
    name = 'debian_arch'
    description = 'Debian arch (--debian-arch)'
    required = False
    default_value = None
    group = PACKAGES_GROUP_NAME


class DebianCompression(sdk2.parameters.String):
    name = 'debian_compression_level'
    description = 'Debian compression (the lower compression, the  faster package build)'
    required = False
    default_value = None
    choices = [
        ('Default', ''),
        ('No', '0'),
        ('Low', '3'),
        ('Medium', '6'),
        ('High', '9'),
    ]
    group = PACKAGES_GROUP_NAME


class DebianCompressionType(sdk2.parameters.String):
    name = 'debian_compression_type'
    description = 'Debian compression type'
    required = False
    default_value = None
    choices = [
        ('Default', ''),
        ('gzip', 'gzip'),
        ('xz', 'xz'),
        ('bzip2', 'bzip2'),
        ('lzma', 'lzma'),
        ('none', 'none'),
    ]
    group = PACKAGES_GROUP_NAME


class DontStoreDebianPackage(sdk2.parameters.Bool):
    name = 'debian_dont_store_package'
    description = "Don't save debian packages in resources"
    required = False
    default_value = False
    group = PACKAGES_GROUP_NAME


class BuildDebianScripts(sdk2.parameters.Bool):
    name = 'build_debian_scripts'
    description = 'Build debian scripts'
    required = False
    default_value = False
    group = PACKAGES_GROUP_NAME


class SloppyDebian(sdk2.parameters.Bool):
    name = 'sloppy_debian'
    description = 'Fast debian packages build'
    required = False
    default_value = False
    sub_fields = {
        "false": [DebianCompression.name],
    }
    group = PACKAGES_GROUP_NAME


class DebianDistribution(sdk2.parameters.String):
    name = 'debian_distribution'
    description = 'Debian distribution'
    required = False
    default_value = 'unstable'
    group = PACKAGES_GROUP_NAME


class DockerImageRepository(sdk2.parameters.String):
    name = 'docker_image_repository'
    description = 'Image repository'
    required = False
    default_value = ""
    group = DOCKER_GROUP_NAME


class DockerSaveImage(sdk2.parameters.Bool):
    name = 'docker_save_image'
    description = 'Save docker image in resource'
    required = False
    default_value = False
    group = DOCKER_GROUP_NAME


class DockerPushImage(sdk2.parameters.Bool):
    name = 'docker_push_image'
    description = 'Push docker image'
    required = False
    default_value = False
    group = DOCKER_GROUP_NAME


class DockerRegistry(sdk2.parameters.String):
    name = 'docker_registry'
    description = 'Docker registry'
    required = False
    default_value = "registry.yandex.net"
    group = DOCKER_GROUP_NAME


class DockerUser(sdk2.parameters.String):
    name = 'docker_user'
    description = 'Docker user'
    required = False
    default_value = ""
    group = DOCKER_GROUP_NAME


class DockerTokenVaultName(sdk2.parameters.String):
    name = 'docker_token_vault_name'
    description = 'Docker token vault name'
    required = False
    default_value = ""
    group = DOCKER_GROUP_NAME


class DockerYavToken(sdk2.parameters.YavSecret):
    name = 'docker_yav_token'
    description = 'Docker YAV OAuth token (use instead of vault name)'
    required = False
    group = DOCKER_GROUP_NAME


class DockerBuildNetwork(sdk2.parameters.String):
    name = 'docker_build_network'
    description = 'Docker build network'
    required = False
    default_value = "host"
    group = DOCKER_GROUP_NAME


class DockerBuildArg(DictWithNullableValues):
    name = 'docker_build_arg'
    description = 'Docker build arg (use "None" to specify --build-arg flag without a value, in which case the value from the local environment will be propagated into the container)'
    default_value = None
    group = DOCKER_GROUP_NAME


class RawPackage(sdk2.parameters.Bool):
    name = 'raw_package'
    description = 'Raw package (used with "tar" package type to get package content without tarring)'
    required = False
    default_value = False
    group = PACKAGES_GROUP_NAME


class RawPackagePath(sdk2.parameters.String):
    name = 'raw_package_path'
    description = (
        'Override name of top directory, when using raw_package option. '
        'By default top directory name is equal to ${package_name}.${package.version}'
    )
    required = False
    default_value = None
    group = PACKAGES_GROUP_NAME


class HostPlatform(sdk2.parameters.String):
    name = 'host_platform'
    description = 'Host platform'
    required = False
    default_value = ''


class TargetPlatform(sdk2.parameters.String):
    name = 'target_platform'
    description = 'Target platform'
    required = False
    default_value = ''


class FullStripBinaries(sdk2.parameters.Bool):
    name = 'full_strip_binaries'
    description = 'Strip all (like strip without parameters)'
    required = False
    default_value = False
    group = PACKAGES_GROUP_NAME


class StripBinaries(sdk2.parameters.Bool):
    name = 'strip_binaries'
    description = 'Strip debug information (i.e. strip -g)'
    required = False
    default_value = False
    sub_fields = {
        'true': [
            FullStripBinaries.name,
        ]
    }
    group = PACKAGES_GROUP_NAME


class CreateDebugPackages(sdk2.parameters.Bool):
    name = 'create_debug_packages'
    description = 'Create separate packages with debug info'
    required = False
    default_value = False
    group = PACKAGES_GROUP_NAME


class CompressPackageArchive(sdk2.parameters.Bool):
    name = 'compress_package_archive'
    description = 'Compress package archive'
    required = False
    default_value = True
    group = PACKAGES_GROUP_NAME


class CompressionFilter(sdk2.parameters.String):
    name = 'compression_filter'
    description = 'Compression filter for tar (compression must be enabled)'
    required = False
    choices = [(item, item) for item in ('gzip', 'zstd')]
    default_value = 'gzip'
    group = PACKAGES_GROUP_NAME


class CompressionLevel(sdk2.parameters.Integer):
    name = 'compression_level'
    description = 'Compression level used by filter'
    required = False
    default_value = None
    group = PACKAGES_GROUP_NAME


class UcCodec(sdk2.parameters.String):
    name = 'uc_codec'
    description = 'Codec name for uc compression'
    required = False
    default_value = None
    group = PACKAGES_GROUP_NAME


class OverwriteReadOnlyFiles(sdk2.parameters.Bool):
    name = 'overwrite_read_only_files'
    description = 'Overwrite read-only files in package'
    required = False
    default_value = False
    group = PACKAGES_GROUP_NAME


class EnsurePackagePublished(sdk2.parameters.Bool):
    name = 'ensure_package_published'
    description = 'fail ya package if publishing was required but failed'
    required = False
    default_value = False
    group = PUBLISHING_GROUP_NAME


class SemiClearBuild(sdk2.parameters.Bool):
    name = 'semi_clear_build'
    description = 'Use temporary build directory (can be used only with clear build)'
    required = False
    default_value = False


class ClearBuild(sdk2.parameters.Bool):
    name = 'clear_build'
    description = 'Clear build'
    required = False
    default_value = False
    sub_fields = {
        'true': [
            SemiClearBuild.name,
        ]
    }


class ArchitectureAll(sdk2.parameters.Bool):
    name = 'architecture_all'
    description = 'Architecture: all'
    required = False
    default_value = False
    group = PACKAGES_GROUP_NAME


class RunMediumTests(sdk2.parameters.Bool):
    name = 'run_medium_tests'
    description = 'Run medium tests'
    required = False
    default_value = False
    group = TESTS_GROUP_NAME


class RunLongTests(sdk2.parameters.Bool):
    name = 'run_long_tests'
    description = 'Run long tests'
    required = False
    default_value = False
    group = TESTS_GROUP_NAME


class IgnoreFailTests(sdk2.parameters.Bool):
    name = 'ignore_fail_tests'
    description = 'Ignore fail tests'
    required = False
    default_value = False
    group = TESTS_GROUP_NAME


class RunTests(sdk2.parameters.Bool):
    name = 'run_tests'
    description = 'Run tests after build'
    required = True
    default_value = False
    sub_fields = {
        'true': [
            RunMediumTests.name,
            RunLongTests.name,
            IgnoreFailTests.name
        ]
    }
    group = PACKAGES_GROUP_NAME


class CacheTests(sdk2.parameters.Bool):
    name = 'cache_tests'
    description = 'Cache test results (stable test uids)'
    required = False
    default_value = False
    group = TESTS_GROUP_NAME


class ResourceType(sdk2.parameters.String):
    name = 'resource_type'
    description = 'Created package resource type, or \';\'-separated list'
    required = True
    default_value = resource_types.YA_PACKAGE.name
    group = RESOURCES_GROUP_NAME


class PackageResourceTtl(sdk2.resource.Attributes.TTL):
    name = "package_resource_ttl"
    description = "Created package resource TTL"
    required = False
    default_value = 4
    group = RESOURCES_GROUP_NAME


class ResourceId(sdk2.parameters.String):
    """
    If specified, task will use existing resources instead of creating new ones
    """
    name = 'resource_id'
    description = 'Previously created package resource id, or \';\'-separated list.'
    required = False
    default_value = ''
    group = RESOURCES_GROUP_NAME


class ParentResourceId(sdk2.parameters.ParentResource):
    """
    If specified, task will use existing resources instead of creating new ones
    """
    name = 'resource_id'
    description = 'Previously created package resource ids'
    required = False
    default_value = []
    group = RESOURCES_GROUP_NAME
    multiple = True


class BuildLogsTtl(sdk2.resource.Attributes.TTL):
    name = consts.BUILD_LOGS_TTL
    description = "BUILD_LOG resources TTL"
    required = False
    default_value = 4
    group = RESOURCES_GROUP_NAME


class SaveBuildOutput(sdk2.parameters.Bool):
    name = 'save_build_output'
    description = 'Save build output in a separate resource'
    required = False
    default_value = True
    group = RESOURCES_GROUP_NAME


class Changelog(sdk2.parameters.String):
    name = 'changelog'
    description = 'Changelog (message text or path to changelog file arcadia relative)'
    required = False
    default_value = ''
    multiline = True
    group = PUBLISHING_GROUP_NAME


class KeyUser(sdk2.parameters.String):
    name = 'key_user'
    description = 'Name of the user to sign the package'
    required = False
    choices = [(v, k) for k, v in ya_package_consts.VAULT_OWNERS.items()]
    default_value = None
    group = PUBLISHING_GROUP_NAME


class PublishTo(sdk2.parameters.String):
    name = 'publish_to'
    description = 'Publish to'
    required = False
    choices = [(item, item) for item in sorted(set(ya_package_consts.DUPLOAD_CONF.keys()))]
    default_value = None
    group = PUBLISHING_GROUP_NAME


class ArtifactoryUpload(sdk2.parameters.Bool):
    name = 'artifactory_upload'
    description = 'Upload package to artifactory'
    required = False
    default_value = False
    group = PUBLISHING_GROUP_NAME


class MultiplePublishTo(sdk2.parameters.String):
    name = 'multiple_publish_to'
    description = 'Publish to'
    required = False
    default_value = ''
    group = PUBLISHING_GROUP_NAME


class MultiplePublishMapping(sdk2.parameters.Dict):
    name = 'publish_to_mapping'
    description = "Package file -> repos ';' separated (has higher priority if filled)"
    default_value = None
    group = PUBLISHING_GROUP_NAME


class MultiplePublish(sdk2.parameters.Bool):
    name = 'multiple_publish'
    description = 'Multiple publish'
    required = False
    default_value = False
    sub_fields = {
        'false': [
            PublishTo.name
        ],
        'true': [
            MultiplePublishTo.name
        ]
    }
    group = PUBLISHING_GROUP_NAME


class WheelUploadRepo(sdk2.parameters.String):
    name = 'wheel_upload_repo'
    description = 'Publish to'
    required = False
    default_value = "https://pypi.yandex-team.ru/simple"
    group = PUBLISHING_GROUP_NAME


class WheelPython3(sdk2.parameters.Bool):
    name = 'wheel_python3'
    description = 'build python 3 package'
    required = False
    default_value = False
    group = PACKAGES_GROUP_NAME


class WheelAccessKeyTokenVaultName(sdk2.parameters.String):
    name = 'wheel_access_key_token'
    description = 'Wheel access key token vault name'
    required = False
    default_value = ""
    group = PUBLISHING_GROUP_NAME


class WheelSecretKeyTokenVaultName(sdk2.parameters.String):
    name = 'wheel_secret_key_token'
    description = 'Wheel secret key token vault name'
    required = False
    default_value = ""
    group = PUBLISHING_GROUP_NAME


class PublishPackage(sdk2.parameters.Bool):
    name = 'publish_package'
    description = 'Publish package'
    required = True
    default_value = True
    group = PUBLISHING_GROUP_NAME


class NpmRegistry(sdk2.parameters.String):
    name = 'npm_registry'
    description = 'Publish to'
    required = False
    default_value = "https://npm.yandex-team.ru"
    group = PUBLISHING_GROUP_NAME


class NpmLogin(sdk2.parameters.String):
    name = 'npm_login'
    description = 'npm user login'
    required = False
    default_value = ""
    group = PUBLISHING_GROUP_NAME


class NpmPasswordVaultName(sdk2.parameters.String):
    name = 'npm_password_vault_name'
    description = 'npm password vault name'
    required = False
    default_value = ""
    group = PUBLISHING_GROUP_NAME


class UseYaDev(sdk2.parameters.Bool):
    name = 'use_ya_dev'
    description = 'Use ya-dev to build'
    required = False
    default_value = False
    group = ADVANCED_GROUP_NAME


class BeVerbose(sdk2.parameters.Bool):
    name = 'be_verbose'
    description = 'Verbose'
    required = False
    default_value = False
    group = ADVANCED_GROUP_NAME


class UseNewFormat(sdk2.parameters.Bool):
    name = 'use_new_format'
    description = 'Use new ya package json format'
    required = False
    default_value = False
    group = ADVANCED_GROUP_NAME


class CustomVersion(sdk2.parameters.String):
    name = 'custom_version'
    description = 'Custom version'
    required = False
    default_value = None
    group = ADVANCED_GROUP_NAME


class PackageResourceDescription(sdk2.parameters.Dict):
    name = 'package_resource_description'
    description = "Package file -> created package description"
    required = False
    default_value = None
    group = ADVANCED_GROUP_NAME


class PackageResourceAttrs(sdk2.parameters.Dict):
    name = 'package_resource_attrs'
    description = "Package resource attributes"
    default_value = None
    group = ADVANCED_GROUP_NAME


class Container(sdk2.parameters.Container):
    name = consts.SANDBOX_CONTAINER
    description = 'Container the task should execute in'
    default_value = None
    required = False
    group = ADVANCED_GROUP_NAME


class AdhocPackages(sdk2.parameters.List, sdk2.parameters.String):
    name = 'adhoc_packages'
    description = "Adhoc packages (pure json)"
    required = False
    default_value = None
    multiline = True
    group = ADVANCED_GROUP_NAME
    value_type = sdk2.parameters.String


class CreateBuildOutputResource(sdk2.parameters.Bool):
    name = 'create_build_output_resource'
    description = 'Create BUILD_OUTPUT resource'
    required = False
    default_value = False
    group = TESTS_GROUP_NAME


class ReleaseToNanny(sdk2.parameters.Bool):
    name = "release_to_nanny"
    description = 'Enable release to Nanny'
    required = False
    default_value = False


class PackageType(sdk2.parameters.String):
    name = 'package_type'
    description = 'Package type: debian or tarball'
    required = True
    choices = [
        (ya_package_consts.PackageType.AAR.value, ya_package_consts.PackageType.AAR.value),
        (ya_package_consts.PackageType.DEBIAN.value, ya_package_consts.PackageType.DEBIAN.value),
        (ya_package_consts.PackageType.TARBALL.value, ya_package_consts.PackageType.TARBALL.value),
        (ya_package_consts.PackageType.DOCKER.value, ya_package_consts.PackageType.DOCKER.value),
        (ya_package_consts.PackageType.RPM.value, ya_package_consts.PackageType.RPM.value),
        (ya_package_consts.PackageType.WHEEL.value, ya_package_consts.PackageType.WHEEL.value),
        (ya_package_consts.PackageType.NPM.value, ya_package_consts.PackageType.NPM.value)
    ]
    default_value = ya_package_consts.PackageType.DEBIAN.value
    sub_fields = {
        ya_package_consts.PackageType.DEBIAN.value: [
            SloppyDebian.name,
            DebianCompression.name,
            DebianCompressionType.name,
            ForceDupload.name,
            DuploadMaxAttempts.name,
            BuildDebianScripts.name,
            DontStoreDebianPackage.name,
            DebianDistribution.name,
            DebianArch.name,

            PublishPackage.name,
            MultiplePublish.name,
            MultiplePublishMapping.name,
            MultiplePublishTo.name,
            PublishTo.name,
        ],
        ya_package_consts.PackageType.DOCKER.value: [
            DockerImageRepository.name,
            DockerSaveImage.name,
            DockerPushImage.name,
            DockerUser.name,
            DockerTokenVaultName.name,
            DockerBuildNetwork.name,
            DockerBuildArg.name,
            DockerRegistry.name,
        ],
        ya_package_consts.PackageType.RPM.value: [
            PublishPackage.name,
            MultiplePublish.name,
            MultiplePublishTo.name,
            PublishTo.name,
        ],
        ya_package_consts.PackageType.AAR.value: [
            PublishPackage.name,
            MultiplePublish.name,
            MultiplePublishTo.name,
            PublishTo.name,
        ],
        ya_package_consts.PackageType.WHEEL.value: [
            WheelPython3.name,
            PublishPackage.name,
            WheelUploadRepo.name,
            WheelAccessKeyTokenVaultName.name,
            WheelSecretKeyTokenVaultName.name,
        ],
        ya_package_consts.PackageType.NPM.value: [
            PublishPackage.name,
            NpmRegistry.name,
            NpmLogin.name,
            NpmPasswordVaultName.name,
        ],
        ya_package_consts.PackageType.TARBALL.value: [
            PublishPackage.name,
            ArtifactoryUpload.name,
            MultiplePublish.name,
            MultiplePublishTo.name,
            PublishTo.name,
        ],
    }


class ClearBuild2(sdk2.parameters.Bool):
    name = consts.CLEAR_BUILD_KEY
    description = 'Clear build'
    default_value = False
    group = BASE_BUILD_GROUP_NAME


class StripBinaries2(sdk2.parameters.Bool):
    name = consts.STRIP_BINARIES
    description = 'Strip result binaries'
    default_value = False
    group = BASE_BUILD_GROUP_NAME


class UseDevVersion(sdk2.parameters.Bool):
    name = consts.USE_DEV_VERSION
    description = 'Use dev version of build system'
    default_value = False
    group = BASE_BUILD_GROUP_NAME


class KeepOn(sdk2.parameters.Bool):
    name = consts.KEEP_ON
    description = 'Build and test as much as possible'
    default_value = False
    group = BUILD_SYSTEM_GROUP_NAME


class CheckReturnCode(sdk2.parameters.Bool):
    name = consts.CHECK_RETURN_CODE
    description = 'Check "ya make" return code'
    default_value = True
    group = BUILD_SYSTEM_GROUP_NAME


class FailedTestsCauseError(sdk2.parameters.Bool):
    name = consts.FAILED_TESTS_CAUSE_ERROR
    description = 'Set FAILURE status to the task if tests failed (return code should be checked)'
    default_value = True
    group = BUILD_SYSTEM_GROUP_NAME


class ForceBuildDepends(sdk2.parameters.Bool):
    name = consts.FORCE_BUILD_DEPENDS
    description = "Build by test's DEPENDS anyway"
    group = BUILD_SYSTEM_GROUP_NAME
    default_value = False


class ForceVCSInfoUpdate(sdk2.parameters.Bool):
    name = consts.FORCE_VCS_INFO_UPDATE
    description = "Force VCS info update"
    group = BUILD_SYSTEM_GROUP_NAME
    default_value = False


class IgnoreRecurses(sdk2.parameters.Bool):
    name = consts.IGNORE_RECURSES
    description = "Build immediate target only (don't follow any RECURSEs when tests are not requested)"
    group = BUILD_SYSTEM_GROUP_NAME
    default_value = False


class AddResult(sdk2.parameters.List):
    name = consts.YA_ADD_RESULT
    description = "Add files with following filename suffixes to build artifacts"
    value_type = sdk2.parameters.String
    default_value = []
    group = BUILD_SYSTEM_GROUP_NAME


class Coverage(sdk2.parameters.Bool):
    name = consts.COVERAGE
    description = 'Tests coverage'
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class JavaCoverage(sdk2.parameters.Bool):
    name = consts.JAVA_COVERAGE
    description = 'Tests coverage for java'
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class CoveragePrefixFilter(sdk2.parameters.String):
    name = consts.COVERAGE_PREFIX_FILTER
    description = 'Tests coverage prefix filter'
    required = False
    group = TEST_SYSTEM_GROUP_NAME


class CoverageExcludeRegexp(sdk2.parameters.String):
    name = consts.COVERAGE_EXCLUDE_REGEXP
    description = 'Exclude matching files from coverage report'
    required = False
    default_value = '.*\\.pb\\.(h|cc)'
    group = TEST_SYSTEM_GROUP_NAME


class CoverageUseUnifiedAgentParameter(sdk2.parameters.Bool):
    name = consts.COVERAGE_UNIFIED_AGENT
    description = 'Use new upload scheme for coverage'
    required = False
    group = TEST_SYSTEM_GROUP_NAME
    default_value = False


class CoverageUnifiedAgentSidParameter(sdk2.parameters.String):
    name = consts.COVERAGE_UNIFIED_AGENT_SID
    description = 'Coverage sid (used in new upload scheme)'
    required = False
    group = TEST_SYSTEM_GROUP_NAME
    default_value = ''


class CoverageUseUnifiedAgentStrictParameter(sdk2.parameters.Bool):
    name = consts.COVERAGE_UNIFIED_AGENT_STRICT
    description = 'Use stirct new upload scheme for coverage'
    required = False
    group = TEST_SYSTEM_GROUP_NAME
    default_value = False


class Sanitize(sdk2.parameters.String):
    name = consts.SANITIZE
    description = 'Build with specified sanitizer'
    required = False
    group = TEST_SYSTEM_GROUP_NAME
    default_value = None
    choices = [
        ('none', None),
        ('address', 'address'),
        ('leak', 'leak'),
        ('memory', 'memory'),
        ('thread', 'thread'),
        ('undefined', 'undefined')
    ]


class LTO(sdk2.parameters.Bool):
    name = consts.LTO
    description = 'Build with LTO'
    default_value = False
    group = BUILD_SYSTEM_GROUP_NAME


class ThinLTO(sdk2.parameters.Bool):
    name = consts.THINLTO
    description = 'Build with ThinLTO'
    default_value = False
    group = BUILD_SYSTEM_GROUP_NAME


class TestParameter(parameters.Bool):
    name = consts.TESTS_REQUESTED
    description = 'Run tests'
    group = TEST_SYSTEM_GROUP_NAME
    default_value = False


class TestThreads(sdk2.parameters.Integer):
    name = consts.TEST_THREADS
    description = 'Test threads (no limit by default)'
    required = False
    default_value = 0
    group = TEST_SYSTEM_GROUP_NAME


class UseSystemPythonParameter(sdk2.parameters.Bool):
    name = consts.USE_SYSTEM_PYTHON
    description = 'Use system Python to build python libraries'
    group = BUILD_SYSTEM_GROUP_NAME
    default_value = False


class JUnitReport(sdk2.parameters.Bool):
    name = consts.JUNIT_REPORT
    description = 'Add junit report for tests'
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class TestFilters(sdk2.parameters.String):
    name = consts.TEST_FILTERS
    description = 'Test filters (for example "TUtilUrlTest::TestSchemeGet TUtilUrlTest::TestGetZone")'
    required = False
    group = TEST_SYSTEM_GROUP_NAME


class TestSizeFilter(sdk2.parameters.String):
    name = consts.TEST_SIZE_FILTER
    description = 'Filter tests by size'
    required = False
    group = TEST_SYSTEM_GROUP_NAME
    default_value = None
    choices = [
        ('none', None),
        ('small', 'small'),
        ('medium', 'medium'),
        ('small,medium', 'small,medium'),
        ('fat', 'fat'),
        ('large', 'large'),
    ]


class TestTypeFilter(sdk2.parameters.String):
    name = consts.TEST_TYPE_FILTER
    description = 'Filter tests by suite type (pytest/unittest/jtest) (semicolon separated)'
    required = False
    group = TEST_SYSTEM_GROUP_NAME
    default_value = ''


class TestCustomParameters(sdk2.parameters.String):
    name = consts.TEST_PARAMS_KEY
    description = 'Test parameters (for example "key1=val1 ... keyN=valN")'
    required = False
    group = TEST_SYSTEM_GROUP_NAME


class OutputOnlyTests(sb_params.SandboxBoolParameter):
    name = consts.OUTPUT_ONLY_TESTS
    description = 'Add only tests to the results root'
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class DefinitionFlags(sdk2.parameters.String):
    name = consts.DEFINITION_FLAGS_KEY
    description = 'Definition flags (for example "-Dkey1=val1 ... -DkeyN=valN")'
    required = False
    group = BUILD_SYSTEM_GROUP_NAME


class DirOutputs(sdk2.parameters.Bool):
    name = consts.DIR_OUTPUTS
    description = "Don't archive test node outputs"
    required = False
    default_value = True
    group = TEST_SYSTEM_GROUP_NAME


class YaTimeout(sdk2.parameters.Integer):
    name = consts.YA_TIMEOUT
    description = 'Timeout for ya command'
    required = False
    default_value = 10800  # 3h
    group = BUILD_SYSTEM_GROUP_NAME


class BuildSystem(sdk2.parameters.String):
    name = consts.BUILD_SYSTEM_KEY
    description = 'Build system'
    required = True
    default_value = consts.YMAKE_BUILD_SYSTEM
    group = BUILD_SYSTEM_GROUP_NAME
    choices = [
        ('Ya', consts.YMAKE_BUILD_SYSTEM),
        # TODO: remove after experiments
        ('Ya force', consts.YA_MAKE_FORCE_BUILD_SYSTEM),
        ('semi-distbuild', consts.SEMI_DISTBUILD_BUILD_SYSTEM),
        ('distbuild', consts.DISTBUILD_BUILD_SYSTEM),
    ]
    sub_fields = {
        consts.YMAKE_BUILD_SYSTEM: [consts.TARGET_PLATFORM_KEY],
        consts.YA_MAKE_FORCE_BUILD_SYSTEM: [consts.TARGET_PLATFORM_KEY],
    }


class DistbuildPool(sdk2.parameters.String):
    name = consts.DISTBUILD_POOL
    description = 'Distbuild pool'
    required = False
    group = BUILD_SYSTEM_GROUP_NAME


# TODO: Remove this parameter. Use BuildSystem instead
class BuildSystemPlain(BuildSystem):
    sub_fields = None


class HostPlatform2(sdk2.parameters.String):
    name = consts.HOST_PLATFORM_KEY
    description = 'Host platform (Only for non-default toolchain)'
    required = False
    group = BUILD_SYSTEM_GROUP_NAME


class TargetPlatform2(sdk2.parameters.String):
    name = consts.TARGET_PLATFORM_KEY
    description = 'Target platform (deprecated: use "Target platform flags" instead)'
    required = False
    group = BUILD_SYSTEM_GROUP_NAME


class TargetPlatformFlags(sdk2.parameters.String):
    name = consts.TARGET_PLATFORM_FLAGS
    description = 'Target platform flags (only for cross-compilation)'
    required = False
    group = BUILD_SYSTEM_GROUP_NAME


class HostPlatformFlags(sdk2.parameters.String):
    name = consts.HOST_PLATFORM_FLAGS
    description = 'Host platform flags (used for cross-compilation and building of tools)'
    required = False
    group = BUILD_SYSTEM_GROUP_NAME


class UseYaBloat(sdk2.parameters.String):
    name = consts.USE_YA_BLOAT
    description = 'Use ya bloat for specified binaries (semicolon separated)'
    required = False
    group = OPTIONAL_GROUP_NAME


class JsonPrefix(sb_params.SandboxStringParameter):
    name = consts.JSON_PREFIX
    description = 'Prefix for graph uids'
    required = False
    group = OPTIONAL_GROUP_NAME


class ReportTestsOnly(sdk2.parameters.Bool):
    name = consts.REPORT_TESTS_ONLY
    description = 'Keep only test entries in report'
    required = False
    group = TEST_SYSTEM_GROUP_NAME


class TestLogLevel(sdk2.parameters.String):
    name = consts.TEST_LOG_LEVEL
    description = 'Specifies logging level for output test logs'
    required = False
    group = TEST_SYSTEM_GROUP_NAME
    default_value = consts.DEFAULT_TEST_LOG_LEVEL
    choices = [
        ('Debug', consts.DEBUG_TEST_LOG_LEVEL),
        ('Info', consts.INFO_TEST_LOG_LEVEL),
        ('Warning', consts.WARNING_TEST_LOG_LEVEL),
        ('Error', consts.ERROR_TEST_LOG_LEVEL),
        ('Critical', consts.CRITICAL_TEST_LOG_LEVEL),
    ]


class TestTag(sdk2.parameters.String):
    name = consts.TEST_TAG
    description = 'Run tests that have specified tag'
    required = False
    group = TEST_SYSTEM_GROUP_NAME


class TestAllureReport(sdk2.parameters.Bool):
    name = consts.ALLURE_REPORT
    description = 'Generate allure report'
    required = False
    group = TEST_SYSTEM_GROUP_NAME


class TestAllureReportTTL(sdk2.parameters.Integer):
    name = consts.ALLURE_REPORT_TTL
    description = 'Allure report TTL'
    required = False
    default_value = 4
    group = TEST_SYSTEM_GROUP_NAME


class DisableTestTimeout(sdk2.parameters.Bool):
    name = consts.DISABLE_TEST_TIMEOUT
    description = 'Turn off timeout for tests (only for local runs, incompatible with --cache-tests, --dist)'
    required = False
    group = TEST_SYSTEM_GROUP_NAME
    default_value = False


class BuildType(sdk2.parameters.String):
    name = consts.BUILD_TYPE_KEY
    description = 'Build type'
    required = True
    group = BUILD_SYSTEM_GROUP_NAME
    default_value = consts.DEFAULT_BUILD_TYPE
    choices = [
        ('Release', consts.RELEASE_BUILD_TYPE),
        ('Debug', consts.DEBUG_BUILD_TYPE),
        ('Profile', consts.PROFILE_BUILD_TYPE),
        ('Coverage', consts.COVERAGE_BUILD_TYPE),
        ('Release with debug info', consts.RELEASE_WITH_DEBUG_INFO_BUILD_TYPE),
        ('Valgrind (debug)', consts.VALGRIND_DEBUG_BUILD_TYPE),
        ('Valgrind (release) (only for build with ya)', consts.VALGRIND_RELEASE_BUILD_TYPE),
    ]


class BuildOutputHtmlTtl(sdk2.parameters.Integer):
    name = consts.BUILD_OUTPUT_HTML_TTL
    description = 'TTL for BUILD_OUTPUT_HTML resources'
    required = False
    # default_value is not set intentionally, see BuildOutputTtl for details
    group = BASE_BUILD_GROUP_NAME


class BuildOutputTtl(sdk2.parameters.Integer):
    name = consts.BUILD_OUTPUT_TTL
    description = 'TTL for BUILD_OUTPUT resources'
    required = False
    # default_value is not set intentionally to easily override this parameter with class variable likewise
    # https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/jupiter/YaMake/__init__.py?blame=true&rev=5029296#L12
    group = BASE_BUILD_GROUP_NAME


class DoNotRemoveResources(sb_params.SandboxBoolParameter):
    description = 'Set "do not remove" for resources'
    name = 'do_not_remove_resources'
    default_value = False


class DoNotRemoveResources2(sdk2.parameters.Bool):
    description = 'Set "do not remove" for resources'
    name = 'do_not_remove_resources'
    default_value = False


class ArcadiaUrl(sdk2.parameters.ArcadiaUrl):
    name = consts.ARCADIA_URL_KEY
    required = True
    description = 'Svn url for arcadia'
    default_value = svn.Arcadia.trunk_url()
    group = BASE_BUILD_GROUP_NAME


class ArcadiaFrozenRepositoryPackage(sdk2.parameters.String):
    name = consts.ARCADIA_FREPKAGE
    description = 'Arcadia frozen repository package'
    default_value = None
    group = BASE_BUILD_GROUP_NAME


class ArcadiaFrozenRepositoryPackageTargetUid(sdk2.parameters.String):
    name = consts.ARCADIA_FREPKAGE_TARGET_UID
    description = 'Strip graph from frepkage using target uid as single result uid'
    default_value = None
    group = BASE_BUILD_GROUP_NAME


class UseArcadiaApiFuse(sdk2.parameters.Bool):
    name = consts.USE_AAPI_FUSE
    description = 'Use arcadia-api fuse'
    default_value = True
    group = BASE_BUILD_GROUP_NAME


class UseArcadiaApiFuseAsDefault(sdk2.parameters.Bool):
    name = consts.USE_AAPI_FUSE
    description = 'Use arcadia-api fuse'
    default_value = True
    group = BASE_BUILD_GROUP_NAME


class UseArcInsteadOfArcadiaApi(sdk2.parameters.Bool):
    name = consts.USE_ARC_INSTEAD_OF_AAPI
    description = 'Use arc fuse instead of aapi'
    default_value = True
    group = BASE_BUILD_GROUP_NAME


class UseArcInsteadOfArcadiaApiAsDefault(sdk2.parameters.Bool):
    name = consts.USE_ARC_INSTEAD_OF_AAPI
    description = 'Use arc fuse instead of aapi'
    default_value = True
    group = BASE_BUILD_GROUP_NAME


class MinimizeArcMountPath(sdk2.parameters.Bool):
    name = consts.MINIMIZE_ARC_MOUNT_PATH
    description = 'Minimize Arc mount path prefix length'
    default_value = False
    group = BASE_BUILD_GROUP_NAME


class AllowArcadiaApiFallback(sdk2.parameters.Bool):
    name = consts.ALLOW_AAPI_FALLBACK
    description = 'Fallback to svn/hg if AAPI services are temporary unavailable'
    default_value = False
    group = BASE_BUILD_GROUP_NAME


class ArcSecret(sdk2.parameters.YavSecret):
    name = consts.ARC_SECRET
    description = 'Secret with arc token'
    group = BASE_BUILD_GROUP_NAME


class YavToken(sdk2.parameters.YavSecret):
    name = consts.YAV_TOKEN
    description = 'YAV OAuth token for Arc for receive data decryption key'
    group = BASE_BUILD_GROUP_NAME


class CheckoutParameter(sdk2.parameters.Bool):
    name = consts.CHECKOUT
    description = 'Use selective checkout (run ya make with --checkout)'
    required = False
    default_value = False


class CheckoutModeParameter(sdk2.parameters.String):
    name = consts.CHECKOUT_MODE
    description = 'Selective checkout mode'
    required = False
    default_value = consts.CHECKOUT_MODE_MANUAL
    choices = [
        (consts.CHECKOUT_MODE_AUTO, consts.CHECKOUT_MODE_AUTO),
        (consts.CHECKOUT_MODE_MANUAL, consts.CHECKOUT_MODE_MANUAL),
    ]
    sub_fields = {
        consts.CHECKOUT_MODE_MANUAL: [
            CheckoutParameter.name
        ]
    }


class CheckoutModeAutoParameter(CheckoutModeParameter):
    default_value = consts.CHECKOUT_MODE_AUTO


UseArcadiaApiFuse.sub_fields = {
    'true': [AllowArcadiaApiFallback.name, UseArcInsteadOfArcadiaApi.name]
}


class ForceDisableArcadiaTestsData(sdk2.parameters.Bool):
    name = consts.FORCE_DISABLE_ARCADIA_TESTS_DATA
    description = 'Force disable arcadia_tests_data preparing'
    required = False
    default_value = False


class BundleKey(sdk2.parameters.Bool):
    name = consts.BUILD_BUNDLE_KEY
    description = 'Build bundle (multiple archs)'
    default_value = False
    group = BASE_BUILD_GROUP_NAME


class MakeContextOnDistbuild(sdk2.parameters.Bool):
    name = consts.MAKE_CONTEXT_ON_DISTBUILD
    description = 'Make context on distbuild'
    default_value = False
    group = BASE_BUILD_GROUP_NAME


class ArcadiaPatch(sdk2.parameters.String):
    name = consts.ARCADIA_PATCH_KEY
    description = (
        "Apply patch (link to Arcanum review formatted as `arc:id`, "
        "diff file rbtorrent, paste.y-t.ru link or plain text). "
        "Doc: https://nda.ya.ru/3QTTV4"
    )
    default_value = ''
    multiline = True
    group = BASE_BUILD_GROUP_NAME

    @classmethod
    def cast(cls, value):
        if isinstance(value, (six.text_type, six.binary_type)):
            return six.ensure_text(value)
        elif value is None:
            return None if cls.required else u''
        else:
            return six.text_type(value)


class RamDriveSize(sdk2.parameters.Integer):
    name = consts.RAM_DRIVE_SIZE
    description = 'Ram drive size (Mb)'
    required = False
    default_value = 0
    group = TEST_SYSTEM_GROUP_NAME


class VaultOwner(sdk2.parameters.String):
    name = consts.VAULT_OWNER
    description = 'Vault owner'
    default_value = ''
    group = ENVIRONMENT_GROUP_NAME


class VaultKeyName(sdk2.parameters.String):
    name = consts.VAULT_KEY_NAME
    description = 'Vault key name'
    default_value = ''
    group = ENVIRONMENT_GROUP_NAME


class BuildArcadiaUrl(sb_params.SandboxArcadiaUrlParameter):
    @classmethod
    def cast(cls, value):
        value = super(BuildArcadiaUrl, cls).cast(value)
        if value:
            parsed_url = svn.Arcadia.parse_url(value)
            try:
                super(BuildArcadiaUrl, cls).cast(svn.Arcadia.replace(value, path=parsed_url.path + "/devtools"))
            except ValueError:
                super(BuildArcadiaUrl, cls).cast(svn.Arcadia.replace(value, path=parsed_url.path + "/web"))
        return value


class CacheTestResults(sdk2.parameters.Bool):
    name = consts.CACHE_TEST_RESULTS
    description = 'Cache test results'
    required = False
    group = TEST_SYSTEM_GROUP_NAME
    default_value = False


class TestsRetriesCount(sdk2.parameters.Integer):
    name = consts.TESTS_RETRIES
    description = 'Tests retries'
    required = False
    default_value = 1
    group = TEST_SYSTEM_GROUP_NAME


class TestsFailureCode(sdk2.parameters.Integer):
    name = consts.TEST_FAILURE_CODE_KEY
    description = 'Tests failure exit code'
    required = False
    default_value = 10
    group = TEST_SYSTEM_GROUP_NAME


class CanonizeTests(sdk2.parameters.Bool):
    name = consts.CANONIZE_TESTS
    description = 'Canonize tests'
    required = False
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class CanonizePR(sdk2.parameters.Bool):
    name = consts.CANONIZE_PR
    description = 'Create auto-merge PR with canonization data (arc only)'
    required = False
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class PGOAdd(sdk2.parameters.Bool):
    name = consts.PGO_ADD
    description = 'Create PGO profile'
    default_value = False
    group = BUILD_SYSTEM_GROUP_NAME


class PGOUse(sdk2.parameters.Resource):
    name = consts.PGO_USE
    description = 'Use PGO profiles'
    multiple = True
    resource_type = resource_types.PGO_PROFILE_RESOURCE
    group = BUILD_SYSTEM_GROUP_NAME


class PGOMergeTimeout(sdk2.parameters.Integer):
    name = consts.PGO_MERGE_TIMEOUT
    description = 'Timeout for llvm-profdata tool( merge PGO raw profiles )'
    default_value = 600
    required = True
    group = BUILD_SYSTEM_GROUP_NAME


class JvmArgs(sdk2.parameters.String):
    name = consts.JVM_ARGS_KEY
    description = 'JVM args'
    required = False
    group = JAVA_GROUP_NAME


class Musl(sdk2.parameters.Bool):
    name = consts.MUSL
    description = 'Build with musl-libc'
    default_value = False
    group = BUILD_SYSTEM_GROUP_NAME


class UsePrebuiltTools(sdk2.parameters.String):
    name = consts.USE_PREBUILT_TOOLS
    description = 'Use prebuilt tools'
    required = False
    group = BUILD_SYSTEM_GROUP_NAME
    default_value = 'default'
    choices = [
        ('Default', 'default'),
        ('Yes', 'yes'),
        ('No', 'no'),
    ]


class Sonar(sdk2.parameters.Bool):
    name = consts.SONAR
    description = 'Analyze code with sonar'
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class SonarOptions(sdk2.parameters.String):
    name = consts.SONAR_OPTIONS
    description = 'Sonar options( semicolon separated )'
    required = False
    group = TEST_SYSTEM_GROUP_NAME


class SonarProjectFilter(sdk2.parameters.String):
    name = consts.SONAR_PROJECT_FILTER
    description = 'Sonar project filter'
    required = False
    group = TEST_SYSTEM_GROUP_NAME


class SonarDefaultProjectFilter(sdk2.parameters.Bool):
    name = consts.SONAR_DEFAULT_PROJECT_FILTER
    description = 'Use default sonar project filter'
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class JavacOptions(sdk2.parameters.String):
    name = consts.JAVAC_OPTIONS
    description = 'Javac options( semicolon separated )'
    required = False
    group = JAVA_GROUP_NAME


class DNS64(sb_params.SandboxBoolParameter):
    name = consts.DNS64
    description = 'Use DNS64'
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class RunTaggedTestsOnSandboxOption(sdk2.parameters.Bool):
    name = consts.RUN_TAGGED_TESTS_ON_SANDBOX
    description = (
        'Run tests tagged with ya:force_sandbox on Sandbox '
        'using subtask for each test (applying requirements properly)'
    )
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class RunTaggedTestsOnYtOption(sdk2.parameters.Bool):
    name = consts.RUN_TAGGED_TESTS_ON_YT
    description = 'Run tests tagged with ya:yt as YT vanilla operation'
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class TraceYaOutputOption(sdk2.parameters.Bool):
    name = consts.TRACE_YA_OUTPUT
    description = 'Trace "ya make" output'
    default_value = False
    group = DEBUG_GROUP_NAME


class CreateHtmlResultsParameter(sdk2.parameters.Bool):
    name = consts.CREATE_HTML_RESULTS_RESOURCE
    description = "Create html resource from 'ya make' results"
    required = False
    default_value = False
    group = TEST_SYSTEM_GROUP_NAME


class EnvironmentVarsParam(sdk2.parameters.String):
    name = consts.ENV_VARS
    description = textwrap.dedent('''\
        Environment variables (e.g. VAR1=val1 VAR2=\'v a l 2\').

        May be used with Vault:
        - $(vault:file:owner:name); or
        - $(vault:value:owner:name)

        In the first case template will be replaced with path to the file with secret,
        in second case template will be replaced with value of the secret.

        Example:
            - YT_TOKEN='$(vault:value:yazevnul:yazevnul-yt_token)'
            - YT_TOKEN_PATH='$(vault:file:yazevnul:yazevnul-yt_token)'
    ''')
    default_value = ''
    group = ENVIRONMENT_GROUP_NAME


class EnvironmentVarsParamSecret(sdk2.parameters.YavSecret):
    name = consts.ENV_VARS_SECRET
    description = textwrap.dedent('''\
        Secret containing environment variables.
        Secret keys will be treated as environment variable names with corresponding value
    ''')
    required = False
    group = ENVIRONMENT_GROUP_NAME


class YtStore(sdk2.parameters.Bool):
    name = consts.YA_YT_STORE
    description = 'Use YT store'
    default_value = True
    group = YT_STORE_GROUP_NAME


class YtProxy(sdk2.parameters.String):
    name = consts.YA_YT_PROXY
    description = 'YT store proxy'
    required = False
    group = YT_STORE_GROUP_NAME


class YtDir(sdk2.parameters.String):
    name = consts.YA_YT_DIR
    description = 'YT store cypress path'
    required = False
    group = YT_STORE_GROUP_NAME


class YtTokenVaultOwner(sdk2.parameters.String):
    name = consts.YA_YT_TOKEN_VAULT_OWNER
    description = 'YT token vault owner'
    required = False
    group = YT_STORE_GROUP_NAME


class YtTokenVaultName(sdk2.parameters.String):
    name = consts.YA_YT_TOKEN_VAULT_NAME
    description = 'YT token vault name'
    required = False
    group = YT_STORE_GROUP_NAME


class YtTokenYavSecret(sdk2.parameters.YavSecret):
    name = consts.YA_YT_TOKEN_YAV_SECRET
    description = 'Secret with YT store token'
    group = YT_STORE_GROUP_NAME
    required = False


class ArtifactoryPassword(sdk2.parameters.YavSecret):
    name = "artifactory_password"
    description = 'Secret with artifactory password'
    group = PUBLISHING_GROUP_NAME
    required = False


class YtPut(sdk2.parameters.Bool):
    name = consts.YA_YT_PUT
    description = 'Upload build artifacts to YT store'
    default_value = False
    group = YT_STORE_GROUP_NAME


class YtStoreCodec(sdk2.parameters.String):
    name = consts.YA_YT_STORE_CODEC
    description = 'YT store codec'
    required = False
    group = YT_STORE_GROUP_NAME


class YtReplaceResult(sdk2.parameters.Bool):
    name = consts.YA_YT_REPLACE_RESULT
    description = 'Build only targets that need to be uploaded to the YT store'
    default_value = False
    group = YT_STORE_GROUP_NAME


class YtReplaceResultAddObjects(sdk2.parameters.Bool):
    name = consts.YA_YT_REPLACE_RESULT_ADD_OBJECTS
    description = 'Tune yt_replace_result option: add objects (.o) files to build results'
    default_value = False
    group = YT_STORE_GROUP_NAME


class YtReplaceResultRmBinaries(sdk2.parameters.Bool):
    name = consts.YA_YT_REPLACE_RESULT_RM_BINARIES
    description = 'Tune yt_replace_result option: remove all non-tool binaries from build results'
    default_value = False
    group = YT_STORE_GROUP_NAME


class YtStoreExclusive(sdk2.parameters.Bool):
    name = consts.YA_YT_STORE_EXCLUSIVE
    description = "Use YT store exclusively - fail on cache miss"
    default_value = False
    group = YT_STORE_GROUP_NAME


class YtMaxCacheSize(sdk2.parameters.Integer):
    name = consts.YA_YT_MAX_CACHE_SIZE
    description = 'YT max cache size (bytes)'
    required = False
    group = YT_STORE_GROUP_NAME


class YtStoreThreads(sdk2.parameters.Integer):
    name = consts.YA_YT_STORE_THREADS
    description = 'YT max download threads'
    required = False
    group = YT_STORE_GROUP_NAME
    default_value = None


class YtStoreRefreshOnRead(sdk2.parameters.Bool):
    name = consts.YA_YT_STORE_REFRESH_ON_READ
    description = "On read mark cache items as fresh (simulate LRU)"
    default_value = False
    group = YT_STORE_GROUP_NAME


YtStore.sub_fields = {
    'true': [
        YtProxy.name,
        YtDir.name,
        YtTokenVaultOwner.name,
        YtTokenVaultName.name,
        YtTokenYavSecret.name,
        YtPut.name,
        YtStoreCodec.name,
        YtReplaceResult.name,
        YtMaxCacheSize.name,
        YtStoreThreads.name,
    ]
}


YtPut.sub_fields = {
    'true': [
        YtStoreCodec.name,
        YtReplaceResult.name,
        YtMaxCacheSize.name,
    ]
}


class SeparateDebug(sb_params.SandboxBoolParameter):
    name = consts.STRP_DBG
    description = 'Separate debug info'
    default_value = True
    group = BUILD_SYSTEM_GROUP_NAME


# sdk2 parameters
class ArcadiaParameters(sdk2.Task.Parameters):
    checkout_arcadia_from_url = ArcadiaUrl()
    build_bundle = BundleKey()
    arcadia_patch = ArcadiaPatch()
    arcadia_frepkage = ArcadiaFrozenRepositoryPackage()
    arcadia_frepkage_target_uid = ArcadiaFrozenRepositoryPackageTargetUid()
    arc_secret = ArcSecret()


class CheckDepsParameters(sdk2.Task.Parameters):
    check_dependencies = dependency_banner.CheckDependencies()
    banned_dependencies = dependency_banner.BannedDependencies()
    whitelisted_dependencies = dependency_banner.WhitelistedDependencies()
    ban_ut_dependencies = dependency_banner.BanUtDependencies()


class TestSpecificParameters(sdk2.Task.Parameters):
    test = TestParameter()
    test_threads = TestThreads()
    test_params = TestCustomParameters()
    test_filters = TestFilters()
    test_size_filter = TestSizeFilter()
    test_tag = TestTag()
    test_type_filter = TestTypeFilter()
    allure_report = TestAllureReport()
    allure_report_ttl = TestAllureReportTTL()
    report_tests_only = ReportTestsOnly()
    junit_report = JUnitReport()
    test_log_level = TestLogLevel()
    tests_coverage = Coverage()
    java_coverage = JavaCoverage()
    coverage_prefix_filter = CoveragePrefixFilter()
    coverage_exclude_regexp = CoverageExcludeRegexp()
    coverage_unified_agent = CoverageUseUnifiedAgentParameter()
    coverage_unified_agent_sid = CoverageUnifiedAgentSidParameter()
    coverage_unified_agent_strict = CoverageUseUnifiedAgentStrictParameter()
    sanitize = Sanitize()
    disable_test_timeout = DisableTestTimeout()
    cache_test_results = CacheTestResults()
    ram_drive_size = RamDriveSize()
    tests_retries = TestsRetriesCount()
    test_failure_code = TestsFailureCode()
    canonize_tests = CanonizeTests()
    canonize_pr = CanonizePR()
    sonar = Sonar()
    sonar_options = SonarOptions()
    sonar_project_filter = SonarProjectFilter()
    sonar_default_project_filter = SonarDefaultProjectFilter()
    javac_options = JavacOptions()
    dns64 = DNS64()
    run_tagged_tests_on_sandbox = RunTaggedTestsOnSandboxOption()
    run_tagged_tests_on_yt = RunTaggedTestsOnYtOption()
    dir_outputs = DirOutputs()
    create_html_results_resource = CreateHtmlResultsParameter(default_value=True)


class DebugSpecificParameters(sdk2.Task.Parameters):
    trace_ya_output = TraceYaOutputOption()


class JavaSpecificParameters(sdk2.Task.Parameters):
    jvm_args = JvmArgs()


class YtStoreParameters(sdk2.Task.Parameters):
    ya_yt_store = YtStore()
    ya_yt_proxy = YtProxy()
    ya_yt_dir = YtDir()
    ya_yt_token_vault_owner = YtTokenVaultOwner()
    ya_yt_token_vault_name = YtTokenVaultName()
    ya_yt_token_yav_secret = YtTokenYavSecret()
    ya_yt_put = YtPut()
    ya_yt_store_codec = YtStoreCodec()
    ya_yt_replace_result = YtReplaceResult()
    ya_yt_replace_result_add_objects = YtReplaceResultAddObjects()
    ya_yt_replace_result_rm_binaries = YtReplaceResultRmBinaries()
    ya_yt_max_cache_size = YtMaxCacheSize()
    ya_yt_store_threads = YtStoreThreads()
    ya_yt_store_refresh_on_read = YtStoreRefreshOnRead()


class BuildSystemParameters(sdk2.Task.Parameters):
    clear_build = ClearBuild()
    build_type = BuildType()
    build_system = BuildSystem()
    distbuild_pool = DistbuildPool()
    definition_flags = DefinitionFlags()
    check_return_code = CheckReturnCode()
    failed_tests_cause_error = FailedTestsCauseError()
    keep_on = KeepOn()
    test_params = TestCustomParameters()
    host_platform = HostPlatform()
    target_platform = TargetPlatform()
    target_platform_flags = TargetPlatformFlags()
    sanitize = Sanitize()


class ArcaiaApiParameters(sdk2.Task.Parameters):
    use_aapi_fuse = UseArcadiaApiFuse()
    use_arc_instead_of_aapi = UseArcInsteadOfArcadiaApi()
    aapi_fallback = AllowArcadiaApiFallback()
    minimize_arc_mount_path = MinimizeArcMountPath()


class ArcaiaApiParametersAsDefault(sdk2.Task.Parameters):
    use_aapi_fuse = UseArcadiaApiFuseAsDefault()
    use_arc_instead_of_aapi = UseArcInsteadOfArcadiaApiAsDefault()
    aapi_fallback = AllowArcadiaApiFallback()
    minimize_arc_mount_path = MinimizeArcMountPath()


class CopyTrunk(sdk2.parameters.Bool):
    name = consts.COPY_TRUNK
    description = 'Try to checkout branch by copying from trunk'
    default_value = False
    group = BASE_BUILD_GROUP_NAME


class LTOBuildParameters(sdk2.Task.Parameters):
    lto = LTO()
    thinlto = ThinLTO()


class MuslBuildParameters(sdk2.Task.Parameters):
    musl = Musl()


class UsePrebuiltToolsParamerters(sdk2.Task.Parameters):
    use_prebuilt_tools = UsePrebuiltTools()


# sdk1 functions to get parameters
def get_sdk1_parameters(sdk2_parameters_class):
    # some sdk2 parameters are not supported in sdk1
    EXCLUDE_PARAMETERS = ["client_tags", "json_parameter"]
    return list(p for p in sdk2_parameters_class if p.name not in EXCLUDE_PARAMETERS)


def get_common_build_params(default=consts.YMAKE_BUILD_SYSTEM):
    return get_arcadia_params() + get_build_system_params()


def get_build_system_params():
    return get_sdk1_parameters(BuildSystemParameters)


def get_arcadia_params():
    return get_sdk1_parameters(ArcadiaParameters)


def get_check_deps_params():
    return get_sdk1_parameters(CheckDepsParameters)


def get_test_specific_params():
    return get_sdk1_parameters(TestSpecificParameters)


def get_java_specific_params():
    return get_sdk1_parameters(JavaSpecificParameters)


def get_yt_store_params():
    return get_sdk1_parameters(YtStoreParameters)


def get_aapi_parameters():
    return get_sdk1_parameters(ArcaiaApiParameters)


def get_aapi_parameters_as_default():
    return get_sdk1_parameters(ArcaiaApiParametersAsDefault)


def get_lto_parameters():
    return get_sdk1_parameters(LTOBuildParameters)


def get_musl_parameters():
    return get_sdk1_parameters(MuslBuildParameters)


def get_use_prebuilt_tools_parameters():
    return get_sdk1_parameters(UsePrebuiltToolsParamerters)
