# coding: utf-8

import os
import platform

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import environments

from sandbox import sdk2
from sandbox.sdk2.vcs import svn
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects import resource_types


class Parameters(sdk2.Parameters):
    ttl = sdk2.parameters.Integer("TTL for resources", default=30)
    use_sandbox_gcc = sdk2.parameters.Bool("Use Sandbox GCC", default_value=True)
    target_platform_alias = sdk2.parameters.String("Target platform alias")
    build_version = sdk2.parameters.String("Build version")


def drop_skynet_paths(env_value):
    """ Cut-off skynet pathes. """
    return ":".join(filter(lambda x: not x.startswith("/skynet/"), env_value.split(":")))


def prepare_compilers(env):
    gcc = environments.GCCEnvironment(version="4.9.2")
    gcc.prepare()
    compilers = gcc.get_compilers()
    env["CC"] = compilers["CC"]
    env["CXX"] = compilers["CXX"]
    compiler_lib_path = os.path.join(gcc.get_environment_folder(), gcc.name, "lib")
    env["LD_LIBRARY_PATH"] = "{}:{}".format(
        compiler_lib_path, env["LD_LIBRARY_PATH"]
    ) if "LD_LIBRARY_PATH" in env else compiler_lib_path

    return compilers


def create_resource(is_inplace_archive, result_archive_name, result_archive_path, result_dir, env):
    try:
        with sdk2.helpers.ProcessLog(logger="tar") as pl:
            pl.logger.propagate = 1
            if is_inplace_archive:
                sp.Popen(
                    ["tar", "-czf", result_archive_path] + os.listdir(result_dir),
                    cwd=result_dir,
                    env=env,
                    stdout=pl.stdout, stderr=sp.STDOUT
                ).wait()
            else:
                sp.Popen(
                    ["tar", "-czf", result_archive_name, result_dir],
                    env=env,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT
                ).wait()
    except Exception as e:
        raise common.errors.TaskFailure(e.message)


class ScriptBuildTask2(sdk2.Task):
    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.Group.OSX
        cores = 4
        dns = ctm.DnsType.DNS64
        environments = (environments.SvnEnvironment(),)

    class Parameters(Parameters):
        tool_name = sdk2.parameters.String("Tool name for build")
        script_name = sdk2.parameters.String("Script name (from default script dir)")
        inplace_archive = sdk2.parameters.Bool("Create archive inplace?")

    SVN_SCRIPTS_DIR = "arcadia:/arc/trunk/arcadia/devtools/tools_build"
    LOGS_DIR = "logs"

    RESOURCE_TYPE = resource_types.SCRIPT_BUILD_RESULT

    def run_script(self, version, result_dir_path, env, extra_params=None):
        if extra_params is None:
            extra_params = []

        with sdk2.helpers.ProcessLog(self, logger="build") as pl:
            pl.logger.propagate = 1
            sp.Popen(
                [
                    "/skynet/python/bin/python",
                    self.abs_path("scripts/{script_name}".format(script_name=self.get_script_name())),
                    "--version={version}".format(version=version) if version else "",
                    "--logs={logsdir}".format(logsdir=self.path(self.LOGS_DIR))
                ] + extra_params + [result_dir_path],
                shell=True, env=env, stdout=pl.stdout, stderr=sp.STDOUT
            ).wait()

    def get_resource_description(self, version, target):
        return "{tool_name} {version} for {target}".format(
            tool_name=self.Parameters.tool_name,
            version=version,
            target=target
        )

    def get_script_name(self):
        script_name = self.Parameters.script_name
        script_name_by_tool_name = "build_{tool_name}.py".format(tool_name=self.Parameters.tool_name)
        return script_name or script_name_by_tool_name

    @property
    def build_dir_name(self):
        """
        If `None` then task directory is used for the build process.

        :return: str
        """
        return "build"

    def get_result_dir_name(self):
        return self.Parameters.tool_name

    def create_resource(self, result_dir, env):
        result_archive_name = "{}.tgz".format(self.Parameters.tool_name)

        create_resource(
            self.Parameters.inplace_archive,
            result_archive_name,
            self.path(result_archive_name),
            result_dir,
            env
        )

        return result_archive_name

    @property
    def current_platform(self):
        return platform.platform()

    def export_scripts(self):
        svn.Arcadia.export(self.SVN_SCRIPTS_DIR, "scripts")
        if not os.path.exists("scripts"):
            raise common.errors.TaskError("Failed to export build script from svn")

    def on_execute(self):
        env = dict(os.environ)
        if self.Parameters.use_sandbox_gcc:
            self.Context.compilers = prepare_compilers(env)

        # Some clients may not have valid temporary directory, so we use custom.
        if not os.path.exists("_tmp"):
            os.mkdir("_tmp")
        env["TMPDIR"] = self.path("_tmp")

        # Drop path set by skynet python binary
        for env_key in ("LD_LIBRARY_PATH", "LIBRARY_PATH"):
            if env_key in env:
                env[env_key] = drop_skynet_paths(env[env_key])

        self.export_scripts()
        result_dir_name = self.get_result_dir_name()
        paths.remove_path(self.path(self.LOGS_DIR))
        paths.remove_path(self.path(result_dir_name))

        version = self.Parameters.build_version

        build_error = None

        build_dir_name = self.build_dir_name
        old_dir = None

        if build_dir_name:
            build_dir_abs = self.path(build_dir_name)
            paths.remove_path(build_dir_abs)
            os.makedirs(build_dir_abs)

            old_dir = os.getcwd()
            os.chdir(build_dir_abs)

        try:
            self.run_script(version, self.path(result_dir_name), env)
        except common.errors.SubprocessError as e:
            build_error = e
        finally:
            if old_dir:
                os.chdir(old_dir)

        if os.path.exists(self.LOGS_DIR):
            resource_types.BUILD_LOGS(
                self,
                "Build logs",
                self.LOGS_DIR,
            )

        if build_error is not None:
            if build_dir_name:
                resource_types.BUILD_LOGS(
                    self,
                    "Build directory",
                    build_dir_name,
                )
            raise common.errors.TaskFailure(build_error)

        if not os.path.exists(result_dir_name):
            raise common.errors.UnknownTaskType("Can't find {name} in task directory".format(name=result_dir_name))

        resource_path = self.create_resource(result_dir_name, env)

        attrs = {"platform": self.current_platform, "ttl": self.Parameters.ttl}

        if self.Parameters.build_version:
            attrs["version"] = self.Parameters.build_version

        self.RESOURCE_TYPE(
            self,
            self.get_resource_description(version, self.Parameters.target_platform_alias),
            resource_path,
            **attrs
        )
