# -*- coding:utf-8 -*-
import six

import glob
import json
import logging
import os
import platform
import random
import re
import shutil
import string
import time

from six.moves.urllib import parse as urlparse

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox import common
from sandbox.common import fs
from sandbox.sandboxsdk import channel
# TODO: replace errors with common.errors
from sandbox.sandboxsdk import errors
from sandbox.sdk2 import svn

from sandbox.projects import resource_types
from sandbox.projects.common import constants as consts
from sandbox.projects.common import string as string_lib
from sandbox.projects.common import context_managers
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.build import dependency_banner
from sandbox.projects.common.build import streaming_client
from sandbox.projects.common.sdk_compat import task_helper
from sandbox.projects.common.vcs import aapi

re_art = re.compile(
    r'^(?P<path>[/a-zA-Z0-9_\-.]+)(?:=(?P<dest>[/a-zA-Z0-9_\-.]+))?$')
re_dart_test_data = re.compile(
    r'^TEST-DATA: "?(?P<data>[/a-zA-Z0-9_\-.; ]+)"?$', re.M)
re_dart_test_data_sep = re.compile(
    r'[; ]+')
re_target_android = re.compile(
    r'--target-platform=\w+-android-\w+'
)

BUILD_DIR = 'build'
PACK_DIR = 'pack'
DEFAULT_RESULT_DESCRIPTION = 'Arcadia Project'

GET_ARCADIA_USE_YA_CLONE = 0
GET_ARCADIA_USE_SANDBOX_CACHE = 1
GET_ARCADIA_USE_AAPI_FUSE = 2
GET_ARCADIA_USE_ARC_VCS = 3
GET_ARCADIA_USE_FROZEN_REPOSITORY_PACKAGE = 4
GET_ARCADIA_USE_ARC_FUSE = 5

TEST_SIZES = ('small', 'medium', 'fat', 'large')
STREAM_TYPES = ('style', 'configure', 'build', 'test')

GROUPS_WITH_FORCE_ARC = {
    "SANDBOX", "OTHER", "guest", "aripinen", "AUTOBUDGET", "ADVQUALITY", "AB-TESTING", "KWYT", "BANNERLAND",
    "RTMR-DEV", "STORAGE-DEV", "NEWS_TESTS", "SEARCH-RELEASERS", "NEURAL-NET"
}
TASKS_WITH_FORCE_ARC = {"YA_MAKE", "YA_PACKAGE", "BUILD_NEURAL_NET"}


def uniq_string_generator(size=6, chars=string.ascii_lowercase + string.digits):
    return lambda: ''.join(random.choice(chars) for _ in range(size))


gen16 = uniq_string_generator(16)


def _get_random_uid(prefix=None):
    return "-".join(filter(None, ["rnd", str(prefix), gen16()]))


def _get_random_hid():
    random.seed(time.time())
    return random.randint(0, (1 << 64) - 1)


def build_error_result(task, task_id, snippet, error_type, status):
    test_info = task_helper.input_or_ctx_field(task, "expected_test_info", "{}")
    suite_path = task_helper.input_or_ctx_field(task, "targets", "{}")

    test_info = json.loads(test_info)
    owners = test_info.get("owners", {})
    return [{
        "id": test_info.get("suite_id", "rnd_suite_id-{}".format(_get_random_uid(task_id))),
        "hid": test_info.get("suite_hid", _get_random_hid()),
        "toolchain": test_info.get("toolchain", "unknown_toolchain"),
        "path": suite_path,
        "type": "test",
        "size": test_info.get("size", "fat"),
        "suite": True,
        "name": test_info.get("suite_name", "unknown_suite"),
        "status": status,
        "suite_status": status,
        "error_type": error_type,
        "rich-snippet": snippet,
        "uid": "sb-{}".format(_get_random_uid(task_id)),
        "owners": owners,
        "tags": test_info.get("tags", ["ya:external"]),
        "links": {
            "task": ["https://sandbox.yandex-team.ru/task/{}/view".format(task_id)],
        },
    }]


def is_required_multislot(client_tags):
    require_multislot = any(
        (ctc.Tag.MULTISLOT in p and ctc.Tag.MULTISLOT not in n) for p, n in
        ctc.Tag.Query.predicates(client_tags.strip().upper())
    )
    return require_multislot


def get_local_output_dir(task):
    dir = task_helper.ctx_field(task, consts.LOCAL_OUTPUT_DIR, True)
    if task_helper.ctx_field(task, consts.YA_YT_REPLACE_RESULT, False):
        dir = False
    return dir


def get_build_type(task):
    return task_helper.input_or_ctx_field(task, consts.BUILD_TYPE_KEY, consts.DEFAULT_BUILD_TYPE)


def get_build_def_flags(task):
    return task_helper.input_or_ctx_field(task, consts.DEFINITION_FLAGS_KEY, '')


def get_target_platform(task):
    return task_helper.input_or_ctx_field(task, consts.TARGET_PLATFORM_KEY)


def get_target_platform_flags(task):
    flags = task_helper.input_or_ctx_field(task, consts.TARGET_PLATFORM_FLAGS, '')
    if len(task.get_binaries_for_ya_bloat()) > 0:
        flags = ' '.join([flags, '--target-platform-flag=DUMP_LINKER_MAP'])
    return flags


def get_host_platform_flags(task):
    return task_helper.input_or_ctx_field(task, consts.HOST_PLATFORM_FLAGS, '')


def get_output_dir(task):
    return task_helper.input_or_ctx_field(task, consts.OUTPUT_DIR_PATH)


def get_build_path(task):
    if task_helper.is_sdk2(task):
        abs_path = str(task.path())
    else:
        abs_path = channel.channel.task.abs_path("")
    build_path = sdk2.paths.get_unique_file_name(abs_path, BUILD_DIR)
    return build_path, sdk2.paths.make_folder(build_path)


def get_yt_store_vault_data(task):
    """
    :return YT token on success, `None` on YAV/Sandbox Vault operation error
    """
    owner_or_name = task_helper.input_or_ctx_field(task, consts.YA_YT_TOKEN_VAULT_OWNER)
    name = task_helper.input_or_ctx_field(task, consts.YA_YT_TOKEN_VAULT_NAME)

    yav_secret = task_helper.input_or_ctx_field(task, consts.YA_YT_TOKEN_YAV_SECRET)
    try:
        if yav_secret:
            if isinstance(yav_secret, six.string_types):
                # sdk1-only spike: decode manually
                yav_secret = sdk2.yav.Secret.__decode__(yav_secret)

            logging.debug(
                'yav secret from param `%r`, `%r`',
                yav_secret,
                getattr(yav_secret, 'default_key', 'NO_ATTRIBUTE'),
            )
            # secret = sdk2.yav.Secret(yav_secret)
            # secret_data = secret.value()
            secret_data = yav_secret.value()
            logging.info('Successfully obtained YT store token from YAV [%s]', yav_secret)
            return secret_data

        owner = owner_or_name or task.owner
        key_name = name or 'YT_STORE_TOKEN'
        secret_data = sdk2.Vault.data(owner, key_name)
        logging.info('Successfully obtained YT store token from Sandbox Vault [%s, %s]', owner, key_name)
        return secret_data

    except (common.errors.VaultError, sdk2.yav.Yav.Error) as ex:
        logging.warning('Unable to get YT token from vault, YT store disabled: %r', ex)

    return None


def generate_fake_results(task, snippet, filename):
    data = {
        "results": build_error_result(task, task.id, snippet, "REGULAR", "FAILED"),
    }
    with open(filename, "w") as afile:
        json.dump(data, afile)


def get_resource_path(task, resource):
    if task_helper.is_sdk2(task):
        return str(resource.path)
    return resource.abs_path()


def mark_resource_ready(task, resource):
    if task_helper.is_sdk2(task):
        resource.ready()
    else:
        task.mark_resource_ready(resource.id)


def append_description(task, data):
    if task_helper.is_sdk2(task):
        task.Parameters.description += data
    else:
        task.description += data


def publish_results(task, resource, filename, check_rc=True, html_results_resource=None):
    logging.debug("Calling publish_results %s", locals())
    if not resource:
        return
    # Наличие и корректность results.json проверяется только в
    # не пользовательском режиме (check_return_code == False)
    if os.path.exists(filename):
        logging.debug("Results.json size: %s", os.path.getsize(filename))
        if os.path.getsize(filename) > 50:
            path = get_resource_path(task, resource)
            with open(filename) as src, open(path, "w") as dst:
                shutil.copyfileobj(src, dst)
            mark_resource_ready(task, resource)
        else:
            logging.info("Results.json seems to be corrupted (extremely small size): {}".format(filename))

        if html_results_resource:
            try:
                from sandbox.projects.devtools.reports import html_report

                html_data = html_report.gen_html_from_results(filename)
            except ImportError:
                doc_url = "https://wiki.yandex-team.ru/sandbox/tasks/binary"
                html_data = "<h3>Failed to generate report - looks like it's not a binary task</h3>" \
                    "Read more about binary tasks: <a href='{doc_url}'>{doc_url}</a>".format(
                        doc_url=doc_url,
                    )

            path = get_resource_path(task, html_results_resource)
            with open(path, "w") as afile:
                afile.write(html_data)

            mark_resource_ready(task, html_results_resource)
    else:
        logging.info("Results.json doesn't exist: {}".format(filename))
        if not check_rc:
            raise Exception('cannot find file results.json')


def report_streaming_results(streaming_link, streaming_check_id, results, stream_partition):
    client = streaming_client.StreamingClient(streaming_link, streaming_check_id, stream_partition)
    client.send_chunk(results['results'])

    for size in TEST_SIZES:
        client.close_stream_by_size('test', size)

    for tp in STREAM_TYPES:
        client.close_stream(tp)

    client.close()


def get_check_deps(task):
    """
        Dump dependency list and validate it
    """
    return task_helper.input_field(task, dependency_banner.CHECK_DEPS_KEY, False)


def check_aapi_available(task):
    try:
        return arcadia_sdk.wait_aapi_url(task_helper.ctx_field(task, consts.ARCADIA_URL_KEY, ''))
    except aapi.ArcadiaApiCommandFailed as e:
        logging.warning("AAPI availability check failed: %r", e)
        return False


def is_arc_path(task):
    return arcadia_sdk.is_arc_path(task_helper.ctx_field(task, consts.ARCADIA_URL_KEY, ''))


def _get_checkout(task):
    """
        Выкачиваем исходники и ставим ссылку room/arcadia.
    """
    if task_helper.is_sdk2(task):
        checkout_auto = task.Parameters.checkout_mode == consts.CHECKOUT_MODE_AUTO
        if checkout_auto:
            parsed_url = svn.Arcadia.parse_url(task.Parameters.checkout_arcadia_from_url)
            checkout = not parsed_url.trunk and not task.Parameters.arcadia_patch
        else:
            checkout = task.Parameters.checkout
        logging.info("Use selective checkout: %s", checkout)
    else:
        # Патч работает с селективным чекаутом, только если это прямая ссылка на зипатч
        checkout = task.should_checkout()
    return checkout


def check_dist_build_parameters(task, build_system, make_context_on_distbuild):
    if (
        make_context_on_distbuild
        and build_system not in (
            consts.DISTBUILD_BUILD_SYSTEM,
            consts.DISTBUILD_FORCE_BUILD_SYSTEM,
        )
    ):
        raise errors.SandboxTaskFailureError(
            'MakeContextOnDistbuild can be used only with distbuild and force_distbuild systems for now'
        )
    if make_context_on_distbuild and get_check_deps(task):
        raise errors.SandboxTaskFailureError('MakeContextOnDistbuild can not be used with check_dependencies parameter')


def check_parameters_compatibility(task, make_context_on_distbuild):
    use_aapi_fuse = task_helper.input_field(task, consts.USE_AAPI_FUSE)
    use_arc_instead_of_aapi = task_helper.input_field(task, consts.USE_ARC_INSTEAD_OF_AAPI)
    aapi_fallback = task_helper.input_field(task, consts.ALLOW_AAPI_FALLBACK)
    use_arc_vcs = is_arc_path(task)
    use_fuse = use_aapi_fuse or use_arc_vcs
    frepkage = task_helper.input_field(task, consts.ARCADIA_FREPKAGE)

    if frepkage:
        return False, GET_ARCADIA_USE_FROZEN_REPOSITORY_PACKAGE

    if make_context_on_distbuild:
        return False, GET_ARCADIA_USE_YA_CLONE

    if use_aapi_fuse and aapi_fallback and not use_arc_vcs and not use_arc_instead_of_aapi:
        if not task.check_aapi_available():
            use_aapi_fuse = False
            logging.info('Disable AAPI, because service is temporary unavailable or inapplicable')

    fuse_available = None
    if use_fuse:
        fuse_available = arcadia_sdk.fuse_available()
        if not fuse_available:
            raise errors.SandboxTaskFailureError('Fuse is not available on {} yet'.format(platform.platform()))

    logging.info(
        'Mount = %s (using %s)',
        use_fuse, 'Arc VCS' if use_arc_vcs else ('Arc Fuse' if use_arc_instead_of_aapi else 'AAPI')
    )

    if use_fuse:
        checkout = False
    else:
        checkout = _get_checkout(task)

    if use_arc_vcs:
        mode = GET_ARCADIA_USE_ARC_VCS
    elif use_aapi_fuse:
        if use_arc_instead_of_aapi:
            mode = GET_ARCADIA_USE_ARC_FUSE
        else:
            mode = GET_ARCADIA_USE_AAPI_FUSE
    elif checkout:
        mode = GET_ARCADIA_USE_YA_CLONE
    else:
        if task.type in TASKS_WITH_FORCE_ARC and task.owner in GROUPS_WITH_FORCE_ARC:
            if fuse_available is None:
                fuse_available = arcadia_sdk.fuse_available()
            if fuse_available:
                logging.info("Try to use arc instead of Sandbox cache")
                mode = GET_ARCADIA_USE_ARC_FUSE
            else:
                mode = GET_ARCADIA_USE_YA_CLONE
        else:
            mode = GET_ARCADIA_USE_SANDBOX_CACHE

    return checkout, mode


def clone(task):
    source_dir_origin = arcadia_sdk.do_clone(task_helper.ctx_field(task, consts.ARCADIA_URL_KEY), task)
    report_config_path = task_helper.ctx_field(task, consts.REPORT_CONFIG_PATH)

    if report_config_path:
        svn.Arcadia.update(os.path.join(source_dir_origin, report_config_path), parents=True)
    return source_dir_origin


def get_source_dirs(task, mode, arc_oauth_token=None, fetch_all=False):
    if mode == GET_ARCADIA_USE_YA_CLONE:
        return (
            context_managers.nullcontext(clone(task)),
            context_managers.nullcontext(None),
        )

    elif mode == GET_ARCADIA_USE_SANDBOX_CACHE:
        return (
            context_managers.nullcontext(task.get_arcadia_src_dir()),
            context_managers.nullcontext(None),
        )

    elif mode == GET_ARCADIA_USE_AAPI_FUSE or mode == GET_ARCADIA_USE_ARC_FUSE:
        arcadia_url = task_helper.ctx_field(task, consts.ARCADIA_URL_KEY, '')
        atd_url = os.path.normpath(svn.Arcadia.append(arcadia_url, os.path.join('..', 'arcadia_tests_data')))
        use_arc_instead_of_aapi = (mode == GET_ARCADIA_USE_ARC_FUSE)
        minimize_arc_mount_path = task_helper.input_field(task, consts.MINIMIZE_ARC_MOUNT_PATH, False)
        return (
            arcadia_sdk.mount_arc_path(
                arcadia_url,
                use_arc_instead_of_aapi=use_arc_instead_of_aapi,
                arc_oauth_token=arc_oauth_token,
                fetch_all=fetch_all,
                minimize_mount_path=minimize_arc_mount_path,
            ),
            arcadia_sdk.mount_arc_path(
                atd_url,
                use_arc_instead_of_aapi=use_arc_instead_of_aapi,
                arc_oauth_token=arc_oauth_token,
                fetch_all=fetch_all,
                minimize_mount_path=minimize_arc_mount_path,
            ) if svn.Arcadia.check(atd_url) else context_managers.nullcontext(None)
        )

    elif mode == GET_ARCADIA_USE_ARC_VCS:
        arcadia_url = task_helper.ctx_field(task, consts.ARCADIA_URL_KEY, '')
        minimize_arc_mount_path = task_helper.input_field(task, consts.MINIMIZE_ARC_MOUNT_PATH, False)
        return (
            arcadia_sdk.mount_arc_path(
                arcadia_url,
                fetch_all=fetch_all,
                minimize_mount_path=minimize_arc_mount_path,
            ),
            context_managers.nullcontext(None),
        )

    elif mode == GET_ARCADIA_USE_FROZEN_REPOSITORY_PACKAGE:
        frepkage_root = arcadia_sdk.setup_frepkage(task_helper.ctx_field(task, consts.ARCADIA_FREPKAGE))
        task_helper.ctx_field_set(task, consts.ARCADIA_FREPKAGE_ROOT, frepkage_root)
        return (
            context_managers.nullcontext(os.path.join(frepkage_root, 'arcadia')),
            context_managers.nullcontext(None),
        )

    else:
        raise Exception('Unknown mode %s', mode)


def ensure_link(path, link):
    if os.path.exists(link):
        if not os.path.islink(link):
            raise errors.SandboxTaskUnknownError('Cannot create symlink, non-link already exists: ' + link)
        os.remove(link)
    fs.make_symlink(path, link)


def get_targets(task):
    return string_lib.parse_list_exact(task_helper.input_field(task, 'targets', ''))


def parse_arts(value):
    arts = []
    for art in string_lib.parse_list_exact(value):
        matched_art = re_art.match(art)
        if not matched_art:
            continue
        art = matched_art.groupdict()
        if art['dest'] is None:
            art.pop('dest')
        arts.append(art)
    return arts


def get_arts(task):
    return parse_arts(task_helper.input_field(task, 'arts', ''))


def get_arts_source(task):
    return parse_arts(task_helper.input_field(task, 'arts_source', ''))


def pack_arts(arts, pack_dir, base_dir):
    logging.info('pack_arts: base_dir: %s', base_dir)
    for art in arts:
        logging.info('pack_arts: processing art\n%s', art)  # RMDEV-2344
        glob_path = os.path.join(base_dir, art['path'])
        logging.info('pack_arts: glob by `%s`', glob_path)  # RMDEV-2344
        art_paths = glob.glob(glob_path)
        if not art_paths:
            raise errors.SandboxTaskFailureError('Artifact(s) not found: ' + art['path'])
        for path in art_paths:
            filename = os.path.basename(path)
            dest = art.get('dest', '')
            sdk2.paths.copy_path(
                path, os.path.join(pack_dir, dest, filename),
                copy_function=arcadia_sdk.hardlink_or_copy,
            )


def get_binaries_for_ya_bloat(task):
    binaries = task_helper.input_field(task, consts.USE_YA_BLOAT, '')
    binaries = binaries.strip(';')
    return binaries.split(';') if binaries else []


def post_build(task, source_dir, output_dir):
    binaries = task.get_binaries_for_ya_bloat()
    arcadia_sdk.do_bloat(source_dir, output_dir, binaries)


def update_atd(task, checkout, source_dir, targets, clear_build, build_system, use_dev_version):
    if checkout:
        logging.info("Perform selective checkout before dumping tests")
        arcadia_sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM, source_dir, targets, build_threads=0, checkout=True, clear_build=clear_build
        )

    if build_system in [consts.DISTBUILD_FORCE_BUILD_SYSTEM]:
        logging.info(
            "Don't obtain arcadia_tests_data - distbuild mode doesn't required atd paths to build graph")
    else:
        atd_paths = arcadia_sdk.get_required_atd_files(source_dir, targets, use_dev_version, clear_build)
        logging.debug("Paths to arcadia_tests_data: %s", atd_paths)
        if atd_paths:
            parsed_url = urlparse.urlparse(task_helper.ctx_field(task, consts.ARCADIA_URL_KEY))
            if parsed_url.scheme == svn.Arcadia.ARCADIA_HG_SCHEME:
                raise errors.SandboxTaskFailureError("arcadia_tests_data is not supported in Arcadia-Hg")
            elif parsed_url.scheme == svn.Arcadia.ARCADIA_ARC_SCHEME:
                raise errors.SandboxTaskFailureError("arcadia_tests_data is not supported in Arcadia Arc VCS")

            arcadia_sdk.update_atd(atd_paths)
            return arcadia_sdk.get_atd_path()
        else:
            logging.debug("Tests don't require arcadia_tests_data")


def get_resources(task):
    arts = (
        parse_arts(task_helper.input_or_ctx_field(task, 'arts', ''))
        + parse_arts(task_helper.input_or_ctx_field(task, 'arts_source', ''))
    )
    if arts:
        project = {}
        if task_helper.input_or_ctx_field(task, "result_resources_types"):
            resources_types = string_lib.parse_list_exact(task_helper.input_or_ctx_field(task, "result_resources_types"))
            if len(resources_types) != len(arts):
                raise errors.SandboxTaskFailureError("Arts count should match result_resources_types count")
            for index, art, resource_type in zip(range(len(arts)), arts, resources_types):
                project[resource_type + "_art_" + str(index)] = {
                    "resource_type": resource_type,
                    "description": task_helper.input_or_ctx_field(
                        task,
                        "result_rd",
                        "{res_type} built from task {task_id}".format(res_type=resource_type, task_id=task.id)
                    ),
                    "resource_path": os.path.basename(arts[index]["path"]),
                }
            return project

        if task_helper.input_or_ctx_field(task, 'result_rt'):
            project['resource_type'] = task_helper.input_or_ctx_field(task, 'result_rt')
        if task_helper.input_or_ctx_field(task, 'result_rd'):
            project['description'] = task_helper.input_or_ctx_field(task, 'result_rd')
        if task_helper.input_or_ctx_field(task, 'result_single_file'):
            if len(arts) > 1:
                raise errors.SandboxTaskFailureError('Multiple artifacts detected in single-file artifact mode')
            if 'dest' in arts[0]:
                raise errors.SandboxTaskFailureError('Artifact destination set in single-file artifact mode')
            project['resource_path'] = os.path.basename(arts[0]['path'])
        return {'project': project}
    return {}


def process_resources(task, resources):
    for resource_name, params in six.iteritems(resources):
        assert type(params) == dict
        params.setdefault('description', DEFAULT_RESULT_DESCRIPTION)
        params.setdefault('resource_path', '')
        if task_helper.is_sdk2(task):
            params.setdefault('resource_type', sdk2.Resource['ARCADIA_PROJECT'])
        else:
            params.setdefault('resource_type', resource_types.ARCADIA_PROJECT)
        if os.path.isabs(params['resource_path']):
            raise errors.SandboxTaskUnknownError(
                'Resource paths in `get_resources` must be relative and match arts paths'
            )
        params['resource_path'] = os.path.join(PACK_DIR, params['resource_path']).rstrip(os.sep)
        task_helper.ctx_field(task, 'ap_packs')[resource_name] = task.create_resource(**params).id


def get_resources_attrs(task=None):
    attrs = {}

    if not task:
        return attrs

    attrs['project'] = {}
    if task_helper.input_or_ctx_field(task, 'result_ttl'):
        val = task_helper.input_or_ctx_field(task, 'result_ttl').lower()
        attrs['project']['ttl'] = val if val == 'inf' else int(val)

    result_attrs = task_helper.input_or_ctx_field(task, 'result_attrs', {})
    if not isinstance(result_attrs, dict):
        result_attrs = json.loads(result_attrs)
    attrs['project'].update(result_attrs)
    return attrs


def get_ram_drive_path(task):
    """
        Получение пути до RAM диска на основе tmpfs
        :return: путь до диска (PosixPath)
    """
    if task.ramdrive:
        return task.ramdrive.path
    return None


def is_build_bundle_task(task):
    return bool(task_helper.input_field(task, consts.BUILD_BUNDLE_KEY))


def finalize_resources(task):
    base_attrs = task.get_base_resource_attrs()
    base_attrs['build_platform'] = task_helper.ctx_field(task, 'ap_build_platform')
    base_attrs['build_tool'] = task_helper.input_field(task, consts.BUILD_SYSTEM_KEY)
    if task_helper.input_field(task, consts.USE_SYSTEM_PYTHON):
        base_attrs['with_system_python'] = str(True)
    resources_attrs = task.get_resources_attrs()
    for resource_name, resource_id in six.iteritems(task_helper.ctx_field(task, 'ap_packs', {})):
        attrs = base_attrs.copy()
        resource_attrs = resources_attrs.get('project')
        if resource_attrs:
            assert type(resource_attrs) == dict
            attrs.update(resource_attrs)
        if task_helper.is_sdk2(task):
            resource = sdk2.Resource[resource_id]
            for attr_name, attr_value in six.iteritems(attrs):
                try:
                    setattr(resource, attr_name, attr_value)
                except AttributeError:
                    if attr_value is None:
                        logging.debug('Skip attribute: %s', attr_name)
                    else:
                        logging.debug('Try to set name: %s, value: %s', attr_name, attr_value)
                        task.server.resource[resource_id].attribute({"name": attr_name, "value": attr_value})
                        logging.debug('Attribute {"name": %s, "value": %s} is OK', attr_name, attr_value)
            sdk2.ResourceData(resource).ready()
        else:
            for attr_name, attr_value in six.iteritems(attrs):
                channel.channel.sandbox.set_resource_attribute(resource_id, attr_name, attr_value)
            task.mark_resource_ready(resource_id)


def create_default_file(abs_path):
    with open(abs_path, 'w') as afile:
        afile.write("{}")
