import json
import logging
from six.moves import urllib


_STREAMING_REQUEST_TIMEOUT = 180
_CHUNK_SIZE = 10000


def http_patch(url, data, headers=None, timeout=30):
    logging.debug('PATCH request using urllib2 %s, %s bytes', url, len(data))
    req = urllib.request.Request(url, data, headers=headers)
    req.get_method = lambda: 'PATCH'
    return urllib.request.urlopen(req, timeout=timeout).read()


def http_post(url, data, headers=None, timeout=30):
    logging.debug('POST request using urllib2 %s, %s bytes', url, len(data))
    req = urllib.request.Request(url, data, headers=headers)
    return urllib.request.urlopen(req, timeout=timeout).read()


class StreamingClient(object):
    _logger = logging.getLogger('StreamingClient')

    def __init__(self, url, stream_id, stream_partition):
        self._url = url
        self._stream_id = stream_id
        self._stream_partition = stream_partition

    def send_chunk(self, entries):
        def chunks(lst, size):
            while lst:
                yield lst[:size]
                lst = lst[size:]

        for chunk in chunks(entries, _CHUNK_SIZE):
            self._send_chunk(chunk)

    def _send_chunk(self, entries):
        url = '{}/chunks/?{}'.format(self._url, urllib.parse.urlencode({'check_id': self._stream_id, 'stream_partition': self._stream_partition}))
        self._logger.debug('Send %s entries to url %s', len(entries), url)
        try:
            response = http_post(url, json.dumps({'results': entries}), {'Content-Type': 'application/json'}, timeout=_STREAMING_REQUEST_TIMEOUT)
        except urllib.error.HTTPError as e:
            logging.error("HTTPError: %s", e.read())
            raise
        self._logger.debug('The chunk is sent to %s, response is %s', url, response)

    def close_stream(self, stream_type):
        url = '{}/checks/{}?{}'.format(self._url, self._stream_id, urllib.parse.urlencode({'stream_partition': self._stream_partition}))
        self._logger.debug('Close stream %s by url %s', stream_type, url)
        message = json.dumps({"action": "finish_test_type", "test_type": stream_type})
        response = http_patch(url, message, {'Content-Type': 'application/json'}, timeout=_STREAMING_REQUEST_TIMEOUT)
        self._logger.debug('The stream %s is closed by url %s, response is %s', stream_type, url, response)

    def close_stream_by_size(self, stream_type, size):
        url = '{}/checks/{}?{}'.format(self._url, self._stream_id, urllib.parse.urlencode({'stream_partition': self._stream_partition}))
        self._logger.debug('Close stream %s with size %s by url %s', stream_type, size, url)
        message = json.dumps({"action": "finish_test_type_size", "test_type": stream_type, "test_size": size})
        response = http_patch(url, message, {'Content-Type': 'application/json'}, timeout=_STREAMING_REQUEST_TIMEOUT)
        self._logger.debug('The stream %s with size %s is closed by url %s, response is %s', stream_type, size, url, response)

    def close_substream(self, tp, toolchain, size):
        url = '{}/checks/{}?{}'.format(self._url, self._stream_id, urllib.parse.urlencode({'stream_partition': self._stream_partition}))
        self._logger.debug('Close substream by url %s for type %s, toolchain %s, size %s', url, tp, toolchain, size)
        message = json.dumps({"action": "finish_stream", "toolchain": toolchain, "test_type": tp, "test_size": size})
        response = http_patch(url, message, {'Content-Type': 'application/json'}, timeout=_STREAMING_REQUEST_TIMEOUT)
        self._logger.debug('The substream is closed by url %s for type %s, toolchain %s, size %s, response is %s', url, tp, toolchain, size, response)

    def close(self):
        url = '{}/checks/{}?{}'.format(self._url, self._stream_id, urllib.parse.urlencode({'stream_partition': self._stream_partition}))
        self._logger.debug('Close check by url %s', url)
        response = http_patch(url, json.dumps({'action': 'finish'}), {'Content-Type': 'application/json'}, timeout=_STREAMING_REQUEST_TIMEOUT)
        self._logger.debug('The stream is closed by url %s, response is %s', url, response)
