# -*- coding: utf-8 -*-
import json
import os
import shutil
import tarfile

import sandbox.common as common
import sandbox.common.mds as mds
import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters
import sandbox.projects.common.build.BuildArcadiaProjectsForAll as bapfa
import sandbox.projects.resource_types as resource_types


class BuildArcadiaProjectsForAll(bapfa.BuildArcadiaProjectsForAll):

    class Parameters(bapfa.BuildArcadiaProjectsForAll.Parameters):
        group_by_platform = parameters.Bool('Group result artifacts into archives by platform', default_value=True, required=True)
        resource_description = parameters.String('Resource description')

    def _on_process_finished_tasks(self):
        if self.Parameters.group_by_platform:
            self.save_resources()
        super(BuildArcadiaProjectsForAll, self)._on_process_finished_tasks()

    def save_resources(self):
        by_platform = {"data": {}}
        resource_type = resource_types.ARCADIA_PROJECT_TGZ
        platforms = {}
        for task in self.find():
            sub_resources = resource_type.find(task=task).limit(10)

            if not sub_resources.count:
                raise Exception("Task {} does not have expected {} resources".format(task.id, resource_type))

            for resource in sub_resources:
                subresource_platform = resource.platform
                if subresource_platform not in platforms:
                    platforms[subresource_platform] = {}

                src_path = str(sdk2.ResourceData(resource).path)
                dst_path = str(self.path("{}/{}".format(subresource_platform, os.path.basename(src_path))))
                dst_dir = os.path.dirname(dst_path)
                if not os.path.exists(dst_dir):
                    os.makedirs(dst_dir)

                if tarfile.is_tarfile(src_path):
                    with tarfile.open(src_path, 'r:gz') as tar:
                        tar.extractall(path=dst_dir)
                else:
                    if os.path.isdir(src_path):
                        shutil.copytree(src_path, dst_dir)
                    else:
                        shutil.copy(src_path, dst_dir)

                platforms[subresource_platform].update(dict((k, v) for k, v in resource))

        for platform in platforms:
            attrs = platforms[platform]
            src_path = str(self.path(platform))
            tar_name = '{}-{}.tgz'.format(self.Parameters.resource_description, platform)
            grouped_res = resource_type(self, '{} for {}'.format(self.Parameters.resource_description, platform),
                                        tar_name, **attrs)
            res_data = sdk2.ResourceData(grouped_res)
            with tarfile.open(str(res_data.path), 'w:gz') as tar:
                for artifact_name in os.listdir(src_path):
                    tar.add(os.path.join(src_path, artifact_name), arcname=artifact_name)
            res_data.ready()
            grouped_res.reload()

            by_platform['data'][platform] = {
                'url': grouped_res.http_proxy,
                'md5': grouped_res.md5,
                'urls': [grouped_res.http_proxy],
            }

            if self.Parameters.backup_to_mds:
                mds_url = self.save_resource_to_mds(str(res_data.path), grouped_res)
                by_platform['data'][platform]['urls'].append(mds_url)

        by_platform_file = "by_platform.json"
        with open(by_platform_file, "w") as f:
            json.dump(by_platform, f, indent=4)

        mapping_resource = resource_types.PLATFORM_MAPPING(self, '', by_platform_file, ttl='inf' if self.Parameters.do_not_remove_resources else None)

        if self.Parameters.backup_to_mds:
            self.save_resource_to_mds(by_platform_file, mapping_resource)

        sdk2.ResourceData(mapping_resource).ready()

    def save_resource_to_mds(self, path, resource):
        mds_namespace = self.Parameters.mds_namespace
        mds_vault_owner = self.Parameters.mds_token_vault_owner or "SANDBOX"
        mds_vault_name = self.Parameters.mds_token_vault_name or "mds_token"
        mds_token = sdk2.Vault.data(mds_vault_owner, mds_vault_name)
        mds_key = mds.MDS().upload(
            path=path,
            mds_name=os.path.basename(path) + '.' + str(resource.id),
            token=mds_token,
            namespace=mds_namespace,
        )
        if not mds_namespace:
            mds_url = common.config.Registry().client.mds.dl.url + "/" + mds_key
        else:
            mds_url = self.Parameters.mds_download_url + "/" + mds_key

        resource.mds = mds_url
        return mds_url
