# -*- coding: utf-8 -*-
import os

from sandbox import sdk2
from sandbox.projects.common.build import KosherYaMake
from sandbox.projects.common import error_handlers as eh


def format_target_key(resource, source_path):
    return "{res_type}__{source_path}".format(res_type=str(resource), source_path=source_path)


def get_project_params(build_resources=None, source_resources=None):
    """
        Generate checkboxes in task Parameters class for given build_resources and source_resources.
        They could be chosen when new task is created.
        :param build_resources: list with Sandbox resource classes, e.g. [RpsLimiterBinary, RpsLimiterAgentBinary]
        :param source_resources: list with tuples (sandbox_resource_class, path_to_file/directory), e.g.
        [
            (MatrixConfigTesting, "alice/matrix/configs/testing_config.json"),
            (MatrixConfigProduction, "alice/matrix/configs/production_config.json"),
        ]
    """
    class Parameters(KosherYaMake.KosherYaMake.Parameters):
        ya_make_project_params = None  # remove this params as we build specific target(s)
        resources_to_build = build_resources or []
        resources_from_sources = source_resources or []
        with sdk2.parameters.CheckGroup("Resources to build") as target_resources:
            for resource in resources_to_build:
                setattr(target_resources.values, str(resource), target_resources.Value(str(resource), checked=True))
            for resource, source_path in resources_from_sources:
                setattr(
                    target_resources.values,
                    format_target_key(resource, source_path),
                    target_resources.Value(
                        "{res_type} with source {source_path}".format(res_type=str(resource), source_path=source_path),
                        checked=True,
                    )
                )
    return Parameters


class YaMakeTemplate(KosherYaMake.KosherYaMake):
    """
        Класс для простого формирования сборочных задач.
        Вместо выбора таргетов - выбор собираемых ресурсов.
        Отдельно запускать эту задачу не стоит.
        Пример переопределения и создания сборочной задачи:

        class BuildMiddlesearch(YaMakeTemplate):
            class Parameters(get_project_params([MiddlesearchBinary, GdbToolkitBinary])):
                pass

        Для корректной работы у передаваемых в функцию get_project_params(...) ресурсов в качестве build_resources
        должен быть определен атрибут arcadia_build_path - таргет сборки.
        Если же необходимо получить ресурс, содержимое которого - файл или директория из Аркадии, необходимо
        воспользоваться параметром source_resources, передав в него пары - (resource, source_path).

        Работающий пример лежит в sandbox/projects/stoker/BuildStokerYa/__init__.py
    """

    class Parameters(get_project_params([])):
        pass

    @property
    def target_resource_classes(self):
        resources = []
        for resource in self.Parameters.resources_to_build:
            if resource in self.Parameters.target_resources:
                resources.append(resource)
        return resources

    @property
    def source_classes(self):
        resources = []
        for resource, source_path in self.Parameters.resources_from_sources:
            if format_target_key(resource, source_path) in self.Parameters.target_resources:
                resources.append((resource, source_path))
        return resources

    def get_resources(self):
        resources = {}
        for resource in self.target_resource_classes:
            resources.update({
                str(resource): {
                    "description": "Build executable",
                    "resource_path": os.path.join(str(resource), os.path.basename(resource.arcadia_build_path)),
                    "resource_type": resource,
                }
            })
        for index, resource in enumerate(self.source_classes):
            resource, file_path = resource
            resources.update({
                "{res_type}_art_{index}".format(res_type=str(resource), index=index): {
                    "description": "Build executable",
                    "resource_path": os.path.join(str(resource), os.path.basename(file_path)),
                    "resource_type": resource,
                }
            })
        return resources

    def get_targets(self):
        return [resource.arcadia_build_path.rstrip("/") for resource in self.target_resource_classes]

    def get_arts_source(self):
        return [
            {
                "path": source_path.rstrip("/"),
                "dest": str(resource),
            }
            for resource, source_path in self.source_classes
        ]

    def get_arts(self):
        return [
            {
                "path": resource.arcadia_build_path.rstrip("/"),
                "dest": str(resource),
            }
            for resource in self.target_resource_classes
        ]

    def on_enqueue(self):
        if len(self.target_resource_classes) == 0 and len(self.source_classes) == 0:
            raise Exception("No resources to build")
        for resource in self.target_resource_classes:
            eh.ensure(
                hasattr(resource, "arcadia_build_path"),
                "Resource {} does not have 'arcadia_build_path' attribute".format(str(resource))
            )
        for resource in self.source_classes:
            eh.ensure(
                len(resource) == 2,
                "Resource {} should be tuple with 2 elements - resource and source_path".format(str(resource))
            )
        super(YaMakeTemplate, self).on_enqueue()
