"""
Common parts of YaMake2 and YaPackage2 tasks
"""
import logging

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2 import service_resources

import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.projects.release_machine.client as client

from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import string
from sandbox.projects.common.build.sdk import sdk_compat as ya_sdk_compat


def _notify_vault_error(task, token_name):
    # notify about failure to Center
    try:
        from release_machine.common_proto import events_pb2
        event = events_pb2.EventData(
            general_data=events_pb2.EventGeneralData(
                hash=string.md5_hexdigest(task.id),
                component_name='release_machine',
                referrer="sandbox_task:{}".format(task.id),
            ),
            custom_message_data=events_pb2.CustomMessageData(
                message="Task {task_link} cannot access {token_name}".format(
                    task_link=lb.task_link(task.id, task.type),
                    token_name=token_name,
                ),
                condition_tag="token_error",
            ),
        )
        rm_client = client.RMClient()
        rm_client.post_proto_events([event, ])
    except Exception as exc:
        eh.log_exception('Cannot report about {} failure'.format(token_name), exc)


def prepare_arc_token(task, mode):
    logging.debug('prepare_arc_token mode: %s', mode)
    if mode != ya_sdk_compat.GET_ARCADIA_USE_ARC_FUSE:
        return None

    logging.debug('prepare_arc_token mode: GET_ARCADIA_USE_ARC_FUSE')
    # GET_ARCADIA_USE_ARC_FUSE mode:
    try:
        return sdk2.Vault.data(task.owner, 'ARC_TOKEN')
    except common.errors.VaultError as ex:
        logging.warning('Unable to get specifed ARC token from vault: %r', ex)

        try:
            return sdk2.Vault.data('SEARCH-RELEASERS', 'ARC_TOKEN')
        except common.errors.VaultError:
            logging.warning(
                'No access to common ARC_TOKEN for owner `%s`, '
                'please ask mvel@ to gain access for specifed owner',
                task.owner,
            )
            _notify_vault_error(task, 'ARC_TOKEN')
    # okay
    return None


def prepare_yt_store_params(task):
    yt_store_token = None

    if task.Parameters.ya_yt_store:
        # first, try to obtain token using given credentials
        yt_store_token = ya_sdk_compat.get_yt_store_vault_data(task)

        if not yt_store_token:
            # RMDEV-2147: Activate fallback mode: try to obtain common token
            try:
                yt_store_token = sdk2.Vault.data('SEARCH-RELEASERS', 'YT_STORE_TOKEN')
            except common.errors.VaultError:
                logging.warning(
                    'No access to common YT_STORE_TOKEN for owner `%s`, '
                    'please ask mvel@ to gain access for specifed owner',
                    task.owner,
                )

        if not yt_store_token:
            _notify_vault_error(task, 'YT_STORE_TOKEN')

    if yt_store_token is not None:
        logging.info("Using YT_STORE_TOKEN with md5 hash `%s`", string.md5_hexdigest(yt_store_token))

    yt_store_params = sdk.YtStoreParams(
        task.Parameters.ya_yt_store,
        yt_store_token,
        task.Parameters.ya_yt_proxy,
        task.Parameters.ya_yt_dir,
        task.Parameters.ya_yt_put,
        task.Parameters.ya_yt_store_codec,
        task.Parameters.ya_yt_replace_result,
        task.Parameters.ya_yt_replace_result_add_objects,
        task.Parameters.ya_yt_replace_result_rm_binaries,
        task.Parameters.ya_yt_max_cache_size,
        task.Parameters.yt_store_exclusive,
        task.Parameters.ya_yt_store_threads,
        task.Parameters.ya_yt_store_refresh_on_read,
    )
    yt_store_params.report_status(task)

    return yt_store_params


class BinaryMixin(object):
    """
    Mixin for automated search of binary resource.
    Class has to be used together with LastBinaryTaskRelease class and its parameters.
    """

    @property
    def binary_executor_query(self):
        if type(self).type in (
            # Redefine selector only for specific task types, see DEVTOOLSSUPPORT-7819 for reference.
            'YA_MAKE_2',
            'YA_PACKAGE_2',
            'KOSHER_YA_MAKE',
            'KOSHER_YA_PACKAGE',
        ):
            return {
                "attrs": {
                    "tasks_bundle": "YA_MAKE_BUNDLE",
                    "released": self.Parameters.binary_executor_release_type,
                },
                "state": [ctr.State.READY, ]
            }
        return super(BinaryMixin, self).binary_executor_query

    def on_create(self):
        if self.Parameters.binary_executor_release_type == "custom":
            return

        # to work correctly on Windows, only the WINDOWS tag needs to be specified, see
        # https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/common/build/YaMake2/__init__.py?rev=8537631#L465
        windows_platform = all(ctc.Tag.WINDOWS in tag_set for tag_set in self.Requirements.client_tags)

        # tasks for darwin should be labeled with one of OSX_* tags.
        darwin_platform = any(all(tag in tag_set for tag_set in self.Requirements.client_tags) for tag in [
            ctc.Tag.OSX_BIG_SUR
        ])

        if windows_platform:
            self.Requirements.tasks_resource = service_resources.SandboxTasksBinary.find(
                owner="SANDBOX",
                attrs={
                    "released": "stable",
                    "task_type": str(self.type),
                    "target_platform": "windows",
                },
            ).first()
        elif darwin_platform:
            self.Requirements.tasks_resource = service_resources.SandboxTasksBinary.find(
                owner="SANDBOX",
                attrs={
                    "released": "stable",
                    "task_type": str(self.type),
                    "target_platform": "darwin",
                },
            ).first()
        else:
            self.Requirements.tasks_resource = service_resources.SandboxTasksBinary.find(
                owner="RELEASE_MACHINE",
                attrs={
                    "tasks_bundle": "YA_MAKE_BUNDLE",
                    "released": "stable",
                },
            ).first()

        super(BinaryMixin, self).on_create()
