# -*- coding: utf-8 -*-

import itertools
import json
import logging
import os

from contextlib import contextmanager
from sandbox import sandboxsdk
from sandbox.projects.common import error_handlers as eh


def extract_test_data_paths(packages, arcadia_path, test_data_prefix):
    test_data = []

    seen_packages = set()
    packages = set(packages)

    while packages:

        package = packages.pop()
        seen_packages.add(package)

        if not os.path.isabs(package):
            eh.ensure(
                not package.startswith('arcadia/'),
                'Package path {} is not relative to arcadia/'.format(package)
            )

            package_file_path = os.path.join(arcadia_path, package)
        else:
            package_file_path = package

        eh.ensure(
            os.path.exists(package_file_path),
            'Package path {} does not exist in arcadia'.format(package),
        )

        try:
            parsed_package = json.load(open(package_file_path))
        except Exception as e:
            raise Exception("Failed to load package {}: {}".format(package_file_path, e))
        for element in parsed_package.get('data', []):
            source = element.get('source', None) or element.get('src', None)
            if source and source.get('type', '').lower() in ['data', 'test_data']:
                path = source.get('path', None)
                if path and path.startswith(test_data_prefix.rstrip('/') + '/'):
                    test_data.append(path)

        for item in parsed_package.get('include', []):
            if isinstance(item, dict):
                include_package = item.get("package")
            else:
                include_package = item
            if include_package and include_package not in seen_packages:
                packages.add(include_package)

    return test_data


def checkout_test_data(test_data, svn_root, arcadia_url, task):
    """
        Checkout subdirectories from arcadia_tests_data and data.

        :param packages: list of json files in ya package format
        :param arcadia_url: url to arcadia svn(possibly in a branch)
        :param svn_root: root directory for arcadia, arcadia_tests_data and data
        :param task: current BUILD_PACKAGE task
    """
    created_data_directories = set()
    for data_directory, path in test_data:
        svn_path = os.path.normpath(sandboxsdk.svn.Arcadia.append(arcadia_url, os.path.join('..', path)))
        sandboxsdk.svn.ArcadiaTestData.get_arcadia_test_data(task, svn_path)

        if data_directory not in created_data_directories:
            data_directory_location = sandboxsdk.svn.ArcadiaTestData.test_data_location(svn_path)[0]
            logging.info('New data directory location %s', data_directory_location)

            symlink_path = os.path.join(svn_root, data_directory)
            os.symlink(data_directory_location, symlink_path)
            logging.info('Symlink created %s -> %s', symlink_path, data_directory_location)

            created_data_directories.add(data_directory)


def _parse_multi_parameter(p):
    return [s for s in itertools.chain.from_iterable(x.split(';') for x in p.split()) if s]


# This function is needed for short period of time after release
# of dbg_resource_id functionality, to allow correct work of task, created
# before release.
def _get_resources_pair(descr):
    return descr if isinstance(descr, (tuple, list)) else (descr, None)


@contextmanager
def custom_env(env):
    orig = os.environ.copy()
    for k, v in env.items():
        os.environ[k] = v
    yield os.environ
    os.environ = orig
