# -*- coding: utf-8 -*-
"""
Calendar API

CAUTION! No Sandbox dependencies allowed

"""

from datetime import datetime, timedelta
import requests


class CalendarApi(object):
    API_URL = 'http://calendar-api.tools.yandex.net/internal'
    API_TIME_FMT = '%Y-%m-%dT%H:%M:%S'

    def _get_request(self, operation, params=None, headers=None, api_timeout=120.0):
        # prepare request
        url = self.API_URL + operation
        if headers is None:
            headers = {
                'Accept': 'application/json',
                'Accept-Encoding': 'gzip, identity'
            }
            if hasattr(self, 'token'):
                headers['Authorization'] = 'OAuth {}'.format(self.token)

        # make request
        try:
            r = requests.get(url, params=params, headers=headers, timeout=api_timeout)
            r.raise_for_status()
        except requests.exceptions.Timeout:
            return None
        except requests.exceptions.RequestException:
            return None
        except IOError:
            return None

        # dump result
        try:
            return r.json()
        except Exception:
            return None

    def _get_event(self, params):
        operation = '/get-events'
        return self._get_request(operation, params)

    def _find_longest_event(self, events):
        res_idx, max_duration = None, timedelta(seconds=0)
        for idx, event in enumerate(events):
            start, end = event['startTs'], event['endTs']
            duration = datetime.strptime(end, self.API_TIME_FMT) - datetime.strptime(start, self.API_TIME_FMT)
            if duration > max_duration:
                max_duration = duration
                res_idx = idx
        return res_idx

    def get_duty(self, calendar_id, day=0):
        now = datetime.now()
        finish_duty_time = now.replace(hour=6, minute=30)
        finish_duty_time += timedelta(days=day-now.weekday())
        params = {
            'from': finish_duty_time.strftime(self.API_TIME_FMT),
            'to': finish_duty_time.strftime(self.API_TIME_FMT),
            'tz': 'Europe/Moscow',
            'layerId': calendar_id
        }
        body = self._get_event(params)
        if not body or 'events' not in body or len(body['events']) < 1:
            return None

        events = body['events']
        if len(events) > 1:
            idx = self._find_longest_event(events)
            event = events[idx]
        else:
            event = events[0]

        if 'attendees' not in event or len(event['attendees']) < 1:
            return None
        return event['attendees'][0]['login']
