# -*- coding: utf-8 -*-

import os
import six
import logging
import re
import tempfile

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import remove_path

from sandbox.projects.common import decorators

logger = logging.getLogger(__name__)


def _make_blinov_request(instance_tag_name, cms_configuration='HEAD'):
    """
        Генерирует запрос в формате Блинова
        :param instance_tag_name instance-tag или список instance-tag-ов
            (если указан список тегов, выполняется пересечение; если в списке
            тегов имеется еще один список, выполняется ИЛИ)
        :param cms_configuration имя конфигурации

        Пример:
            print _make_blinov_request([
                    'a_metaprj_video',
                    'a_itype_mmeta',
                    ['a_ctype_priemka', 'a_ctype_priemkainproduction']
                ])

        Вывод:
            I@a_metaprj_video . I@a_itype_mmeta . [I@a_ctype_priemka I@a_ctype_priemkainproduction] . C@HEAD
    """

    request = []
    if instance_tag_name and isinstance(instance_tag_name, six.string_types):
        request.append("I@" + instance_tag_name)
    elif isinstance(instance_tag_name, (list, tuple)):
        for tag in instance_tag_name:
            if isinstance(tag, six.string_types):
                request.append("I@" + tag)
            elif isinstance(tag, (list, tuple)):
                request.append("[" + " ".join("I@" + t for t in tag) + "]")

    if isinstance(cms_configuration, six.string_types):
        request.append("C@" + cms_configuration)

    return " . ".join(request)


def _cms_list_instances(instance_tag_name, cms_configuration='HEAD'):
    """
        Получить список инстансов для указанного тега (тегов) и конфигурации CMS

        :param instance_tag_name: имя тега инстанса (или список имён тегов)
            (если указан список тегов, выполняется пересечение)
        :param cms_configuration: тип конфигурации CMS
        :return: список имён найденных инстансов
    """
    request = _make_blinov_request(
        instance_tag_name=instance_tag_name,
        cms_configuration=cms_configuration
    )
    logging.debug("Blinov request = '%s'", request)
    return get_instances_by_blinov_req(request)


def get_instances_by_blinov_req(request):
    from library.sky.hostresolver import Resolver
    return [
        instance
        for instance_list in Resolver().resolveInstances(request).itervalues()
        for instance in instance_list
    ]


@decorators.retries(3, delay=60, backoff=3)
def instances_by_host(host):
    host_query = "h@" + host
    logging.info("Resolving instance by '%s'", host_query)
    shards_instances = get_instances_by_blinov_req(host_query)
    if not shards_instances:
        raise SandboxTaskFailureError("No host instances resolved by query {}".format(host_query))
    return shards_instances


def _cms_list_resources(
    instance_name=None,
    instance_tag_name=None,
    cms_configuration='HEAD',
):
    """
        Получить список ресурсов по имени инстанса CMS или по имени тега (тегов) CMS
        (если указан список тегов, выполняется пересечение).

        :param instance_name: имя инстанса
        :param instance_tag_name: имя тега инстанса (или список тегов)
        :param cms_configuration: тип конфигурации CMS
        :return: генератор с найденными ресурсами
    """
    import api.cms
    logger.info(
        "Call CMS listResources method with parameters:\n"
        "conf: %s\n"
        "instanceName: %s\n"
        "instanceTagName: %s",
        cms_configuration, instance_name, instance_tag_name
    )

    if instance_name is not None:
        instances = [instance_name]
    else:
        instances = [
            instance for shard, instance in _cms_list_instances(instance_tag_name, cms_configuration)
        ]

    return [
        resource
        for resource_list in api.cms.Registry.listInstancesResources(cms_configuration, instances).itervalues()
        for resource in resource_list
    ]


def get_shard_hosts(shard_name, cms_configuration='HEAD'):
    """
        Получить список хостов, на которых дожен лежать шард

        :param shard_name: имя шарда
        :param cms_configuration: тип конфигурации CMS
        :return: список хостов, если хосты не найдены, возвращается пустой список
    """
    import api.cms
    logger.info('Get hosts list for shard %s from configuration %s', shard_name, cms_configuration)
    registry_items = list(api.cms.Registry.listShardOnHosts(conf=cms_configuration, shardName=shard_name))
    if registry_items:
        return registry_items[0].hostlist.split(' ')[1:]
    else:
        return []


def get_cms_instances(instance_tag_name, cms_configuration='HEAD'):
    """
        Получить список инстансов для указанного тега (тегов) и конфигурации
        (если указан список тегов, выполняется пересечение с использованием
        калькулятора Блинова)
    """

    result = []

    for shard, instance in _cms_list_instances(instance_tag_name, cms_configuration):
        instance = instance.split('@', 1)[0].split(':')
        result.append({
            'shard': shard,
            'host': instance[0],
            'port': int(instance[1]),
        })

    return result


def get_cms_shards(instance_tag_name, cms_configuration='HEAD'):
    """
        Получить имена шардов по имени тега (списка тегов) из CMS
        (если указан список тегов, выполняется пересечение с использованием
        калькулятора Блинова)

        :param tag_name: название тега
        :param cms_configuration: тип конфигурации CMS
        :result: список имён найденных шардов
    """

    logger.info(
        'Get shards for tag %s from configuration %s',
        instance_tag_name, cms_configuration
    )

    return sorted(set(
        shard for shard, instance in _cms_list_instances(instance_tag_name, cms_configuration)
    ))


def get_cms_resource(
    local_path,
    instance_name=None,
    instance_tag_name=None,
    cms_configuration='HEAD',
    field_patterns=None,
):

    def check_pattern(field_patterns, res):
        for field, pattern in field_patterns.items():
            if not re.match(pattern, res.get(field, '')):
                return False
        return True

    resources = _cms_list_resources(instance_name, instance_tag_name, cms_configuration)

    for r in resources:
        if r['local_path'] != local_path:
            continue

        if field_patterns:
            if not check_pattern(field_patterns, r):
                continue
        return r

    available_paths = set([r['local_path'] for r in resources])
    logging.info("Available local_paths in CMS: %s", ", ".join(available_paths))

    raise SandboxTaskFailureError(
        "Resource with local_path {} not found, "
        "see logs for list of available local_paths".format(local_path)
    )


def is_instance_with_tag(host, port, tag_name, cms_configuration="HEAD"):
    """
        Returns true if instance has specified tag
    """
    import api.cms
    instance_name = "{}:{}".format(host, port)
    instance_tags = [
        tag.name
        for tag in api.cms.Registry.listAllSearchInstanceTags(cms_configuration, host, instance_name)
    ]
    return tag_name in instance_tags


def load_cms_resource(
    local_path,
    instance_name=None,
    instance_tag_name=None,
    cms_configuration='HEAD',
    field_patterns=None,
):
    r = get_cms_resource(
        local_path,
        instance_name=instance_name,
        instance_tag_name=instance_tag_name,
        cms_configuration=cms_configuration,
        field_patterns=field_patterns,
    )
    logger.info('Loading cms resource, remote path: %s', r['remote_path'])
    return _get_remote_file_text(r['remote_path'])


def _get_remote_file_text(remote_path):
    """
        remote_copy не может нормально работать с разными протоколами

        при копировани через rbtorrent нужно чтобы remote_path указывал на директорию

        а при копировании по http под linux remote_path должен указывать на файл
        wget -T 60 -O /place/vartmp/tmpOHVMWD/
            http://cmsearch.yandex.ru/res/gencfg/stable-46-r15/w-generated/all/las1-0597:8038.cfg
        выдает ошибку /place/vartmp/tmpOHVMWD/: Is a directory
    """

    tmpdir = tempfile.mkdtemp() + '/'
    os.chmod(tmpdir, 0o777)

    if remote_path.startswith('rbtorrent'):
        channel.task.remote_copy(remote_path, tmpdir)
        file_path = os.path.join(tmpdir, os.listdir(tmpdir)[0])
    else:
        file_path = os.path.join(tmpdir, 'file.txt')
        channel.task.remote_copy(remote_path, file_path)

    text = open(file_path).read()
    logging.debug("Read %s bytes from %s", len(text), file_path)

    remove_path(tmpdir)

    return text
