# -*- coding: utf-8 -*-

import logging
from itertools import izip_longest
import datetime

__all__ = ['OutputOptions', 'Viewer', 'Aggregator']

infolevels = set([0, 1, 2, 3, 666])


class OutputOptions(object):
    def __init__(self, output, width, infolevel):
        if type(output) is not file:
            raise IOError('incorrect output file')
        if width < 80:
            raise ValueError('make width more than 80. it\'s too tight, srsly')
        if infolevel not in infolevels:
            raise ValueError('unsupported infolevel {0}'.format(infolevel))

        self.output = output
        self.width = width
        self.infolevel = infolevel


class Viewer(object):
    """
        creates viewer functional object
        viewer represents analyse results according to options
    """

    def __init__(self, options):
        self.options = options
        self.infolevel_reporters = {
            0: self.__report0,
            1: self.__report1,
            2: self.__report2,
            3: self.__report3,
            666: self.__report666
        }

    def __call__(self, urls, result):
        self.infolevel_reporters[self.options.infolevel](urls, result)
        return self.__aggregate(urls, result)

    def __print(self, s):
        if type(s) is not str and type(s) is not unicode:
            raise TypeError('tried to print non-string {0}'.format(s))
        self.options.output.write(s + '\n')

    def __report0(self, urls, result):
        pass

    def __report1(self, urls, result):
        def get_query_text(url):
            url = url.replace(url[:url.find('?')+1], '')
            return url
        if result['edit_dist'] > 0:
            self.__print(get_query_text(urls[0]))

    def __report2(self, urls, result):
        if len(result['displaced']) > 0:
            self.__print('=' * self.options.width)
            self.__print(urls[0])
            self.__print(urls[1])
            self.__print('Edit distance is {0}'.format(result['edit_dist']))
            self.__print('Unmatched (unique) results count is {0}'.format(result['unmatched_count']))
            self.__print('Displaced results count is {0}'.format(result['displaced_count']))
            self.__print('')
        for i in result['displaced']:
            self.__two_col_print(
                result['docs'][0][i]['header'],
                result['docs'][1][i]['header'],
                num=i+1
            )
            self.__two_col_print(
                result['docs'][0][i]['url'],
                result['docs'][1][i]['url'],
                splitter='/'
            )
            self.__print('')

    def __report3(self, urls, result):
        self.__print('=' * self.options.width)
        self.__print(urls[0])
        self.__print(urls[1])
        self.__print('Edit distance is {0}'.format(result['edit_dist']))
        self.__print('Unmatched (unique) results count is {0}'.format(result['unmatched_count']))
        self.__print('Displaced results count is {0}'.format(result['displaced_count']))
        self.__print('')
        for i in xrange(len(result['docs'][0])):
            self.__two_col_print(
                result['docs'][0][i]['header'],
                result['docs'][1][i]['header'],
                num=i+1
            )
            self.__two_col_print(
                result['docs'][0][i]['url'],
                result['docs'][1][i]['url'],
                splitter='/'
            )
            self.__print('')

    def __report666(self, urls, result):
        def get_query_text(url):
            url = url.replace(url[:url.find('?') + 1], '')
            return url
        self.__print('{0} {1}'.format(len(result['docs'][0]), get_query_text(urls[0])))

    def __aggregate(self, urls, result):
        return result['edit_dist']

    def __two_col_print(self, s1, s2, num=None, splitter=' '):
        def smart_splitter(string, width):
            res = [[]]
            _len = 0
            for word in string.strip(splitter).split(splitter):
                if _len + len(word) + len(splitter) * (1 + len(res[-1])) <= width:
                    res[-1].append(word)
                    _len += len(word)
                else:
                    res.append([])
                    res[-1].append(word)
                    _len = len(word)
            res = [splitter.join(item) for item in res]
            if len(res[-1]) == 0:
                res.pop()
            if not splitter.isspace():
                res[:-1] = [x + splitter for x in res[:-1]]
            res = map(lambda x: x[:width-3] + '...' if len(x) > width else x, res)
            return res

        width = self.options.width
        if num is None:
            num = ' ' * 4
        col_width = width/2 - 6
        if type(s1) is not unicode:
            try:
                s1 = unicode(s1, 'utf-8')
            except Exception:
                try:
                    s1 = unicode(s1, 'latin-1')
                except Exception:
                    logging.error('Unicode Decode Error: {0}'.format(s1))
        if type(s2) is not unicode:
            try:
                s2 = unicode(s2, 'utf-8')
            except Exception:
                try:
                    s2 = unicode(s2, 'latin-1')
                except Exception:
                    logging.error('Unicode Decode Error: {0}'.format(s2))
        for line1, line2 in izip_longest(
            smart_splitter(s1, col_width),
            smart_splitter(s2, col_width),
            fillvalue=' ' * col_width
        ):
            self.__print(
                u'{i:<4}{0:{width}}    {i:<4}{1:{width}}'.format(line1, line2, i=num, width=col_width)
            )
            num = ' ' * 4


class Aggregator(object):
    """
        creates functional object that generates
        final report on multiple checks
    """

    def __init__(self, options):
        self.options = options

    def __call__(self, views=None):
        if views is None:
            # preliminary operations
            if self.options.infolevel in (1, 2):
                self.options.output.write('Differences found in the following queries...\n\n')
        else:
            # finalizing operations
            self.options.output.write('\nSUMMARY\n')
            diff_count = len(filter(lambda x: x, views))
            total_count = len(views)
            total_dist = sum(views)
            mean_dist = float(sum(views))/len(views)
            mean_nonzero = reduce(lambda x, y: x + int(y > 0), views, )
            if mean_nonzero != 0:
                mean_nonzero = float(sum(views)) / mean_nonzero
            s = 'Differences found in {0} checks out of {1}\n'\
                .format(diff_count, total_count)
            s += 'Total edit distance is {0}\n'.format(round(total_dist, 2))
            s += 'Mean edit distance is {0}\n'\
                .format(round(mean_dist, 2))
            s += 'Mean non-zero edit distance is {0}\n'\
                .format(round(mean_nonzero, 2))
            self.options.output.write(s)

            return {
                'time': datetime.datetime.now().strftime('%Y/%m/%d %H:%M:%S'),
                'diff_count': diff_count,
                'total_count': total_count,
                'total_dist': total_dist,
                'mean_dist': mean_dist,
                'mean_nonzero': mean_nonzero
            }
