# coding=utf-8
import urllib

import requests
from requests.exceptions import Timeout, HTTPError
from abc import ABCMeta, abstractproperty

from sandbox.common.utils import get_task_link
from sandbox.projects.common.decorators import retries

CONDUCTOR_URL = "http://c.yandex-team.ru"
BRANCHES = ["unstable", "testing", "prestable", "stable", "hotfix", "fallback"]

STATUSES_INTERMEDIATE = ["new", "in progress", "test", "in test"]
STATUSES_FINAL = ["done", "need info", "maybe_missing", "missing", "obsolete"]
STATUSES = STATUSES_INTERMEDIATE + STATUSES_FINAL


def task_ticket_comment_generator(task, st_ticket=None):
    """
    Генерирует коммент для кондукторного тикета, с указанием таска и ST-тикета.

    :type st_ticket: str
    :type task: sdk2.Task
    :return str
    """

    comment = "Создан в Sandbox: {}".format(get_task_link(task.id))

    if st_ticket is not None:
        comment += "\nТикет: http://st/{}".format(st_ticket)

    return comment


class Api:
    task = None
    auth = None
    timeout = 5
    session = None

    def __init__(self, task, auth, session=None):
        self.task = task
        self.auth = auth
        self.session = SessionWrap(session if session and isinstance(session, requests.Session) else None)

    def ticket_add(self, packages, branch, remove=None, mailcc=None, skip_restart=None, no_autoinstall=None,
                   projects=None, comment_generator=task_ticket_comment_generator, st_ticket=None, deploy_groups=None,
                   workflows=None):
        if not len(packages):
            raise Exception("No packages")

        if branch not in BRANCHES:
            raise Exception("Incorrect branch: %s" % branch)

        payload = [("ticket[branch]", branch), ("ticket[comment]", comment_generator(self.task, st_ticket))]
        packages_index = {}

        for (index, (package, version)) in enumerate(packages.iteritems()):
            payload.append(("package[%d]" % index, package))
            payload.append(("version[%d]" % index, version))
            packages_index[package] = index

        if remove is not None and isinstance(remove, list):
            for package in (package for package in remove if package in packages_index):
                payload.append(("remove[%d]" % packages_index[package], "1"))

        if mailcc is not None and isinstance(mailcc, list):
            payload.append(("ticket[mailcc]", ",".join(mailcc)))

        if skip_restart:
            payload.append(("ticket[skip_restart]", "1"))

        if no_autoinstall:
            payload.append(("ticket[do_not_autoinstall]", "1"))

        if projects is not None and isinstance(projects, list):
            payload.append(("filters[projects][]", ",".join(projects)))
        elif deploy_groups is not None and isinstance(deploy_groups, list):
            for (index, (project, workflow, deploy_group)) in enumerate(deploy_groups):
                payload.append(("filters[deploy_groups][%d][project]" % index, project))
                payload.append(("filters[deploy_groups][%d][workflow]" % index, workflow))
                payload.append(("filters[deploy_groups][%d][deploy_group]" % index, deploy_group))
        elif workflows is not None and isinstance(workflows, list):
            for (index, (project, workflow)) in enumerate(workflows):
                payload.append(("filters[workflows][%d][workflow]" % index, workflow))
                payload.append(("filters[workflows][%d][project]" % index, project))

        url = self.build_url("/auth_update/ticket_add")

        with self.session as session:
            r = session.post(url, headers=self.auth.headers, cookies=self.auth.cookies,
                             timeout=self.timeout, data=dict(payload))
            r.raise_for_status()

            return r.text

    def package_remove(self, package, version, branch, comment_generator=task_ticket_comment_generator, st_ticket=None):
        self.ticket_add({package: version}, branch, remove=[package], st_ticket=st_ticket,
                        comment_generator=comment_generator)

    def ticket_status(self, ticket_id):
        with self.session as session:
            r = session.get(self.build_url("/api/generator/glader.ticket_status", {"ticket": ticket_id}),
                            timeout=self.timeout)
            r.raise_for_status()

            return r.text

    @retries(max_tries=6, exceptions=(Timeout, HTTPError))
    def get_packages_on_host(self, host):
        with self.session as session:
            r = session.get(self.build_url("/api/packages_on_host/" + host, {"format": "json"}), timeout=self.timeout)
            r.raise_for_status()

            return r.json()

    def get_package_version_on_host(self, host, package_name):
        packages = self.get_packages_on_host(host)
        for package in packages:
            if package["name"] == package_name:
                return package["version"]
        return None

    @staticmethod
    def build_url(path, params=None):
        return "%s%s?%s" % (CONDUCTOR_URL, path, urllib.urlencode(params)) if params else "%s%s" % (CONDUCTOR_URL, path)


class SessionWrap:
    _session = None

    def __init__(self, session=None):
        self._session = session

    def __enter__(self):
        if not self._session:
            self._session = requests.Session().__enter__()

        return self._session

    def __exit__(self, exc_type, exc_val, exc_tb):
        self._session.__exit__(exc_type, exc_val, exc_tb)


class Auth:
    __metaclass__ = ABCMeta

    @abstractproperty
    def headers(self):
        pass

    @abstractproperty
    def cookies(self):
        pass


class SomeAuth(Auth):
    __metaclass__ = ABCMeta

    secret = None

    def __init__(self, secret):
        self.secret = secret


class NoAuth(Auth):
    @property
    def cookies(self):
        return None

    @property
    def headers(self):
        return None


class OauthAuth(SomeAuth):
    @property
    def headers(self):
        return {"Authorization": "OAuth %s" % self.secret}

    @property
    def cookies(self):
        return None


class SessionAuth(SomeAuth):
    @property
    def headers(self):
        return None

    @property
    def cookies(self):
        return {"Session_id": self.secret}


class AuthCookie(SomeAuth):
    @property
    def cookies(self):
        return {"conductor_auth": self.secret}

    @property
    def headers(self):
        return None
