# -*- coding: utf-8 -*-

import logging
import re

from sandbox.projects.common import error_handlers as eh

_RE_TYPE = type(re.compile("", 0))


def get_parameter_values(text, path):
    is_re_path = isinstance(path, _RE_TYPE)
    all_values = []

    class Callbacks:
        def on_empty_line(self):
            pass

        def on_comment(self, line):
            pass

        def on_group_start(self, line, group_name, group_path):
            pass

        def on_group_end(self, line, group_name, group_path):
            pass

        def on_param(self, line, spaces, param_name, value, param_name_with_path, group_path):
            if (is_re_path and path.match(param_name_with_path) or
                    not is_re_path and path == param_name_with_path):
                all_values.append(value)

    _scan(text, Callbacks(), include_group_attributes=is_re_path)

    return all_values


class CallbackCollectLines(object):
    def __init__(self):
        self.lines = []

    def on_empty_line(self):
        self.lines.append('')

    def on_comment(self, line):
        self.lines.append(line)

    def on_group_start(self, line, group_name, group_path):
        self.lines.append(line)

    def on_group_end(self, line, group_name, group_path):
        self.lines.append(line)

    def on_param(self, line, spaces, param_name, value, param_name_with_path, group_path):
        self.lines.append(line)


def parameter_map(func, text, path):
    """
        Применяет func к значению указанного параметра, заменяет параметр возвращенным значением
        Возвращает изменённый конфиг
        Пример добавления переранжирования в параметр конфигуции переранжирований перед остальными опциями
            def add_corrupt_rearrange(val):
                return 'CorruptGrouping(total=true) ' + val
            text = parameter_map(func, text, 'Collection/ReArrangeOptions')
    """
    class Callbacks(CallbackCollectLines):
        def on_param(self, line, spaces, param_name, value, param_name_with_path, group_path):
            if (path == param_name_with_path):
                self.lines.append(param_name + ' ' + func(value))
            else:
                super(Callbacks, self).on_param(line, spaces, param_name, value, param_name_with_path, group_path)

    cb = Callbacks()
    _scan(text, cb)
    return '\n'.join(cb.lines)


def remove_part_sources(text, nparts, ipart):
    """
        Из каждых nparts источников <SearchSource> удаляет один в позиции npart
        (всего конфигурация потеряет ~(1 + 1/npart) источников)
        Возвращается изменённый конфиг
    """
    class CallbacksRmSources(CallbackCollectLines):
        def __init__(self, npart, ipart):
            if ipart >= npart:
                raise Exception('bad arguments')
            super(CallbacksRmSources, self).__init__()
            self.npart = npart
            self.ipart = ipart
            self.curr_part = 0
            self.rm_curr_part = False

        def on_empty_line(self):
            if not self.rm_curr_part:
                super(CallbacksRmSources, self).on_empty_line()

        def on_comment(self, line):
            if not self.rm_curr_part:
                super(CallbacksRmSources, self).on_comment(line)

        def on_group_start(self, line, group_name, group_path):
            if group_name == 'SearchSource':
                if self.curr_part % self.npart == self.ipart:
                    self.rm_curr_part = True
                    return
            super(CallbacksRmSources, self).on_group_start(line, group_name, group_path)

        def on_group_end(self, line, group_name, group_path):
            if not self.rm_curr_part:
                super(CallbacksRmSources, self).on_comment(line)
            if group_name == 'SearchSource':
                if self.curr_part % self.npart == self.ipart:
                    self.rm_curr_part = False
                self.curr_part += 1

        def on_param(self, line, spaces, param_name, value, param_name_with_path, group_path):
            if not self.rm_curr_part:
                super(CallbacksRmSources, self).on_param(line, spaces, param_name, value, param_name_with_path, group_path)

    cb = CallbacksRmSources(nparts, ipart)
    _scan(text, cb)
    return '\n'.join(cb.lines)


def replace_subsources_for_part_sources(text, nparts, ipart, ss='http://localhost:1/yandsearch?@50'):
    """
        В каждых nparts источников <SearchSource> заменяем url подисточника на указанный
        (например для воспроизведенения ситуации неотвечающего источника, -
        всего в конфигурация будет деградировано ~(1 + ~1/nparts) источников)
        Возвращается изменённый конфиг
    """
    class CallbacksReplaceSubSources(CallbackCollectLines):
        def __init__(self, npart, ipart, ss):
            if ipart >= npart:
                raise Exception('bad arguments')
            super(CallbacksReplaceSubSources, self).__init__()
            self.npart = npart
            self.ipart = ipart
            self.curr_part = 0
            self.ss = ss
            self.replace_ss = False

        def on_group_start(self, line, group_name, group_path):
            if group_name == 'SearchSource':
                if self.curr_part % self.npart == self.ipart:
                    self.replace_ss = True
            super(CallbacksReplaceSubSources, self).on_group_start(line, group_name, group_path)

        def on_group_end(self, line, group_name, group_path):
            if group_name == 'SearchSource':
                if self.curr_part % self.npart == self.ipart:
                    self.replace_ss = False
                self.curr_part += 1
            super(CallbacksReplaceSubSources, self).on_comment(line)

        def on_param(self, line, spaces, param_name, value, param_name_with_path, group_path):
            if self.replace_ss and param_name == 'CgiSearchPrefix':
                self.lines.append('CgiSearchPrefix ' + self.ss)
                return
            super(CallbacksReplaceSubSources, self).on_param(line, spaces, param_name, value, param_name_with_path, group_path)

    cb = CallbacksReplaceSubSources(nparts, ipart, ss)
    _scan(text, cb)
    return '\n'.join(cb.lines)


def set_equal_src(text):
    first_url = None
    cur_url = ''
    res = ''
    text += ' '
    has_weight = False
    for c in text:
        if c == '(' or c == ' ' or c == ')' or c == '@' or (has_weight and c.isdigit()):
            if c == '@':
                has_weight = True
            elif not c.isdigit():
                has_weight = False
            if cur_url:
                if first_url is None:
                    first_url = cur_url
                cur_url = ''
                res += first_url
            res += c
        else:
            cur_url += c
    return res.rstrip(' ')


def set_src_subsources_one_name(text):
    """
        Делаем все url-ы подисточников внутри источника одинаковыми
        (уменьшаем нестабильность выдачи в которой могут использоваться адреса подисточников)
    """
    class CallbacksSetIdenticalSubSources(CallbackCollectLines):
        def on_param(self, line, spaces, param_name, value, param_name_with_path, group_path):
            if param_name == 'CgiSearchPrefix':
                self.lines.append('CgiSearchPrefix ' + set_equal_src(value))
                return
            super(CallbacksSetIdenticalSubSources, self).on_param(line, spaces, param_name, value, param_name_with_path, group_path)

    cb = CallbacksSetIdenticalSubSources()
    _scan(text, cb)
    return '\n'.join(cb.lines)


TMP_DIR_PATTERN = "%%%TMPDIR%%%"


def patch_config(text, patched_params):
    """
        Патчит конфиг поиска.

        :param text: текст конфиг-файла
        :param patched_params: список пар параметров, которые надо пропатчить
            параметры могут иметь следующие типы:
            * строка (новое значение параметра)
            * tuple (в качестве нового значения параметра будет использован первый элемент)
            * инстанс класса ParamPatcherByGroup (новое значение параметра будет получено в результате вызова .patch_param(...))
                note: ParamPatcherByGroup сейчас работает только для групп, не содержащих подгрупп
    """
    patched_params = [(n, v) for n, v in patched_params]
    patched_params_by_groups = [(n, v) for n, v in patched_params if isinstance(v, ParamPatcherByGroup)]
    patched_params = [(n, v) for n, v in patched_params if not isinstance(v, ParamPatcherByGroup)]

    logging.info("patching config with params")
    for name, value in patched_params:
        logging.info("'%s'=%s" % (name, "'%s'" % str(value) if value is not None else None))

    class Callbacks:
        def __init__(self, patched_params):
            self.patched_params = []
            self.group_count = {}

            for param, value in patched_params:
                m = re.match(r'(.*)\[(\d+)\]', param)
                if not m:
                    param_name = param
                    target_count = None
                else:
                    param_name = m.group(1)
                    target_count = int(m.group(2))

                self.patched_params.append((param_name, target_count, value))

                if target_count is not None:
                    path_without_param = param_name[:param_name.rfind('/')]
                    self.group_count[path_without_param] = -1

            self.last_spaces = ""
            self.group_processed_params = {}
            self.tmp_dir_count = 0
            self.patched_lines = []

        def on_empty_line(self):
            self.patched_lines.append('')

        def on_comment(self, line):
            self.patched_lines.append(line)

        def on_group_start(self, line, group_name, group_path):
            self.group_processed_params[group_path] = []
            if group_path in self.group_count:
                self.group_count[group_path] += 1

            self.patched_lines.append(line)

        def on_group_end(self, line, group_name, group_path):
            """
            TODO: This approach will not work for new groups.
            We should append new groups if they are not present in config file
            """
            self._add_new_params(group_path)

            if group_path not in self.group_processed_params:
                raise Exception("cannot process group '%s'" % group_path)
            del self.group_processed_params[group_path]

            self.patched_lines.append(line)

        def _add_new_params(self, group_path):
            for name, target_count, value in self.patched_params:

                if isinstance(value, tuple):
                    value = value[0]

                name_components = name.split("/")
                path = "/".join(name_components[:-1])
                name = name_components[-1]

                if path != group_path:
                    continue
                if name in self.group_processed_params[group_path]:
                    continue
                if value is None:
                    continue
                if target_count is not None and self.group_count[group_path] != target_count:
                    continue

                logging.info("added new param '%s'", name)
                self._add_param(self.last_spaces, name, value, "added param")

            self.last_spaces = ""

        def on_param(self, line, spaces, param_name, old_value, param_name_with_path, group_path):
            self.last_spaces = spaces
            self.group_processed_params[group_path].append(param_name)

            for (name, target_count, value) in self.patched_params:
                if name != param_name_with_path:
                    continue
                if target_count is not None and self.group_count[group_path] != target_count:
                    continue

                if isinstance(value, tuple):
                    value = value[0]
                if value is not None:
                    if value == old_value:
                        self.patched_lines.append("# not touched")
                        self.patched_lines.append(line)
                    else:
                        self._add_param(spaces, param_name, value, "old value='%s'" % old_value)
                else:
                    self.patched_lines.append("# removed param %s" % param_name)
                logging.info("patched param '%s'", param_name)
                return  # param processed, leave handler

            # this param remains unchanged
            self.patched_lines.append(line)

        def _add_param(self, spaces, name, value, comment):
            value = str(value)
            value = self._process_value(value)
            self.patched_lines.append("# " + comment)
            self.patched_lines.append(spaces + "%s %s" % (name, value))

        def _process_value(self, value):
            if value.find(TMP_DIR_PATTERN) != -1:
                value = value.replace(TMP_DIR_PATTERN, "tmpdir.{0}".format(self.tmp_dir_count))
                self.tmp_dir_count += 1
            return value

    callbacks = Callbacks(patched_params)
    _scan(text, callbacks)
    config = "\n".join(callbacks.patched_lines)

    if len(patched_params_by_groups) == 0:
        return config
    else:
        return patch_config_by_groups(config, patched_params_by_groups, callbacks.tmp_dir_count)


class ParamPatcherByGroup:
    def patch_param(self, param_name, old_value, group_values):
        # type: (str, str, Mapping[str, str]) -> str
        # group_values - словарь всех параметров текущей группы
        raise NotImplementedError()


def patch_config_by_groups(text, patched_params, tmp_dir_count):
    # type: (str, List[Tuple[str, ParamPatcherByGroup]], int) -> str

    class Callbacks:
        def __init__(self, patched_params, tmp_dir_count):
            self.patched_params = patched_params
            self.tmp_dir_count = tmp_dir_count
            self.patched_lines = []

            self.is_leaf_group = True
            # each element is tuple (element_type, line, ...)
            # where element_type is one of "group_open", "param", "empty_line", "comment", "group_close"
            self.current_group_elements = []

        def on_empty_line(self):
            if self.is_leaf_group:
                self.current_group_elements.append(("empty_line", ""))
            else:
                self.patched_lines.append("")

        def on_comment(self, line):
            if self.is_leaf_group:
                self.current_group_elements.append(("comment", line))
            else:
                self.patched_lines.append(line)

        def on_group_start(self, line, group_name, group_path):
            for element in self.current_group_elements:
                self.patched_lines.append(element[1])

            self.current_group_elements = [("group_open", line, group_name, group_path)]
            self.is_leaf_group = True

        def on_group_end(self, line, group_name, group_path):
            if not self.is_leaf_group:
                self.patched_lines.append(line)
                return

            self.current_group_elements.append(("group_close", line, group_name, group_path))
            self._patch_group_params(group_path)
            self.current_group_elements = []
            self.is_leaf_group = False

        def _patch_group_params(self, group_path):
            group_values = {
                group_element[3]: group_element[4]
                for group_element in self.current_group_elements if group_element[0] == "param"
            }

            for element in self.current_group_elements:
                if element[0] == "param":
                    _, line, spaces, param_name, old_value = element
                    value = str(old_value)
                    for name, group_patcher in self.patched_params:
                        if name == group_path:
                            value = group_patcher.patch_param(param_name, value, group_values)

                    if value is not None:
                        value = self._process_value(value)
                        if value == old_value:
                            self.patched_lines.append(line)
                        else:
                            self.patched_lines.append("# old value='{}'".format(old_value))
                            self.patched_lines.append(spaces + "{} {}".format(param_name, value))
                    else:
                        self.patched_lines.append("# removed param {}".format(param_name))
                else:
                    self.patched_lines.append(element[1])

        def on_param(self, line, spaces, param_name, value, param_name_with_path, group_path):
            if self.is_leaf_group:
                self.current_group_elements.append(("param", line, spaces, param_name, value))
            else:
                self.patched_lines.append(line)

        def on_eof(self):
            for element in self.current_group_elements:
                self.patched_lines.append(element[1])

        def _process_value(self, value):
            if value.find(TMP_DIR_PATTERN) != -1:
                value = value.replace(TMP_DIR_PATTERN, "tmpdir.{0}".format(self.tmp_dir_count))
                self.tmp_dir_count += 1
            return value

    callbacks = Callbacks(patched_params, tmp_dir_count)
    _scan(text, callbacks)
    callbacks.on_eof()

    return "\n".join(callbacks.patched_lines)


class ConfigSectionAll:
    def __init__(self, section_name):
        self.section_name = section_name

    def check_section(self, name):
        return name == self.section_name

    def check_param(self, name, value):
        return True

    def descr(self):
        logging.info("section {}".format(self.section_name))

    @staticmethod
    def descr_for_removal(section_name, param_name, param_value):
        return "section {}".format(section_name)


class ConfigSectionByParamValue:
    def __init__(self, section_name, param_name_re, param_value_re, invert=False):
        self.section_name = section_name
        self.param_name_re = param_name_re
        self.param_value_re = param_value_re
        self.param_name_compiled_re = re.compile(param_name_re)
        self.param_value_compiled_re = re.compile(param_value_re)
        self.invert = invert

    def check_section(self, name):
        return self.section_name == name

    def check_param(self, name, value):
        name_match = bool(self.param_name_compiled_re.search(name))
        if name_match:
            value_match = bool(self.param_value_compiled_re.search(value))
            return value_match ^ self.invert
        else:
            return False

    def descr(self):
        return "section {} with {} in {}".format(
            self.section_name, self.param_value_re, self.param_name_re)

    def descr_for_removal(self, section_name, param_name, param_value):
        return "section {} with {} ~ /{}/ = {} {}~ /{}/".format(
            section_name,
            param_name, self.param_name_re,
            param_value, {False: "", True: "!"}[self.invert], self.param_value_re)


def remove_section_by_cond(text, cond):

    class Callbacks:
        def __init__(self, cond):
            self.in_target_group = False
            self.patched_lines = []
            self.skipped_lines = []
            self.need_removal = False
            self.cond = cond

            logging.info("removing {} from config".format(self.cond.descr()))

        def on_empty_line(self):
            if self.in_target_group:
                self.skipped_lines.append('')

            self.patched_lines.append('')

        def on_comment(self, line):
            if self.in_target_group:
                self.skipped_lines.append(line)

            self.patched_lines.append(line)

        def on_group_start(self, line, group_name, group_path):
            if self.cond.check_section(group_path):
                self.in_target_group = True
                self.skipped_lines.append(line)
                return
            if self.in_target_group:
                self.skipped_lines.append(line)
                return
            self.patched_lines.append(line)

        def on_group_end(self, line, group_name, group_path):
            if self.cond.check_section(group_path):
                if not self.need_removal:
                    self.patched_lines += self.skipped_lines
                    self.patched_lines.append(line)
                self.need_removal = False
                self.skipped_lines = []
                self.in_target_group = False
                return
            if self.in_target_group:
                self.skipped_lines.append(line)
                return
            self.patched_lines.append(line)

        def on_param(self, line, spaces, param_name, old_value, param_name_with_path, group_path):
            if self.in_target_group:
                self.skipped_lines.append(line)
                if not self.need_removal and self.cond.check_param(param_name, old_value):
                    self.need_removal = True
                    self.patched_lines.append(
                        "# removed {}".format(cond.descr_for_removal(group_path, param_name, old_value))
                    )
                return
            self.patched_lines.append(line)

    callbacks = Callbacks(cond)
    _scan(text, callbacks)

    if callbacks.in_target_group:
        raise Exception("cannot remove {} from config".format(cond.descr()))

    return "\n".join(callbacks.patched_lines)


def remove_section(text, section_name, param_name=None, param_value=None):
    if param_name is None or param_value is None:
        return remove_section_by_cond(text, ConfigSectionAll(section_name))
    else:
        return remove_section_by_cond(text, ConfigSectionByParamValue(
            section_name, "^" + param_name + "$", "^" + param_value + "$"))


def append_section(text, parent_section, child_section):

    class Callbacks:
        def __init__(self, parent_section, child_section):
            self.parent_section = parent_section
            self.child_section = child_section
            self.in_target_group = False
            self.patched_lines = []

        def on_empty_line(self):
            self.patched_lines.append('')

        def on_comment(self, line):
            self.patched_lines.append(line)

        def on_group_start(self, line, group_name, group_path):
            if group_path == self.parent_section:
                self.in_target_group = True
            self.patched_lines.append(line)

        def on_group_end(self, line, group_name, group_path):
            if group_path == self.parent_section and self.in_target_group:
                self.patched_lines.append(child_section)
                self.in_target_group = False
            self.patched_lines.append(line)

        def on_param(self, line, spaces, param_name, old_value, param_name_with_path, group_path):
            self.patched_lines.append(line)

    callbacks = Callbacks(parent_section, child_section)
    _scan(text, callbacks)

    if callbacks.in_target_group:
        raise Exception("Failed to extend section '{}'in config".format(parent_section))

    return "\n".join(callbacks.patched_lines)


class MetasearchSource:
    def __init__(self, source_type):
        self.source_type = source_type
        self.config = {}


def get_metasearch_sources(text):
    sources = []

    class Callbacks:
        def __init__(self):
            self.current_source = None

        def on_empty_line(self):
            pass

        def on_comment(self, line):
            pass

        def on_group_start(self, line, group_name, group_path):
            if group_name == 'SearchSource' or group_name == 'AuxSource':
                self.current_source = MetasearchSource(group_name)

        def on_group_end(self, line, group_name, group_path):
            if self.current_source and self.current_source.source_type == group_name:
                sources.append(self.current_source)
                self.current_source = None

        def on_param(self, line, spaces, param_name, value, param_name_with_path, group_path):
            if self.current_source:
                self.current_source.config[param_name] = value

    _scan(text, Callbacks())

    return sources


def _scan(text, callbacks, include_group_attributes=False):
    current_path = []

    def _process_line(line):
        stripped_line = line.strip(" \t\r")
        if not stripped_line:
            callbacks.on_empty_line()
            return
        if stripped_line.startswith("#"):
            callbacks.on_comment(line)
            return

        group_match = re.match(r"^\s*<(/)?([^\s=>]+)([^>]*)>\s*$", line)

        if group_match:
            group_name = group_match.group(2)

            if group_match.group(1) != "/":
                if include_group_attributes:
                    current_path.append(group_name + group_match.group(3))
                else:
                    current_path.append(group_name)

                callbacks.on_group_start(line, group_name, "/".join(current_path))
            else:
                last_group_name = current_path[-1].split(" ")[0]
                eh.verify(group_name == last_group_name, "Cannot parse config, line: {0}".format(line))

                callbacks.on_group_end(line, group_name, "/".join(current_path))
                current_path.pop()
        else:
            name_value_match = re.match("^\\s*([^\\s]+)\\s*(.*)\\s*$", line)
            eh.verify(name_value_match, "Cannot parse config, line: {0}".format(line))
            param_name = name_value_match.groups()[0]
            value = name_value_match.groups()[1]
            param_name_with_path = "/".join(current_path + [param_name])

            callbacks.on_param(
                line, " " * name_value_match.start(1), param_name,
                value, param_name_with_path, "/".join(current_path))

    for line in text.split("\n"):
        _process_line(line)
