# -*- coding: utf-8 -*-

import json

import sandbox.sdk2 as sdk2
import sandbox.projects.common.string as sandbox_string


class CreateTextResource2(sdk2.Task):
    """
        создает ресурс путем указания типа и заполнения многострочного текстового поля или json
    """

    class Parameters(sdk2.Parameters):
        created_resource_name = sdk2.parameters.String(
            'Created resource name', default='uploaded_resource', required=True)
        with sdk2.parameters.RadioGroup('File content input field type') as input_type:
            input_type.values['text'] = input_type.Value('Plain text', default=True)
            input_type.values['json'] = input_type.Value('JSON')

        with input_type.value['text']:
            resource_file_content = sdk2.parameters.String('Resource file content', multiline=True)
        with input_type.value['json']:
            resource_file_content_json = sdk2.parameters.JSON('Resource file content')

        resource_type = sdk2.parameters.String('Resource type', required=True)
        store_forever = sdk2.parameters.Bool('Store resource forever (assign TTL to infinity)', do_not_copy=True)
        resource_attrs = sdk2.parameters.String('Set attrs to resources (e.g.: attr1=v1, attr2=v2)', do_not_copy=True)

        with sdk2.parameters.Output():
            out_resource_id = sdk2.parameters.Resource('Created resource')

    class Requirements(sdk2.Requirements):
        class Caches(sdk2.Requirements.Caches):
            pass

        cores = 1

    def get_file_content(self):
        if self.Parameters.input_type == 'text':
            return self.Parameters.resource_file_content.replace('\r\n', '\n')
        if self.Parameters.input_type == 'json':
            return json.dumps(self.Parameters.resource_file_content_json, sort_keys=True, indent=4, ensure_ascii=False)

    def on_execute(self):
        resource = sdk2.Resource[self.Parameters.resource_type](
            self, 'Resource with data from task input', self.Parameters.created_resource_name)

        resource.path.write_bytes(self.get_file_content().encode('utf-8'))

        if self.Parameters.resource_attrs:
            for k, v in sandbox_string.parse_attrs(self.Parameters.resource_attrs).items():
                setattr(resource, k, v)

        if self.Parameters.store_forever:
            resource.ttl = 'inf'

        self.Parameters.out_resource_id = resource.id
