import os
import itertools
import shutil
import six

from sandbox.sandboxsdk import process


class DebRelease(object):
    """
    Context manager for using debrelease to upload deb packages.

    Usage:
        DUPLOAD_CONF = {
            'common': {
                'fqdn': "dist.yandex.ru",
                'method': "scpb",
                'login': "own_zomb_name",
                'incoming': "/repo/common/mini-dinstall/incoming/",
                'dinstall_runs': 1,
                'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
            },
            'search': {
                'fqdn': "dist.yandex.ru",
                'method': "scpb",
                'login': "own_zomb_name",
                'incoming': "/repo/common/mini-dinstall/incoming/",
                'dinstall_runs': 1,
                'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
            }
        }
        with debpkg.DebRelease(DUPLOAD_CONF) as deb:
            deb.debrelease(['--to', '<repository_id>', ...])
    """

    DUPLOAD_CONF_TEMPLATE = '\n\t'.join((
        "$cfg{{'{title}'}} = {{",
        'fqdn => "{fqdn}",',
        'method => "{method}",',
        'login => "{login}",',
        'incoming => "{incoming}",',
        'options => "{options}",',
        'dinstall_runs => {dinstall_runs},\n}};',))
    DUPLOAD_CONF_PATH = os.path.expanduser('~/.dupload.conf')
    DUPLOAD_CONF_TMP_PATH = os.path.expanduser('~/.local/tmp_dupload.cnf')

    def __init__(self, dupload_config, login='anonymous'):
        """
        :param dupload_config: dictionary with dupload configuration.
        """
        self.__config = {}
        for repo, conf in dupload_config.items():
            d = {
                "fqdn": "{repo}.dupload.dist.yandex.ru".format(repo=repo),
                "method": "scpb",
                "login": login,
                "incoming": "/repo/{0}/mini-dinstall/incoming/".format(repo),
                "dinstall_runs": 1,
                "options": "-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o ConnectionAttempts=3",
            }
            d.update(conf)
            self.__config[repo] = d

    def __enter__(self):
        if os.path.exists(self.DUPLOAD_CONF_PATH):
            shutil.move(self.DUPLOAD_CONF_PATH, self.DUPLOAD_CONF_TMP_PATH)
        self.__dupload_conf = open(self.DUPLOAD_CONF_PATH, 'w')
        self.__dupload_conf.write('package config;\n\n$default_host = undef;\n\n' + '\n\n'.join([
            self.DUPLOAD_CONF_TEMPLATE.format(title=title, **values)
            for title, values in self.__config.items()]))
        self.__dupload_conf.write('\n')
        self.__dupload_conf.close()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if os.path.exists(self.DUPLOAD_CONF_TMP_PATH):
            shutil.move(self.DUPLOAD_CONF_TMP_PATH, self.DUPLOAD_CONF_PATH)
        else:
            os.unlink(self.DUPLOAD_CONF_PATH)

    def debrelease(self, debrelease_args, **run_process_params):
        if isinstance(debrelease_args, (six.text_type, six.binary_type)):
            debrelease_args = (debrelease_args,)
        prefix = 'debrelease'
        cmd = list(itertools.chain((prefix,), debrelease_args))
        run_process_params.update(shell=True, log_prefix=prefix)
        process.run_process(cmd, **run_process_params)
