# -*- coding: utf-8 -*-

# Please DO NOT INSERT Sandbox specific code to this module
from six.moves import range

import inspect
import functools
import logging
from time import sleep


logger = logging.getLogger(__name__)


def retries(
    max_tries,
    delay=1, backoff=2,
    max_delay=float('inf'),
    exceptions=(Exception,),
    hook=None,
    final_hook=None,
    log=True,
    raise_class=None,
    default_instead_of_raise=False,
    default_value=None,
):
    """
        Wraps function into subsequent attempts with increasing delay between attempts.
        Adopted from https://wiki.python.org/moin/PythonDecoratorLibrary#Another_Retrying_Decorator
        :param max_tries: Number of retries before fail.
        :param delay: Initial delay after fail (seconds).
        :param backoff: Delay multiplier.
        :param max_delay: Max possible delay. Delay will not grow after this value.
        :param exceptions: Tuple of Exception classes to catch.
        :param hook: Function to run after every fail.
        :param final_hook: Function to run after last fail if n_try >= max_tries.
        :param log: Log exceptions or not.
        :param raise_class: Allows to redefine raised class. If None, raise default.
        :param default_instead_of_raise: Allows not to fail on bad run, just return default_value.
        :param default_value: If default_instead_of_raise is True, return this value instead of fail.
    """
    def dec(func):
        @functools.wraps(func)
        def f2(*args, **kwargs):
            current_delay = delay
            for n_try in range(0, max_tries + 1):
                try:
                    return func(*args, **kwargs)
                except exceptions as e:
                    if n_try < max_tries:
                        current_delay = min(current_delay, max_delay)
                        if log:
                            logger.warning(
                                "Error in function %s on %s try:\n%s\nWill sleep for %s seconds...",
                                func.__name__, n_try, e, current_delay, exc_info=True,
                            )
                        if hook is not None:
                            hook(n_try, e, current_delay)
                        sleep(current_delay)
                        current_delay *= backoff
                    else:
                        logger.error("Max retry limit %s reached, giving up with error:\n%s", n_try, e)
                        if final_hook is not None:
                            final_hook(e)
                        if default_instead_of_raise:
                            return default_value
                        if raise_class is None:
                            raise
                        else:
                            raise raise_class(
                                "Max retry limit {} reached, giving up with error: {}".format(n_try, str(e))
                            )
        return f2
    return dec


# TODO: use functools.lru_cache decorator in python3
def memoize(func):
    cache = {}

    def wrap(*args):
        if args not in cache:
            cache[args] = func(*args)
        return cache[args]

    return wrap


def memoized_property(fn):
    attr_name = "_memoized_property_{}".format(fn.__name__)

    @property
    @functools.wraps(fn)
    def _memoized_property(self):
        if not hasattr(self, attr_name):
            setattr(self, attr_name, fn(self))
        return getattr(self, attr_name)

    return _memoized_property


def change_exception_to_none(func):
    @functools.wraps(func)
    def wrap(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as e:
            logging.debug("Got exception: %s", e)
            return

    return wrap


def log_start_and_finish(custom_logger=logging, log_level=logging.DEBUG, log_result=False):

    def dec(func):

        @functools.wraps(func)
        def wrap(*args, **kwargs):

            custom_logger.log(log_level, "__START__ %s", func.__name__)
            result = func(*args, **kwargs)

            if log_result:
                logging.log(log_level, "__RESULT__: %s", result)

            custom_logger.log(log_level, "__FINISH__ %s", func.__name__)

            return result

        return wrap

    return dec


def decorate_all_public_methods(decorator):
    def dec(cls):
        for name, method in cls.__dict__.items():
            if name.startswith('_'):  # don't decorate private functions
                continue
            if inspect.isfunction(method):
                setattr(cls, name, decorator(method))
        return cls
    return dec
