import logging
import unittest
from cStringIO import StringIO

from sandbox.projects.common import decorators

TRIES = 0


@decorators.retries(4, delay=0.5, backoff=1)
def try_func(fail_times=0):
    global TRIES
    while TRIES < fail_times:
        TRIES += 1
        raise Exception
    return True


@decorators.retries(2, delay=0.5, backoff=1, default_instead_of_raise=True)
def try_func_safe(fail_times=0):
    global TRIES
    while TRIES < fail_times:
        TRIES += 1
        raise Exception
    return True


class RetriesTest(unittest.TestCase):
    def test_no_retries(self):
        global TRIES
        TRIES = 0
        self.assertTrue(try_func(fail_times=0))

    def test_success_retries(self):
        global TRIES
        TRIES = 0
        self.assertTrue(try_func(fail_times=2))

    @unittest.expectedFailure
    def test_fail_retries(self):
        global TRIES
        TRIES = 0
        try_func(fail_times=6)

    def test_return_default_retries(self):
        global TRIES
        TRIES = 0
        self.assertIsNone(try_func_safe(fail_times=4))


class DecorateAllPublicMethodsTest(unittest.TestCase):
    @classmethod
    def setUpClass(cls):
        cls._root_logger = logging.getLogger()
        cls._root_logger.setLevel(logging.DEBUG)

    def setUp(self):
        self._f = StringIO()
        self._handler = logging.StreamHandler(self._f)
        self._handler.setLevel(logging.DEBUG)
        self._handler.setFormatter(logging.Formatter('%(message)s'))
        self._root_logger.addHandler(self._handler)

        @decorators.decorate_all_public_methods(decorators.log_start_and_finish(self._root_logger))
        class A(object):
            @classmethod
            def class_method(cls):
                logging.debug("This is class_method")
                return "class_method"

            @staticmethod
            def static_method():
                logging.debug("This is static_method")
                return "static_method"

            def instance_method(self):
                logging.debug("This is instance_method")
                return "instance_method"

            @property
            def property_method(self):
                logging.debug("This is property_method")
                return "property_method"

            @decorators.retries(1)
            def decorated_method(self):
                logging.debug("This is decorated_method")
                return "decorated_method"

        self.my_a_class = A

    def tearDown(self):
        self._root_logger.removeHandler(self._handler)
        self._f.close()

    def test_logging_decoration_class_method(self):
        assert self.my_a_class.class_method() == "class_method"
        assert self._f.getvalue() == "This is class_method\n"

    def test_logging_decoration_static_method(self):
        assert self.my_a_class.static_method() == "static_method"
        assert self._f.getvalue() == "This is static_method\n"

    def test_logging_decoration_property(self):
        assert self.my_a_class().property_method == "property_method"
        assert self._f.getvalue() == "This is property_method\n"

    def test_logging_decoration_instance_method(self):
        assert self.my_a_class().instance_method() == "instance_method"
        assert self._f.getvalue() == "__START__ {0}\nThis is {0}\n__FINISH__ {0}\n".format("instance_method")

    def test_logging_decoration_decorated_method(self):
        assert self.my_a_class().decorated_method() == "decorated_method"
        assert self._f.getvalue() == "__START__ {0}\nThis is {0}\n__FINISH__ {0}\n".format("decorated_method")
