"""
    Should NOT have dependencies from sandbox code!
    Used as standalone script!
"""

import cgi
import difflib
import logging
import os
import shutil
import textwrap
import six
from six.moves import range

from sandbox.projects.common import string
from sandbox.projects.common import templates


class DiffType(object):
    REMOVED = "br"
    REMOVED_LIGHT = "br-lt"
    ADDED = "ba"
    ADDED_LIGHT = "ba-lt"
    CHANGED = "bc"
    NODIFF = "bn"
    SIDE_BY_SIDE = "side-by-side"


class PrinterBase(object):
    """
    Common interface for diff printers.

    To create your own printer, you need to process the following steps.
    1. Derive from PrinterBase
    2. Implement _do_indent() and _print_info() methods
    3. Optionally make additional work in on_new_pair() method
    4. Optionally make some work at the end of processing in finalize() method

    Parallel processing
    If the work could be done in parallel, you should do the following.
    5. Return True in supports_parallel_processing() property
    6. Implement get_parallel_subprinter() method
    7. Implement join_parallel_subprinters() method
    """

    def __init__(self, obj_index=0, write_compact_diff=True):
        """
        Create base printer object.

        * obj_index - object pairs sequential number base.
                      Pairs will start with its number and it will be incremented during
                      on_new_pair() calls
        * write_compact_diff - flag that defines whether printer will write compact diff
        """
        self.obj_index = obj_index - 1  # -1 is because than we expect on_new_pair() to be called
        self.compact_diff = {}
        self.write_compact_diff = write_compact_diff
        self._diff_lines_count = 0

    def get_diff_lines_count(self):
        """Return count of diffs."""
        return self._diff_lines_count

    def sched(self, title):
        """
        Schedule new item in diff hierarchy.

        If actual diff will be written, it will be written
        under this stack of titles
        * title - title of current item in diff hierarchy
        """
        self._sched.append([title, False])

    def desched(self):
        """Go to one level higher in diff hierarchy."""
        self._sched.pop()

    def __call__(self, line, diff_class=DiffType.NODIFF):
        """
        Write full and compact diffs.

        * line - could be string (for added and removed items)
                 or tuple (for changed items)
        """
        i = -1
        while not self._sched[i][1]:
            i -= 1
        i += 1  # index of first unwritten sched (always <= 0)

        if diff_class != DiffType.NODIFF:
            self._diff_lines_count += 1

        line = self._diff_line_to_string(line, diff_class)
        self._do_indent(i, diff_class)
        self._print_info(line, diff_class)
        if self.write_compact_diff:
            self._form_compact_diff(line, diff_class)

    def print_item(self, title, item, diff_type):
        """
        Simple interface for printing one single item in diff hierarchy.

        * title - item title
        * item - item value
        """
        self.sched(title)
        self(item, diff_type)
        self.desched()

    def on_new_pair(self, title=""):
        """
        Function needed to call when we are about to compare a new pair of objects.

        This method should be called even if objects are equal
        * title - optional object pair title
        """
        self._sched = [["", True]]
        self.obj_index += 1

    def finalize(self):
        """
        Postprocess printer data after all calls.

        Expected to be runned once after all processing
        """
        pass

    @property
    def supports_parallel_processing(self):
        """
        Return True if printer supports parallel results processing.

        In that case printer must also support get_parallel_subprinter()
        and join_parallel_subprinters() methods
        """
        return False

    def get_parallel_subprinter(self):
        """
        If printer supports parallel processing,
        this function must return printer
        that is called in parallel subtasks ('map' operation)
        to process one pair of compared objects
        """
        raise NotImplementedError

    def join_parallel_subprinters(self, subprinters):
        """
        Function to join results from parallel subprinters ('reduce' operation).
        It is assumed that printer will take all information
        about what to print from each subprinter and join it
        to one consistent result.

        Could be called several times

        The rule of thumb is when result from sequentially calling
        one (parent) printer and result of calling each of its subprinter
        and then joining them together are equal.

        * subprinters - list of subprinters returned
                        from get_parallel_subprinter() method.
                        Standard differ class will pass items in this list
                        in the same order as they were created
        """
        raise NotImplementedError

    def get_compact_diff(self):
        """Return compact diff."""
        return sorted(list(self.compact_diff.items()), key=lambda x: (-len(x[1]), x[0]))

    def _do_indent(self, i, diff_class):
        """
        Perform indent.

        Implement this method in descendant classes
        * i - indents count
        * diff_class - one of DiffType
        """
        raise NotImplementedError

    def _print_info(self, line, diff_class):
        """
        Print one diff line.

        Implement this method in descendant classes
        * line - compared value
        * diff_class - one of DiffType
        """
        raise NotImplementedError

    def _merge_compact_diff(self, compact_diff_from):
        """
        Merge two compact diffs into one.

        You can call it in descendant classes
        to merge parallel results, for example
        * compact_diff_from - compact diff that you want
                              to add to you own compact diff
        """
        for k, v in six.iteritems(compact_diff_from):
            self.compact_diff.setdefault(k, []).extend(v)

    #
    # Base class implementation
    #

    def _form_compact_diff(self, line, diff_class):
        str_head = "/".join([str(title).split(":", 1)[0] for title, _ in self._sched[1:]])
        # logging.info("head = %s", str_head)
        self.compact_diff.setdefault(str_head, []).append((self.obj_index, line, diff_class))

    def _diff_line_to_string(self, line, diff_class):
        # Removing converting to string may lead to strange error when trying to pass result back to main process in parallel processing:
        # MaybeEncodingError: Error sending result: '(<projects.common.differ.printers.PrinterToHtml object at 0x7f583465ad50>, True)'.
        # Reason: 'ReferenceError('weakly-referenced object no longer exists',)'
        # The reason why it is so remains unexplored
        if diff_class == DiffType.CHANGED and isinstance(line, (tuple, list)):
            assert len(line) == 2
            old_line, new_line = tuple(line)
            return string.all_to_str(old_line), string.all_to_str(new_line)
        else:
            return string.all_to_str(line)


class PrinterToList(PrinterBase):
    def __init__(self):
        super(PrinterToList, self).__init__()
        self._diff = []
        self._indent_count = 0

    def desched(self):
        if self._sched[-1][0] and self._sched[-1][1]:
            self._indent_count -= 1
        super(PrinterToList, self).desched()

    def on_new_pair(self, title=""):
        super(PrinterToList, self).on_new_pair(title)
        self._indent_count = 0
        self._diff.append([])

    def _do_indent(self, index_of_unwritten, diff_class):
        for j in range(index_of_unwritten, 0):
            # write all unwritten sheds
            self._sched[j][1] = True
            if self._sched[j][0]:
                self._print_info_impl(self._sched[j][0], diff_class, is_head=True)
            self._indent_count += 1

    def _print_info_impl(self, line, diff_class, is_head=False):
        self._diff[self.obj_index].append((is_head, self._indent_count, line, diff_class))

    def _print_info(self, line, diff_class):
        self._print_info_impl(line, diff_class)

    def get_diff(self):
        return self._diff


class PrinterToHtml(PrinterBase):
    def __init__(self,
                 output_dir_name="",
                 write_compact_diff=True,
                 max_file_size=5 * 1024 * 1024,  # 5 Mb
                 pair_head_template="pair {obj_index}",
                 pair_range_filename_template="r_{obj_index_from:0>6}-{obj_index_to:0>6}.html",
                 obj_head_template="object",
                 obj_title_template="{title}"
                 ):
        super(PrinterToHtml, self).__init__(write_compact_diff=write_compact_diff)
        self.max_file_size = max_file_size
        self.output_dir_name = output_dir_name
        self._current_diff = []
        self._current_title = ""
        self._printed_html_part = ""
        self._has_written = False
        self._last_flushed_index = -1
        self._is_subprinter = False

        # Output string templates
        self.pair_head_template = pair_head_template
        self.pair_range_filename_template = pair_range_filename_template
        self.obj_head_template = obj_head_template
        self.obj_title_template = obj_title_template

    def desched(self):
        if self._sched[-1][0] and self._sched[-1][1]:
            self._current_diff.append(HtmlBlock.end())
        super(PrinterToHtml, self).desched()

    def on_new_pair(self, title=""):
        self._flush_current_pair()
        self._current_title = title
        super(PrinterToHtml, self).on_new_pair(title)

    def _do_indent(self, index_of_unwritten, diff_class):
        for j in range(index_of_unwritten, 0):
            # start blocks with unwritten scheds
            self._sched[j][1] = True
            block_color = diff_class if j == -1 else DiffType.CHANGED
            if self._sched[j][0]:
                self._current_diff.append(HtmlBlock.start(self._sched[j][0], bg_class=block_color))

    def _print_info(self, line, diff_class):
        if diff_class == DiffType.CHANGED:
            changed_lines = line if isinstance(line, (tuple, list)) else line.split("->")
            line1, line2 = match_string_color_changed(changed_lines[0].strip(), changed_lines[1].strip())
            if line1 or line2:
                self._current_diff.append(HtmlBlock.colored_data(line1))
                self._current_diff.append(HtmlBlock.colored_data(line2))
        else:
            self._current_diff.append(HtmlBlock.simple_data(line, text_bg_class=diff_class))

    def _before_pair(self):
        before = [HtmlBlock.start(self.pair_head_template.format(obj_index=self.obj_index), is_open=False)]
        if self._current_title:
            before.append(HtmlBlock.simple_block(self.obj_head_template, self.obj_title_template.format(title=self._current_title), is_open=False, escape=False))
        return before

    def _after_pair(self):
        return [HtmlBlock.end()]

    def _add_new_html_part(self, part_to_add, obj_index, final_flush=False):
        if part_to_add:
            self._has_written = True

        old_len = len(self._printed_html_part)
        new_len = len(part_to_add) + old_len

        if final_flush or old_len >= self.max_file_size or new_len >= self.max_file_size:
            flush_first = old_len > 0 and abs(old_len - self.max_file_size) < abs(new_len - self.max_file_size)

            if flush_first:
                self._flush_current_html(obj_index - 1)
            self._printed_html_part += part_to_add
            if not flush_first:
                self._flush_current_html(obj_index)
        else:
            self._printed_html_part += part_to_add

    def _flush_current_pair(self, final_flush=False):
        part_to_add = ""
        if self._current_diff:
            part_to_add = "\n".join(self._before_pair() + self._current_diff + self._after_pair())
            self._current_diff = []

        self._add_new_html_part(part_to_add, self.obj_index, final_flush)

    def _flush_current_html(self, last_index):
        if self._printed_html_part and not self._is_subprinter:
            path_to_file = os.path.join(
                self.output_dir_name,
                self.pair_range_filename_template.format(
                    obj_index_from=self._last_flushed_index + 1,
                    obj_index_to=last_index
                )
            )
            with open(path_to_file, "w") as f:
                HtmlFile.write(f, self._printed_html_part, buttons=False)
            self._printed_html_part = ""
            self._last_flushed_index = last_index

    def _write_companion(self, template):
        HtmlFile.write_companion(self.output_dir_name, template)

    def _write_compact_diff(self):
        logging.info("start writing compact_diff")
        max_diff_num = 500  # in case of too many diffs (SEARCH-1712)
        full_path_compact = os.path.join(self.output_dir_name, "compact_diff")
        if os.path.exists(full_path_compact):
            shutil.rmtree(full_path_compact)
        os.mkdir(full_path_compact)

        for head, diff in self.get_compact_diff():
            with open(os.path.join(full_path_compact, head.replace("/", "_") + ".html"), "w") as f:
                def write_body():
                    logging.info("head = %s (amount of changes = %s)", head, len(diff))
                    f.write(HtmlBlock.start(
                        "{} (amount of changes = {})".format(head, len(diff)),
                        is_open=False
                    ))
                    for pair_number, line, color in diff[:max_diff_num]:
                        f.write(HtmlBlock.start(self.pair_head_template.format(obj_index=pair_number), bg_class=color))
                        logging.debug("writing compact_diff in pair %s", pair_number)

                        if color == DiffType.CHANGED:
                            changed_lines = line if isinstance(line, (tuple, list)) else line.split("->")
                            first_line, second_line = match_string_color_changed(
                                changed_lines[0].strip(), changed_lines[1].strip()
                            )
                            f.write(HtmlBlock.colored_data(first_line))
                            f.write(HtmlBlock.colored_data(second_line))
                        else:
                            f.write(HtmlBlock.simple_data(line, text_bg_class=color))
                        f.write(HtmlBlock.end())
                    f.write(HtmlBlock.end())

                WriteDiff(f, write_body, addButtons=False)

    def finalize(self):
        self._flush_current_pair(True)
        if not self._is_subprinter:
            if self._has_written:
                self._write_companion('response_diff_style.css')  # TODO: rename
                self._write_companion('response_diff_scripts.js')  # TODO: rename
                if self.write_compact_diff:
                    self._write_compact_diff()
            else:
                self._write_companion('no_diff_detected.html')

    @property
    def supports_parallel_processing(self):
        return True

    def get_parallel_subprinter(self):
        subprinter = PrinterToHtml(output_dir_name="",
                                   write_compact_diff=self.write_compact_diff,
                                   pair_head_template=self.pair_head_template,
                                   pair_range_filename_template=self.pair_range_filename_template,
                                   obj_head_template=self.obj_head_template,
                                   obj_title_template=self.obj_title_template)
        subprinter.obj_index = self.obj_index
        subprinter._is_subprinter = True
        self.obj_index += 1
        return subprinter

    def join_parallel_subprinters(self, subprinters):
        for subprinter in subprinters:
            self._add_new_html_part(subprinter._printed_html_part, subprinter.obj_index)
            self._merge_compact_diff(subprinter.compact_diff)


_HTML_HEADER = templates.get_html_template('header.html')


_LEGEND = """
<table border=1>
    <tr> <th colspan=4> Legend </th> </tr>
    <tr> <td class="{}">Added</td>
         <td class="{}">Changed</td>
         <td class="{}">Removed</td>
         <td class="{}">No diffs</td>
    </tr>
</table><p>
""".format(
    DiffType.ADDED,
    DiffType.CHANGED,
    DiffType.REMOVED,
    DiffType.NODIFF
)

_BUTTONS = """
<input id="btnShowNodes" type="button" onclick="showNodes()" value="Show all nodes" />
<input id="btnHideNodes" type="button" onclick="hideNodes()" value="Show only diffs" style="display: none;"/><p>
"""

_HTML_FOOTER = "</body></html>"


def WriteDiff(file, WriteBody, addLegend=True, addButtons=True, title=None):
    if title is None:
        title = "diff"
    file.write(_HTML_HEADER % title)

    if addLegend:
        file.write(_LEGEND)

    if addButtons:
        file.write(_BUTTONS)

    have_diff = WriteBody()

    file.write(_HTML_FOOTER)

    return have_diff


class HtmlBlock(object):
    START = (
        '<div{style}>'
        '<a href="#" OnClick="return openclose(this);"><b>[{plusminus}]</b></a> {title}'
        '<br>'
        '<div class="ml" style="display: {display}">'
    )
    MIDDLE = '<pre{style}>{data}</pre>'
    COLORED_CHUNKS = '<span{chunk_style}>{chunk_data}</span>'
    END = '</div></div>'

    UNKNOWN_TITLE = "unknown_title"

    @classmethod
    def start(cls, title, bg_class=None, is_open=True):
        return cls.START.format(
            style=cls._get_bg_class_str(bg_class),
            plusminus="-" if is_open else "+",
            title=title or cls.UNKNOWN_TITLE,
            display="block" if is_open else "none"
        )

    @classmethod
    def end(cls):
        return cls.END

    @classmethod
    def simple_data(cls, text, text_bg_class=None, escape=True):
        add_text = string.all_to_str(text)
        if escape:
            add_text = cgi.escape(add_text)
        return cls.MIDDLE.format(
            style=cls._get_bg_class_str(text_bg_class),
            data=add_text,
        )

    @classmethod
    def colored_data(cls, text_chunks, escape=True):
        data = []
        for span_class, text in text_chunks:
            add_text = string.all_to_str(text)
            if escape:
                add_text = cgi.escape(add_text)
            data.append(cls.COLORED_CHUNKS.format(
                chunk_style=cls._get_bg_class_str(span_class),
                chunk_data=add_text
            ))
        return cls.MIDDLE.format(
            style='',
            data="".join(data),
        )

    @classmethod
    def side_by_side_data(cls, text1, text2, escape=True):
        add_text1 = string.all_to_str(text1)
        add_text2 = string.all_to_str(text2)
        if escape:
            add_text1 = cgi.escape(add_text1)
            add_text2 = cgi.escape(add_text2)
        return "\n".join([
            '<table style="table-layout: fixed; width: 96%"><tr><td style="vertical-align: top;">',
            cls.MIDDLE.format(
                style=cls._get_bg_class_str([DiffType.SIDE_BY_SIDE, DiffType.REMOVED_LIGHT]),
                data=add_text1
            ),
            '</td><td style="vertical-align: top;">',
            cls.MIDDLE.format(
                style=cls._get_bg_class_str([DiffType.SIDE_BY_SIDE, DiffType.REMOVED_LIGHT]),
                data=add_text2
            ),
            '/td></tr></table>'
        ])

    @classmethod
    def simple_block(cls, title, text, block_bg_class=None, text_bg_class=None, is_open=True, escape=True):
        return "\n".join([
            cls.start(title, bg_class=block_bg_class, is_open=is_open),
            cls.simple_data(text, text_bg_class=text_bg_class, escape=escape),
            cls.end()
        ])

    @staticmethod
    def _get_bg_class_str(bg_class):
        if bg_class:
            if isinstance(bg_class, (list, tuple)):
                bg_class = " ".join(bg_class)
            return ' class="{}"'.format(bg_class)
        return ''


class HtmlFile(object):
    @staticmethod
    def write_companion(path_to_diffs, template):
        """For css and js files"""
        path_to_companion = os.path.join(path_to_diffs, template)
        logging.info("Try to write companion file: %s", path_to_companion)
        if not os.path.exists(path_to_companion):
            with open(path_to_companion, "w") as f:
                f.write(templates.get_html_template(template))
            logging.info("Companion is written successfully")
        else:
            logging.info("Companion already exists")
        return path_to_companion

    @staticmethod
    def header(template='response_diff_header.html'):
        return templates.get_html_template(template)

    @staticmethod
    def legend():
        return textwrap.dedent(
            """
            <table border=1>
                <tr><th colspan=4> Legend </th></tr>
                <tr><td class="{}">Added</td>
                    <td class="{}">Changed</td>
                    <td class="{}">Removed</td>
                    <td class="{}">No diffs</td>
                </tr>
            </table><p>
            """.format(
                DiffType.ADDED,
                DiffType.CHANGED,
                DiffType.REMOVED,
                DiffType.NODIFF
            )
        )

    @staticmethod
    def buttons():
        return (
            '\n<input id="btnShowNodes" type="button" '
            'onclick="showNodes()" value="Show all nodes"/>\n'
            '<input id="btnHideNodes" type="button" '
            'onclick="hideNodes()" value="Show only diffs" style="display: none;"/><p>\n'
        )

    @staticmethod
    def footer():
        return "</body></html>"

    @classmethod
    def write(cls, f, body, title="diff", legend=True, buttons=True):
        f.write(cls.header() % title)
        if legend:
            f.write(cls.legend())
        if buttons:
            f.write(cls.buttons())
        f.write(body)
        f.write(cls.footer())


def match_string_color_changed(values1, values2):
    values_list1 = []
    values_list2 = []
    try:
        values1 = values1.decode("utf-8")
        values2 = values2.decode("utf-8")
    except Exception:
        logging.debug("Cannot decode strings into utf-8:\n%s\n---\n%s", values1, values2)
    else:
        if len(values1) > 100000 or len(values2) > 100000:  # otherwise it'll take too long
            logging.debug("Skipped very long lines: %s %s", len(values1), len(values2))
            values_list1.append((DiffType.CHANGED, values1))
            values_list2.append((DiffType.CHANGED, values2))
        else:
            s = difflib.SequenceMatcher(a=values1, b=values2)
            a, b = 0, 0
            for block in s.get_matching_blocks():
                if a < block.a or b < block.b:
                    values_list1.append((DiffType.REMOVED, values1[a: block.a]))
                    values_list2.append((DiffType.ADDED, values2[b: block.b]))
                if block.size:
                    a = block.a + block.size
                    b = block.b + block.size
                    values_list1.append((DiffType.CHANGED, values1[block.a: a]))
                    values_list2.append((DiffType.CHANGED, values2[block.b: b]))
    finally:
        return values_list1, values_list2
