# -*- coding: utf-8 -*-
import os

from sandbox.projects.common.differ.ut import common
from sandbox.projects.common.differ import printers


class TestPrinterBase(object):
    class PrinterMock(printers.PrinterBase):
        do_indent_calls = []
        print_info_calls = []

        def _do_indent(self, i, diff_class):
            self.do_indent_calls.append((i, diff_class))

        def _print_info(self, line, diff_class):
            self.print_info_calls.append((line, diff_class))

    def test_printer_base(self):
        printer = self.PrinterMock()
        printer.on_new_pair()
        printer.sched('title')
        printer('line', printers.DiffType.CHANGED)
        printer.desched()

        printer.sched('title')
        printer.sched('subtitle')
        printer('subline', printers.DiffType.ADDED)
        printer.desched()
        printer.desched()

        printer.on_new_pair()
        printer.sched('_title2')
        printer.sched('subtitle2')
        printer('subline2', printers.DiffType.REMOVED)
        printer.desched()
        printer.desched()

        common.assert_equal(printer.do_indent_calls, [(-1, printers.DiffType.CHANGED),
                                                      (-2, printers.DiffType.ADDED),
                                                      (-2, printers.DiffType.REMOVED)])
        common.assert_equal(printer.print_info_calls, [('line', printers.DiffType.CHANGED),
                                                       ('subline', printers.DiffType.ADDED),
                                                       ('subline2', printers.DiffType.REMOVED)])
        common.assert_equal(printer.get_compact_diff(), [('_title2/subtitle2', [(1, 'subline2', printers.DiffType.REMOVED)]),
                                                         ('title', [(0, 'line', printers.DiffType.CHANGED)]),
                                                         ('title/subtitle', [(0, 'subline', printers.DiffType.ADDED)])])

    def test_all(self):
        self.test_printer_base()


class TestHtmlPrinter(object):
    def test_no_diff(self, run_on_new_pair, run_sched):
        dir_name = common.make_tmp_dir('test_no_diff')
        printer = printers.PrinterToHtml(dir_name)
        if run_on_new_pair:
            printer.on_new_pair()
        if run_sched:
            printer.sched('title')
            printer.desched()
        printer.finalize()
        common.assert_equal(os.listdir(dir_name), ['no_diff_detected.html'])

    def test_helper_files(self):
        dir_name = common.make_tmp_dir('test_helper_files')
        printer = printers.PrinterToHtml(dir_name)
        printer.on_new_pair()
        printer.sched('title')
        printer('line')
        printer.desched()
        printer.finalize()
        common.assert_issubset(['response_diff_scripts.js', 'response_diff_style.css'], os.listdir(dir_name))

    def test_compact_diff(self, write_compact_diff):
        dir_name = common.make_tmp_dir('test_compact_diff')
        printer = printers.PrinterToHtml(dir_name, write_compact_diff=write_compact_diff, pair_head_template="response {obj_index}", obj_head_template="query")

        printer.on_new_pair()
        printer.sched('title')
        printer.sched('subtitle')
        printer('line->lines', printers.DiffType.CHANGED)
        printer.desched()
        printer.desched()

        printer.sched('name')
        printer('wow', printers.DiffType.REMOVED)
        printer.desched()

        printer.on_new_pair()
        printer.sched('title')
        printer.sched('subtitle')
        printer('new_str', printers.DiffType.ADDED)
        printer.desched()
        printer.desched()

        printer.finalize()

        files = os.listdir(dir_name)
        if not write_compact_diff:
            assert 'compact_diff' not in files, '{} must not contain compact_diff'.format(files)
            return

        common.assert_issubset(['compact_diff'], files)
        common.assert_equal(['name.html', 'title_subtitle.html'], sorted(os.listdir(os.path.join(dir_name, 'compact_diff'))))

        title_content = common.read_file(os.path.join(dir_name, 'compact_diff', 'title_subtitle.html'))
        common.assert_string_contains(title_content, 'title/subtitle')
        common.assert_string_contains(title_content, 'amount of changes = 2')
        common.assert_string_not_contains(title_content, '->')
        # check that word 'line' is written twice in diff (because it changed)
        first_line_pos = title_content.find('line')
        assert first_line_pos != -1
        second_line_pos = title_content.find('line', first_line_pos + 1)
        assert first_line_pos != second_line_pos
        assert second_line_pos != -1
        # changes (letter 's' was added) are written in different style
        common.assert_string_contains(title_content, '>s<')
        common.assert_string_contains(title_content, 'response 0')
        common.assert_string_contains(title_content, 'response 1')

        name_content = common.read_file(os.path.join(dir_name, 'compact_diff', 'name.html'))
        common.assert_string_contains(name_content, 'name')
        common.assert_string_contains(name_content, 'amount of changes = 1')
        common.assert_string_contains(name_content, 'wow')
        common.assert_string_contains(name_content, 'response 0')

    def test_split_files(self, split):
        dir_name = common.make_tmp_dir('test_split_files')
        printer = printers.PrinterToHtml(dir_name, max_file_size=10 if split else 1024 * 1024)

        printer.on_new_pair('pair 0')
        printer.sched('path0')
        printer('new_str0', printers.DiffType.ADDED)
        printer.desched()

        printer.on_new_pair('pair 1')
        printer.sched('path1')
        printer('new_str1', printers.DiffType.ADDED)
        printer.desched()

        printer.finalize()

        if split:
            common.assert_equal(sorted(['r_000000-000000.html', 'r_000001-000001.html', 'response_diff_scripts.js', 'response_diff_style.css', 'compact_diff']),
                                sorted(os.listdir(dir_name)))
        else:
            common.assert_equal(sorted(['r_000000-000001.html', 'response_diff_scripts.js', 'response_diff_style.css', 'compact_diff']),
                                sorted(os.listdir(dir_name)))

        pair_0_content = common.read_file(os.path.join(dir_name, 'r_000000-000000.html' if split else 'r_000000-000001.html'))
        pair_1_content = common.read_file(os.path.join(dir_name, 'r_000001-000001.html' if split else 'r_000000-000001.html'))

        common.assert_string_contains(pair_0_content, 'pair 0')
        common.assert_string_contains(pair_0_content, 'path0')
        common.assert_string_contains(pair_0_content, 'new_str0')

        common.assert_string_contains(pair_1_content, 'pair 1')
        common.assert_string_contains(pair_1_content, 'path1')
        common.assert_string_contains(pair_1_content, 'new_str1')

    def test_parallel_subprinter(self, with_compact_diff, no_diff_at_all):
        dir_name_sequencial = common.make_tmp_dir('test_parallel_subprinter_sequencial')
        dir_name_parallel = common.make_tmp_dir('test_parallel_subprinter_parallel')
        sequencial_printer = printers.PrinterToHtml(dir_name_sequencial,
                                                    write_compact_diff=with_compact_diff,
                                                    pair_head_template="p {obj_index}",
                                                    obj_head_template="q",
                                                    obj_title_template="**{title}**",
                                                    pair_range_filename_template="rrr_{obj_index_to:0>6}-{obj_index_from:0>6}.html")
        parallel_printer = printers.PrinterToHtml(dir_name_parallel,
                                                  write_compact_diff=with_compact_diff,
                                                  pair_head_template="p {obj_index}",
                                                  obj_head_template="q",
                                                  obj_title_template="**{title}**",
                                                  pair_range_filename_template="rrr_{obj_index_to:0>6}-{obj_index_from:0>6}.html")

        assert parallel_printer.supports_parallel_processing

        def add_pair(printer, is_parallel, pair_num, diff_class):
            printer.on_new_pair('pair ' + str(pair_num))
            printer.sched('path' + str(pair_num))
            if diff_class != printers.DiffType.NODIFF and not no_diff_at_all:
                if diff_class == printers.DiffType.CHANGED:
                    printer(('change_from_' + str(pair_num), 'change_to_' + str(pair_num)), diff_class)
                else:
                    printer('changed_str' + str(pair_num), diff_class)
            printer.desched()
            if is_parallel:
                printer.finalize()

        subprinter0 = parallel_printer.get_parallel_subprinter()
        add_pair(sequencial_printer, False, 0, printers.DiffType.CHANGED)
        add_pair(subprinter0, True, 0, printers.DiffType.CHANGED)

        subprinter1 = parallel_printer.get_parallel_subprinter()
        add_pair(sequencial_printer, False, 1, printers.DiffType.ADDED)
        add_pair(subprinter0, True, 1, printers.DiffType.ADDED)

        subprinter2 = parallel_printer.get_parallel_subprinter()
        add_pair(sequencial_printer, False, 2, printers.DiffType.REMOVED)
        add_pair(subprinter0, True, 2, printers.DiffType.REMOVED)

        subprinter3 = parallel_printer.get_parallel_subprinter()
        add_pair(sequencial_printer, False, 3, printers.DiffType.NODIFF)
        add_pair(subprinter0, True, 3, printers.DiffType.NODIFF)

        subprinter4 = parallel_printer.get_parallel_subprinter()
        add_pair(sequencial_printer, False, 1, printers.DiffType.ADDED)  # the same as 1
        add_pair(subprinter0, True, 1, printers.DiffType.ADDED)  # the same as 1

        sequencial_printer.finalize()
        subprinters = [subprinter0, subprinter1, subprinter2, subprinter3, subprinter4]
        if not with_compact_diff:
            # Check that subprinters has no compact diff
            for subprinter in subprinters:
                common.assert_equal({}, subprinter.compact_diff)
        parallel_printer.join_parallel_subprinters(subprinters)
        parallel_printer.finalize()

        common.assert_directories_equal(dir_name_sequencial, dir_name_parallel)

    def test_all(self):
        self.test_no_diff(False, False)
        self.test_no_diff(True, False)
        self.test_no_diff(True, True)
        self.test_helper_files()
        self.test_compact_diff(False)
        self.test_compact_diff(True)
        self.test_split_files(True)
        self.test_split_files(False)
        self.test_parallel_subprinter(False, False)
        self.test_parallel_subprinter(False, True)
        self.test_parallel_subprinter(True, False)
        self.test_parallel_subprinter(True, True)


def test_all():
    test_base = TestPrinterBase()
    test_base.test_all()

    test_html = TestHtmlPrinter()
    test_html.test_all()
