# -*- coding: utf-8 -*-

import os
import re
import logging
import gzip
import tempfile
import itertools
import uuid
import contextlib
import json

from six.moves.urllib import parse as urlparse

from sandbox import common

import sandbox.sandboxsdk.environments as env
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects import resource_types
from . import resources as dolbilka_resources


DOLBILKA_GROUP = 'Dolbilka parameters'
"""
    Группа для параметров долбилки
"""

_HTTP_URL = 'http://localhost:12345'
"""
    HTTP-хост используемый по умолчанию
"""

_HTTP_COLLECTION = 'yandsearch'
"""
    HTTP-коллекция по умолчанию
"""

_HTTP_REQUEST_LINE = re.compile(r'GET (/[^?]+)(\?[^\s]+)\s+HTTP/[0-9.]+')
"""
    Шаблон строки запроса
"""


class DolbilkaParametersError(common.errors.TaskError):
    """ Error working with dolbilka """


def _ensure_file_exists(file_name, file_type):
    file_name = os.path.abspath(file_name)
    if not os.path.exists(file_name):
        raise DolbilkaParametersError("Cannot find {}: '{}'".format(file_type, file_name))
    if not os.path.isfile(file_name):
        raise DolbilkaParametersError("This path is not a file: '{}'".format(file_name))
    return file_name


class DolbilkaPlanner(object):
    """
        Обёртка для d-planner
    """

    name = 'd_planner'
    LOADER_TYPES = ('plain', 'eventlog', 'accesslog', 'fullreqs', 'planlog', 'loadlog', 'phantom', 'pcap')
    resource_type = dolbilka_resources.DPLANNER_EXECUTABLE

    def __init__(self):
        self.path = self.get_planner_path()

    @classmethod
    def get_planner_path(cls):
        """
            Синхронизировать ресурс последнего релиза planner-а и получить путь до него
            :return: путь до исполняемого файла
            :rtype: str
        """
        return utils.sync_last_stable_resource(cls.resource_type)

    def create_plan(
        self, queries_path,
        result_path=None,
        rps=None,
        loader_type='plain',
        host=None, port=None,
        log_prefix='d_planner_create_plan'
    ):
        """
            Создать план для долбилки

            :param queries_path: путь до файлов с запросами для плана
            :param result_path: путь создаваемого плана.
                                Если не указан, используется путь queries с добавлением в конце .plan
            :param rps: уровень rps в plan режиме в создаваемом плане
            :param loader_type: указывать ли при запуске планнера тип нагрузки (см. DolbilkaPlanner.LOADER_TYPES)
            :param host: название хоста, которое нужно использовать в плане
            :param port: порт, который нужно использовать в плане
            :param log_prefix: префикс для логов
            :return: путь до созданного плана в виде строки
        """
        queries_path = os.path.abspath(queries_path)
        if result_path:
            result_path = os.path.abspath(result_path)
        else:
            result_path = queries_path + '.plan'

        if host is None and loader_type == 'plain':
            # Если сервер и порт не указаны, dplanner считает, что подаваемые на вход строки
            # должны быть валидными урлами вида http://blablabla/some/path?param1=value1&p2=val2...
            # Если строка не удовлетворяет его требованиям, он печатает в stderr ошибку,
            # игнорирует данную строку и продолжает работу, т.е. код возврата равен нулю.
            # Поэтому, если подать на вход неправильные урлы, мы на выходе получим пустой план.
            # Проверка ниже позволяет обнаруживать эту ситуацию чуть раньше и выдавать
            # вменяемое сообщение об ошибке (SEARCH-810).
            if queries_path.endswith(".gz"):
                queries_file = gzip.open(queries_path, 'rb')
            else:
                queries_file = open(queries_path)
            for line in queries_file:
                if '://' not in line:
                    logging.info("Invalid url for dolbilka planner: '%s'", line.strip())
                    # Так как это по сути проблема входных параметров,
                    # данный тип исключения вполне годится в данном случае.
                    # Валить весь таск в FAILURE было бы неправильно.
                    raise DolbilkaParametersError(
                        "Source queries for dolbilka should be valid urls "
                        "when host is not specified. See line sample in task logs. "
                    )

        run_planner_parameters = [self.path, '-l', queries_path, '-o', result_path]
        if loader_type:
            run_planner_parameters.append('-t')
            run_planner_parameters.append(self.check_loader_type(loader_type))
            if rps and loader_type == 'plain':  # параметр rps применим только для plain режима
                run_planner_parameters.append('-q')
                run_planner_parameters.append(self.check_rps(rps))
        if host:
            run_planner_parameters.append('-h')
            run_planner_parameters.append(str(host))
        if port:
            run_planner_parameters.append('-p')
            run_planner_parameters.append(str(port))
        process.run_process(run_planner_parameters, log_prefix=log_prefix)
        return result_path

    def check_loader_type(self, loader_type):
        """
            Проверить значение параметра check_loader_type
        """
        if loader_type in self.LOADER_TYPES:
            return loader_type
        else:
            raise DolbilkaParametersError(
                "DPlanner error. Incorrect loader_type parameter: '{}'".format(loader_type)
            )

    @staticmethod
    def fill_rps_ctx(results, ctx):

        def _get_rate(result, field_name):
            try:
                rate = 0
                count = result.get(field_name, None)
                if count:
                    # reasonable request count order is k*10^5, so round up to 6 digits is OK
                    rate = round(float(count) / int(result['requests']), 6)
            except (ValueError, ZeroDivisionError, KeyError):
                rate = 2  # 200% of errorness in output
                logging.error(
                    ("rate calculation failed: {%s}, and {requests} requests"
                     % field_name).format(**result)
                )
            return rate

        ctx['results'] = results
        ctx['requests_per_sec'] = []
        ctx['fail_rates'] = []
        ctx['notfound_rates'] = []
        for result in results:
            try:
                rps = float(result['rps'])
            except (ValueError, KeyError):
                rps = 0.0
            fail_rate = _get_rate(result, "service_unavailable")
            notfound_rate = _get_rate(result, "requests_not_found")
            ctx['requests_per_sec'].append(rps)
            ctx['fail_rates'].append(fail_rate)
            ctx['notfound_rates'].append(notfound_rate)
        ctx['max_rps'] = max(ctx['requests_per_sec'])
        ctx['max_fail_rate'] = max(ctx['fail_rates'])
        ctx['min_fail_rate'] = min(ctx['fail_rates'])
        ctx['min_notfound_rate'] = min(ctx['notfound_rates'])

    @staticmethod
    def check_rps(rps):
        """
            Проверить значение параметра rps
        """
        try:
            if not int(rps) > 0:
                raise DolbilkaParametersError()
            return str(rps)
        except (ValueError, DolbilkaParametersError):
            raise DolbilkaParametersError('DPlanner error. Incorrect rps parameter: {}'.format(rps))


class DolbilkaExecutorResource(parameters.ResourceSelector):
    name = 'dolbilo_executor_resource_id'
    description = 'Dolbilo executor'
    resource_type = dolbilka_resources.DEXECUTOR_EXECUTABLE
    group = DOLBILKA_GROUP
    required = False


class DolbilkaExecutorRequestsLimit(parameters.SandboxIntegerParameter):
    name = 'dolbilka_executor_requests_limit'
    description = 'Requests limit'
    group = DOLBILKA_GROUP
    default_value = 10000
    dolbilka_name = "queries-limit"


class DolbilkaExecutorTimeLimit(parameters.SandboxIntegerParameter):
    name = 'dolbilka_executor_time_limit'
    description = 'Overall time limit (in seconds)'
    group = DOLBILKA_GROUP
    default_value = 0
    dolbilka_name = "time-limit"


class DolbilkaSessionsCount(parameters.SandboxIntegerParameter):
    name = 'dolbilka_executor_sessions'
    description = 'Total sessions'
    group = DOLBILKA_GROUP
    default_value = 1


class DolbilkaMaximumSimultaneousRequests(parameters.SandboxIntegerParameter):
    name = 'dolbilka_executor_max_simultaneous_requests'
    description = 'Max simultaneous requests'
    group = DOLBILKA_GROUP
    default_value = 26  # Reasonable value: ~ 80% NCPU
    dolbilka_name = 'simultaneous'


class DolbilkaDelayMultiplier(parameters.SandboxFloatParameter):
    name = 'dolbilka_executor_delay_multiplier'
    description = 'Plan delay multiplier'
    group = DOLBILKA_GROUP
    dolbilka_name = 'multiplier'


class DolbilkaBounds(parameters.SandboxStringParameter):
    name = 'dolbilka_executor_bounds'
    description = 'Fixed time bounds in dump, in microseconds'
    group = DOLBILKA_GROUP
    default_value = ''


class DolbilkaFixedRps(parameters.SandboxIntegerParameter):
    name = 'dolbilka_fixed_rps'
    description = 'Fixed RPS (use 0 for default)'
    group = DOLBILKA_GROUP
    default_value = 0
    dolbilka_name = 'rps-fixed'


class DolbilkaScheduledRps(parameters.SandboxStringParameter):
    name = 'dolbilka_scheduled_rps'
    description = 'RPS schedule (leave empty for default)'
    group = DOLBILKA_GROUP
    default_value = ''
    dolbilka_name = 'rps-schedule'


class DolbilkaRequestTimeout(parameters.SandboxIntegerParameter):
    name = 'dolbilka_request_timeout'
    description = 'Request timeout'
    group = DOLBILKA_GROUP
    default_value = 600
    dolbilka_name = 'timeout'


class DolbilkaAugmentUrl(parameters.SandboxStringParameter):
    name = 'dolbilka_augment_url'
    description = 'String to append to each request URL'
    group = DOLBILKA_GROUP
    default_value = ''
    dolbilka_name = 'augmenturl'


class DolbilkaPlanMemory(parameters.SandboxBoolParameter):
    name = 'dolbilka_plan_memory'
    description = 'Load full plan in memory'
    group = DOLBILKA_GROUP
    default_value = False
    dolbilka_name = 'plan-memory'


class DolbilkaExecutorMode(parameters.SandboxStringParameter):
    PLAN_MODE = 'plan'
    DEVASTATE_MODE = 'devastate'
    BINARY_MODE = 'binary'
    FINGER_MODE = 'finger'

    MODES = (
        PLAN_MODE,
        DEVASTATE_MODE,
        BINARY_MODE,
        FINGER_MODE,
    )

    name = 'dolbilka_executor_mode'
    description = 'Executor mode'
    group = DOLBILKA_GROUP
    default_value = FINGER_MODE
    choices = [
        ('Plan', PLAN_MODE),
        ('Devastate', DEVASTATE_MODE),
        ('Binary', BINARY_MODE),
        ('Finger', FINGER_MODE),
    ]
    sub_fields = {
        DEVASTATE_MODE: [DolbilkaMaximumSimultaneousRequests.name],
        PLAN_MODE: [DolbilkaDelayMultiplier.name, DolbilkaFixedRps.name, DolbilkaScheduledRps.name],
        FINGER_MODE: [DolbilkaMaximumSimultaneousRequests.name],
    }
    dolbilka_name = 'mode'


class DolbilkaOutputLenval32(parameters.SandboxBoolParameter):
    name = 'dolbilka_output_lenval32'
    description = 'Dump responses in lenval32 format (SEARCH-2041)'
    group = DOLBILKA_GROUP
    default_value = False
    dolbilka_name = 'output-lenval32'


class DolbilkaExecutor(object):
    """
        Обёртка для d-executor
    """
    name = 'd_executor'
    resource_type = dolbilka_resources.DEXECUTOR_EXECUTABLE

    dolbilka_main_params = (
        DolbilkaExecutorResource,
        DolbilkaSessionsCount,
        DolbilkaBounds,
    )
    dolbilka_args_params = (
        DolbilkaExecutorMode,
        DolbilkaExecutorRequestsLimit,
        DolbilkaExecutorTimeLimit,
        DolbilkaMaximumSimultaneousRequests,
        DolbilkaFixedRps,
        DolbilkaScheduledRps,
        DolbilkaDelayMultiplier,
        DolbilkaRequestTimeout,
        DolbilkaAugmentUrl,
        DolbilkaPlanMemory,
        DolbilkaOutputLenval32,
    )
    input_task_parameters = dolbilka_main_params + dolbilka_args_params

    def __init__(self):
        ctx = channel.task.ctx

        executor_resource_id = utils.get_or_default(ctx, DolbilkaExecutorResource)
        if executor_resource_id:
            self.path = channel.task.sync_resource(executor_resource_id)
        else:
            self.path = self.get_executor_path()

        self.mode = utils.get_or_default(ctx, DolbilkaExecutorMode)
        if self.mode not in DolbilkaExecutorMode.MODES:
            raise DolbilkaParametersError(
                "DExecutor error. Incorrect mode parameter: '{}'".format(self.mode)
            )

        self.delay_multiplier = None
        self.sessions = int(utils.get_or_default(ctx, DolbilkaSessionsCount))
        self.bounds = utils.get_or_default(ctx, DolbilkaBounds).strip()

        if self.bounds:
            try:
                # Проверка, что границы - отсортированные по возрастанию числа
                bounds = self.bounds.split(",")
                bounds = map(int, bounds)
                previous_bound = 0
                for bound in bounds:
                    if bound <= previous_bound:
                        raise DolbilkaParametersError(
                            'Bound "{}" must be greater than "{}"'.format(bound, previous_bound)
                        )
                    previous_bound = bound
            except ValueError:
                raise DolbilkaParametersError('Bad bounds value "{}"'.format(self.bounds))

        self.__dolbilka_args = {}
        for param in self.dolbilka_args_params:
            value = utils.get_or_default(ctx, param)
            if value:
                self.__dolbilka_args[param.dolbilka_name] = value

        self.__dumper = None
        self.__planner = None

        # for backward compatibility
        self.requests = None
        self.max_simultaneous_requests = None
        self.augment_url = None

    @classmethod
    def get_executor_path(cls):
        """
            Синхронизировать ресурс последнего релиза executor-а и получить путь до него
            :return: путь до исполняемого файла
            :rtype: str
        """
        return utils.sync_last_stable_resource(cls.resource_type)

    @property
    def dumper(self):
        if self.__dumper is None:
            self.__dumper = DolbilkaDumper()
        return self.__dumper

    @property
    def planner(self):
        if self.__planner is None:
            self.__planner = DolbilkaPlanner()
        return self.__planner

    @staticmethod
    def check_plan(plan):
        """
            Проверить план долбилки

            :param plan: путь до файла плана
            :return: абсолютный путь до плана, если всё в порядке;
                     в противном случае выбрасывается исключение
        """
        return _ensure_file_exists(plan, 'plan')

    def run_session(
            self,
            plan,
            dump=None,
            host='localhost',
            port=None,
            log_prefix='d_executor_run_session',
            phantom=False,
            save_answers=False,
            circular=True
    ):
        """
           Запустить одну сессию обстрела

           :param plan: путь до файла с планом
           :param dump: путь до дампа результатов
           :param host: заменить в запросах значение хоста на указанное при обстреле
           :param port: заменить в запросах значение порта на указанное при обстреле
           :param log_prefix: префикс для логов
           :param phantom: формировать дамп в формате, совместимом с `phout.txt`
           :param save_answers: сохранять ответы из target
           :param circular: если True, то в режимах DEVASTATE_MODE и FINGER_MODE
                  продолжать долбить запросы по кругу, ровно столько, сколько указано в --queries-limit,
                  если запросов в плане меньше, чем мы попросили
           :return: путь до дампа
        """
        plan = self.check_plan(plan)

        if dump:
            dump = os.path.abspath(dump)
        else:
            dump = plan + '.dump'

        run_executor_parameters = [
            self.path,
            '--plan-file', plan,
            '--output', dump,
        ]

        # for backward compatibility
        if self.requests:
            self.__dolbilka_args[DolbilkaExecutorRequestsLimit.dolbilka_name] = self.requests
        if self.max_simultaneous_requests:
            self.__dolbilka_args[DolbilkaMaximumSimultaneousRequests.dolbilka_name] = self.max_simultaneous_requests
        if self.augment_url:
            self.__dolbilka_args[DolbilkaAugmentUrl.dolbilka_name] = self.augment_url

        for key, value in self.__dolbilka_args.iteritems():
            run_executor_parameters.append("--" + key)
            if not isinstance(value, bool):
                run_executor_parameters.append(str(value))

        def append_parameter(name, value):
            if value:
                run_executor_parameters.append(name)
                run_executor_parameters.append(str(value))

        append_parameter('--replace-host', host)
        append_parameter('--replace-port', port)
        if self.mode:
            append_parameter('--mode', self.mode)
            # параметр количества одинаковых потоков только для devastate и finger режимов
            if self.mode in (DolbilkaExecutorMode.DEVASTATE_MODE, DolbilkaExecutorMode.FINGER_MODE):
                if circular:
                    run_executor_parameters.append('--circular')
            # параметр задержки между запросами только для plan режима
            elif self.mode == DolbilkaExecutorMode.PLAN_MODE:
                append_parameter('--multiplier', self.delay_multiplier)

        if phantom:
            run_executor_parameters.append('--phantom')
        if save_answers:
            run_executor_parameters.append('-d')
        process.run_process(run_executor_parameters, log_prefix=log_prefix)
        return dump

    def get_plan_requests(self, plan, queries=None, result_file=None, debug_mode=False):
        """
            Получить запросы из плана.

            :param plan: путь до файла плана
            :param queries: количество запросов
            :param result_file: файл, куда записать результаты
            :param debug_mode: сохраняет промежуточные результаты в ресурс
            :return: путь до файла с запросами
        """
        plan = self.check_plan(plan)
        if result_file:
            result_file = os.path.abspath(result_file)
        else:
            result_file = plan + '.requests'

        run_executor_parameters = [self.path, '-p', plan, '-H', 'localhost', '-D']
        if queries:
            run_executor_parameters.extend(['-Q', str(queries)])

        if debug_mode:
            debug_result_path = result_file + ".debug"
            with open(debug_result_path, 'w') as requests_file:
                process.run_process(run_executor_parameters, stdout=requests_file, shell=True)

            logging.info('Dumped file size: %s', os.path.getsize(debug_result_path))

            resource = channel.task.create_resource(
                "Dolbilka debug result",
                debug_result_path,
                resource_type=resource_types.OTHER_RESOURCE,
            )
            logging.info('Debug resource id: %s', resource.id)

        run_executor_parameters.extend(['|', 'grep', 'GET', '|', 'awk', "'{print $2}'"])

        with open(result_file, 'w') as requests_file:
            process.run_process(run_executor_parameters, stdout=requests_file, shell=True)

    def run_sessions(
        self,
        plan,
        target,
        host='localhost',
        run_once=False,
        need_warmup=False,
        callback=None,
        delay_multiplier=1.0,
        phantom=False,
        save_answers=False
    ):
        """
            Запустить несколько сессий обстрела объекта target с планом plan

            :param plan: путь до файла с планом
            :param target: поисковый компонент
            :param host: заменить в запросах значение хоста на указанное при обстреле
            :param run_once: нужно ли запускать поисковый компонент один раз или перезапускать после каждой сессии
            :param save_answers: сохранять ответы из target
            :need_warmup: нужно ли запускать дополнительную сессию вначале для прогрева дискового кеша
            :callback: функция будет вызываться после каждой сессии и получать на вход номер сессии начиная с нуля
            :return: список с dict с информацией о прошедших сессиях стрельбы, см. DolbilkaDumper::parse_dolbilka_stat
        """
        result = []
        self.delay_multiplier = delay_multiplier
        sessions = self.sessions
        if need_warmup:
            parsed_stat = self.run_session_and_dumper(plan, target, "warmup", run_once=False, phantom=phantom)
            result.append(parsed_stat)
        if run_once:
            target.start()
            target.wait()
        for i in range(sessions):
            parsed_stat = self.run_session_and_dumper(
                plan,
                target,
                str(i),
                host,
                run_once,
                phantom=phantom,
                save_answers=save_answers
            )
            result.append(parsed_stat)
            if callback:
                callback(i)
        if run_once:
            target.stop()
        return result

    def run_session_and_dumper(
        self,
        plan,
        target,
        session_name,
        host='localhost',
        run_once=False,
        phantom=False,
        save_answers=False,
    ):
        """
            Запустить одну сессию обстрела и после распарсить результаты

            :param plan: путь до файла с планом
            :param target: поисковый компонент
            :param session_name: имя сессии, должно быть уникально при каждом вызове
            :param host: заменить в запросах значение хоста на указанное при обстреле
            :param run_once: нужно ли запускать поисковый компонент один раз или перезапускать после каждой сессии
            :param phantom: Использовать выхлопной формат phantom
            :param save_answers: сохранять ответы из target
            :return: dict с информацией о стрельбе, см. DolbilkaDumper::parse_dolbilka_stat
        """
        session_id = 'test_{}_session_{}'.format(target.name, session_name)
        resource_name = 'test {} session {}'.format(target.name, session_name)
        logging.info('Run dolbilka test session %s', session_id)

        dump_path = channel.task.abs_path('{}.dump'.format(session_id))
        if not run_once:
            target.start()
            target.wait()

        target.use_component(
            lambda: self.run_session(
                plan=plan,
                host=host,
                dump=dump_path,
                port=target.port,
                log_prefix='run_d_executor_{}'.format(session_id),
                phantom=phantom,
                save_answers=save_answers,
            )
        )

        venv_working_dir = 'venv_d_executor_{}'.format(session_id)
        try:
            os.mkdir(venv_working_dir)
        except OSError:
            pass
        self._save_memory_usage(target, venv_working_dir)

        if not run_once:
            target.stop()

        if phantom:
            parsed_stat = {}
            channel.task.create_resource(
                resource_name,
                dump_path,
                resource_type=resource_types.EXECUTOR_STAT_PHOUT,
            )
        else:
            parsed_stat = self.run_dumper(
                dump_path, resource_name,
                bounds=self.bounds,
                session_name=session_name,
            )
            if save_answers:
                attributes = {'session_name': session_name}
                channel.task.create_resource(
                    'dolbilka_dump_with_target_answers',
                    dump_path,
                    resource_types.DOLBILKA_RESPONSES,
                    attributes=attributes
                )
            else:
                # dolbilka dump files are big, so we remove them immediately after parsing
                os.remove(dump_path)  # see SEARCH-1816 for details
        parsed_stat['session_id'] = session_id

        return parsed_stat

    def run_dumper(
        self, dump_path, stat_resource_descr,
        bounds=None,
        session_name=None,
    ):
        result_stat = self.dumper.get_results(dump_path, parse_results=False, bounds=bounds)
        parsed_stat = self.dumper.parse_dolbilka_stat(result_stat)

        stat_file_name = os.path.splitext(os.path.basename(dump_path))[0] + '.stat'

        # stats are useful, store them within 4 months
        attributes = {'ttl': 120}

        if session_name is not None:
            attributes['session_name'] = session_name

        resource = channel.task.create_resource(
            stat_resource_descr,
            channel.task.abs_path(stat_file_name),
            resource_type=resource_types.EXECUTOR_STAT,
            attributes=attributes,
        )
        fu.write_file(resource.path, result_stat)

        return parsed_stat

    @staticmethod
    def _save_memory_usage(target, working_dir='.'):
        """
            Сохраняет использование памяти для дальнейшего отображения в шаблоне

            rss - resident set size, the non-swapped physical memory that a task has used
            vsz - virtual memory size of the process in KiB (1024-byte units).
            Device mappings are currently excluded; this is subject to change.

            :param target: поисковый компонент
        """
        if not target.process:
            return

        with env.VirtualEnvironment(working_dir) as venv:  # todo: get rid of this hack after updating psutil in arcadia
            logging.info('Installing psutil')
            env.PipEnvironment('psutil', version="5.4.3", venv=venv, use_wheel=True).prepare()
            p = process.run_process(
                [venv.executable, os.path.dirname(__file__) + "/memory_usage.py", "-p", str(target.process.pid)],
                outs_to_pipe=True,
            )
            info = json.loads(p.communicate()[0])
            info["anon"] = info["rss"] - info["shared"]

        try:
            # get amount of lockecd memory - linux specific
            proc_path = "/proc/" + str(target.process.pid) + "/status"
            info["vmlck"] = -1
            with open(proc_path, "r") as f:
                for l in f:
                    if l.startswith("VmLck:"):
                        info["vmlck"] = int(l[6:].strip().split(' ')[0]) * 1024  # to be consistent with others, convert to bytes
                        break
        except Exception:
            logging.error("Unable to obtain locked memory size")

        if 'memory_rss' not in channel.task.ctx:
            channel.task.ctx['memory_rss'] = []
        channel.task.ctx['memory_rss'].append(int(info['rss']) / 1024)

        if 'memory_vsz' not in channel.task.ctx:
            channel.task.ctx['memory_vsz'] = []
        channel.task.ctx['memory_vsz'].append(int(info['vms']) / 1024)

        if 'memory_bytes' not in channel.task.ctx:
            channel.task.ctx['memory_bytes'] = []
        channel.task.ctx['memory_bytes'].append(info)


class DolbilkaDumper(object):
    """
        Обёртка для d-dumper
    """
    name = 'd_dumper'
    resource_type = dolbilka_resources.DDUMPER_EXECUTABLE

    @classmethod
    def find_parameter_in_dolbilka_stat(cls, stat_text, template):
        """
            Ищет в text по шаблону parameter_template
            Возвращается значение найденной группы или None

            :param stat_text: текст для поиска
            :param template: скомпилированное регулярное выражение
            :return: если по заданному регулярному выражению было что-то найдено, возвращается значение;
                     в противном случае возвращается None
        """
        parameter_template = re.compile(template)
        parameter_value = parameter_template.search(stat_text)
        if parameter_value:
            return parameter_value.group(1).strip()
        else:
            return None

    @classmethod
    def parse_dolbilka_stat(cls, text):
        """
            Анализирует вывод d-dumper-а, переданного в text

            Возвращаемое значение - dict вида:
            {
                'start_time': 'Время начала обстрела',
                'end_time': 'Время завершения обстрела',
                'full_time': 'Время обстреля',
                'data_readed': 'Сколько данных прочитано в байтах',
                'requests': 'Количество запросов',
                'error_requests': 'Количество error запросов',
                'rps': 'Показатель запросов в секунду',
                'average_request_time': 'Среднее вермя обработки запроса',
                'avg req. len': 'Средняя длина запроса',
                'request_time_deviation': 'Отклоннение в вычислении времени работы запроса',
                'requests_moved': 'Запросов перемещено',
                'requests_not_found': 'Запросов не найдено',
                'requests_ok': 'Успешных запросов',
                'bad_gateway': 'Ошибка HTTP-прокси'
                'service_unavailable': 'Поиск недоступен'
            }
            :param text: текст вывода d-dumper-а
            :return: dict с результатами, его ключи:

        """
        result = {}
        parameters_templates = {
            'start_time': 'start time(.*),',
            'end_time': 'end time(.*),',
            'full_time': 'full time(.*),',
            'data_readed': 'data readed(.*)',
            'requests': 'requests(.*)',
            'error_requests': 'error requests(.*)',
            'rps': 'requests/sec(.*)',
            'average_request_time': 'avg req. time(.*)',
            'avg req. len': 'avg req. len(.*)',
            'request_time_deviation': 'req time std deviation(.*)',
            'requests_moved': 'Moved temporarily(.*)',
            'requests_not_found': 'Not found(.*)',
            'bad_gateway': 'Bad gateway(.*)',
            'requests_ok': 'Ok(.*)',
            'service_unavailable': "Service unavailable(.*)",
        }
        for key, value in parameters_templates.items():
            result[key] = cls.find_parameter_in_dolbilka_stat(text, value)
        return result

    def __init__(self):
        self.path = self.get_dumper_path()

    @classmethod
    def get_dumper_path(cls):
        """
            Синхронизировать ресурс последнего релиза dumper-а и получить путь до него
            :return: путь до исполняемого файла
        """
        return utils.sync_last_stable_resource(cls.resource_type)

    @staticmethod
    def check_dump(dump):
        """
            Проверить дамп долбилки
            :param dump: путь до файла дампа
            :return: абсолютный путь до дампа, если всё в порядке; в противном случае выбрасывается исключение
        """
        return _ensure_file_exists(dump, 'dump')

    def get_results(self, dump, parse_results=False, bounds=None):
        """
            Получить результаты из дампа по заданному пути

            :param dump: путь до дампа
            :param bounds: границы времени, по которым считается % запросов
            :return: вывод дампера в виде строки
        """
        dump = self.check_dump(dump)
        run_dumper_parameters = [self.path, '-a', '-f', dump]
        if bounds:
            run_dumper_parameters.extend(['-m', bounds])
        p = process.run_process(run_dumper_parameters, outs_to_pipe=True)
        result = p.stdout.read()
        if parse_results:
            return self.parse_dolbilka_stat(result)
        return result


def get_queries_from_plan(plan_file_path, out_path, num_queries=0, store_collection=False):
    """
        Получить запросы из плана.

        :param plan_file_path: путь до файла плана
        :param out_path: файл, куда записать результаты
        :param num_queries: количество запросов
        :param store_collection: сохраняет промежуточные результаты в ресурс
        :return: количество полученных запросов
    """
    cmd = '{} -D -p {}'.format(DolbilkaExecutor.get_executor_path(), plan_file_path)
    if num_queries > 0:
        cmd += " -Q {}".format(num_queries)

    logging.info("Getting queries from plan. Plan size is {} bytes".format(os.path.getsize(plan_file_path)))

    counter = 0
    with tempfile.NamedTemporaryFile() as dump_file:
        process.run_process(cmd, wait=True, check=True, stdout=dump_file)

        logging.info("Parsing shots dump ({} bytes)".format(os.path.getsize(dump_file.name)))
        dump_file.seek(0)
        with open(out_path, "w") as out_file:
            for line in dump_file:
                matcher = _HTTP_REQUEST_LINE.match(line)
                if matcher is None:
                    continue
                query = ""
                if store_collection:
                    query += matcher.group(1)
                query += matcher.group(2)
                out_file.write(query + "\n")

                counter += 1

    return counter


def patch_queries(in_file_path, out_file_path, add_cgi_str, remove_cgi_str, queries_limit=None):
    """
        Добавить и/или удалить cgi параметры из запросов
        :param in_file_path: входной файл с запросами
        :param out_file_path: выходной файл с патченными запросами
        :param add_cgi_str: какой параметр следует добавить к запросам
            Если передавать параметры через незаэкранированную ';' - они будут циклически
            дописываться к запросам по одному параметру в запрос.
        :param remove_cgi_str: какой параметр следует удалить из запросов
        :param queries_limit: максимальное количество запросов. None - без ограничений
        :return: количество запросов записанное в патченный файл
    """
    if not add_cgi_str.startswith("&"):
        add_cgi_str = "&" + add_cgi_str
    with open(in_file_path) as in_file:
        patched_iter = itertools.islice(
            queries_generator(
                in_file,
                add_cgi_str,
                remove_cgi_str,
            ),
            queries_limit,
        )
        return fu.write_lines(out_file_path, patched_iter)


def queries_generator(in_file, add_cgi_str, remove_cgi_str):
    # this will also properly handle empty string too
    cgi_params_iterator = itertools.cycle(add_cgi_str.split(";")).__iter__()
    remove_cgi = remove_cgi_str.split(';') if remove_cgi_str else []

    logging.info("Patching queries...")
    # plan can be heavy, so we don't read it into memory
    for line in in_file:
        result = line.strip("\n\r")
        if not result:
            continue
        for cgi_item in remove_cgi:
            if '*' in cgi_item:
                # regex detected
                cgi_item = cgi_item.replace('*', '')
                result = re.sub(cgi_item + r'[^&]+', '&', result)
            else:
                result = result.replace(cgi_item, '')
        result += cgi_params_iterator.next()
        yield result


def convert_queries_to_plan(
    queries_file_name, plan_file_name,
    alter_query=lambda r: r,
    rps=None,
    max_queries=0,
    loader_type='plain',
):
    """
        Обёртка для преобразования запросов в план.

        На вход можно подавать запросы в следующих форматах:
            * PLAIN_TEXT_QUERIES (?cgi1=value1&cgi2=value2....)
            * полноценные урлы вида http://host:port/collection?cgi1=value1&cgi2=value2....
            * запросы в формате <timedelta><TAB><query> (запрос с временной отметкой
                в микросекундах)

        :param queries_file_name: входной файл с текстовыми запросами (если оканчивается на .gz,
            интерпретируется как gzip)
        :param plan_file_name: выходной файл с планом запросов (если оканчивается на .gz,
            интерпретируется как gzip)
        :param alter_query: функция преобразования запроса (по умолчанию ничего не делает)
        :param rps: целевой RPS для плана
        :param max_queries: обрабатывать не более указанного количества запросов (0 = все запросы)
        :param loader_type: указывать ли при запуске планнера тип нагрузки (см. DolbilkaPlanner.LOADER_TYPES)
    """
    need_cleanup = False
    tmp_file_name = None

    def do(in_file, out_file):
        num_queries = 0
        for line in in_file:
            line = line.strip("\n")
            query = line
            if not query:
                continue

            timedelta = ""
            tabpos = line.find("\t")
            # that means we have queries with timings, include \t in timedelta for proper out
            if tabpos != -1:
                query = line[tabpos + 1:]
                timedelta = line[0:tabpos + 1]

            scheme = urlparse.urlparse(query).scheme
            # possible schemes here are http, http2 and https
            prefix = "" if scheme.startswith("http") else _HTTP_URL
            if query[0] == '?':
                prefix += '/{}'.format(_HTTP_COLLECTION)

            out_file.write("{timedelta}{prefix}{query}\n".format(
                timedelta=timedelta,
                prefix=prefix,
                query=alter_query(query)
            ))

            num_queries += 1
            if 0 < max_queries < num_queries:
                break

    def _open(file_name, mode="r"):
        if file_name.endswith(".gz"):
            return gzip.open(file_name, mode + "b")
        else:
            return open(file_name, mode)

    try:
        if loader_type == 'plain':
            tmp_file_name = "tmp-{}".format(os.path.basename(queries_file_name))
            need_cleanup = True
            with _open(queries_file_name) as inp_file:
                with _open(tmp_file_name, "w") as out_file:
                    do(inp_file, out_file)
        else:
            tmp_file_name = queries_file_name
            need_cleanup = False

        dolbilka_planner = DolbilkaPlanner()
        dolbilka_planner.create_plan(tmp_file_name, plan_file_name, rps=rps, loader_type=loader_type)
    finally:
        if need_cleanup:
            paths.remove_path(tmp_file_name)


@contextlib.contextmanager
def tmpfile():
    f = tempfile.NamedTemporaryFile(delete=False)
    try:
        yield f
    finally:
        try:
            os.remove(f.name)
        except OSError:
            pass


def patch_plan(plan_resource_id, patched_plan, new_cgi_param='', remove_cgi_params='', rps=None):
    with tmpfile() as plan_queries:
        unpack_plan_resource(plan_resource_id, plan_queries)
        create_patched_plan(plan_queries, patched_plan, new_cgi_param, remove_cgi_params, rps)


def patch_plan_resource(plan_resource_id, new_cgi_param='', remove_cgi_params='', rps=None):
    """
        Создать новый ресурс плана с добалением и/или удалением cgi параметров
        :param plan_resource_id: id ресурса плана
        :param new_cgi_param: какой параметр следует добавить к запросам
            Если передавать параметры через незаэкранированную ';' - они будут циклически
            дописываться к запросам по одному параметру в запрос.
        :param remove_cgi_params: какой параметр следует удалить из запросов
        :param rps: количество запросов в секунду
        :return пропатченный план
    """
    plan = channel.sandbox.get_resource(plan_resource_id)
    patched_plan = channel.task.create_resource(
        plan.description + " patched",
        "{}.{}.patched".format(plan.file_name, uuid.uuid4()),
        'BASESEARCH_PLAN',
        arch='any',
    )
    patch_plan(plan_resource_id, patched_plan, new_cgi_param, remove_cgi_params, rps)

    channel.task.mark_resource_ready(patched_plan)
    return patched_plan


def patch_plan_queries(plan_resource_id, plan_queries, new_cgi_param='', remove_cgi_params='', rps=None):
    """
        Создать новый ресурс плана с добалением и/или удалением cgi параметров
        :param plan_resource_id: id ресурса плана
        :param plan_queries: распакованный план
        :param new_cgi_param: какой параметр следует добавить к запросам
            Если передавать параметры через незаэкранированную ';' - они будут циклически
            дописываться к запросам по одному параметру в запрос.
        :param remove_cgi_params: какой параметр следует удалить из запросов
        :param rps: количество запросов в секунду
        :return пропатченный план
    """
    plan = channel.sandbox.get_resource(plan_resource_id)
    patched_plan_resource = channel.task.create_resource(
        plan.description + " patched",
        "{}.{}.patched".format(plan.file_name, uuid.uuid4()),
        'BASESEARCH_PLAN',
        arch='any',
    )
    create_patched_plan(plan_queries, patched_plan_resource, new_cgi_param, remove_cgi_params, rps)

    channel.task.mark_resource_ready(patched_plan_resource)
    return patched_plan_resource


def unpack_plan_resource(plan_resource_id, plan_queries):
    plan_path = channel.task.sync_resource(plan_resource_id)
    plan_queries_count = get_queries_from_plan(
        plan_path,
        plan_queries.name,
    )
    logging.info("Extracted {} queries from plan to {}".format(plan_queries_count, plan_queries.name))


def create_patched_plan(plan_queries, patched_plan_resource, new_cgi_param, remove_cgi_params, rps):
    with tmpfile() as patched_queries_file:
        patch_queries(
            plan_queries.name,
            patched_queries_file.name,
            new_cgi_param,
            remove_cgi_params,
        )
        logging.info("Put patched queries to {}".format(patched_queries_file.name))
        convert_queries_to_plan(
            patched_queries_file.name,
            patched_plan_resource.path,
            rps=rps,
        )
        logging.info("Created plan file {} from patched queries".format(patched_plan_resource.path))
