import re

from six.moves import cStringIO as StringIO

from sandbox.projects.common import file_utils as fu


class StatsParser(object):
    """
    Parse dolbilo run result output.

    Docs: https://wiki.yandex-team.ru/jandekspoisk/sepe/dolbilka/#damper
    """

    KEY_VALUE_RE = re.compile(r'([\d\w./ ]+)\s+([\d.]+).*')
    STATUS_BLOCK_DIVISOR = re.compile(r'^\s+$')
    RECORD_RE = re.compile(r' h\[\d+\]\s+\((\d+)\s*\.*(\d*)\)\s+=\s+(\d+)\s+\(([\d.]+)%\),\s+(\d+)\s+\(([\d.]+)%\)')

    def __init__(self, filename):
        # (low, high, value, percents, lesser_percents, less_percents)
        self.records = []
        self.vars = {}
        self.response_statuses = {}
        self.mcr_sec_intervals = []
        self.val_intervals = []
        try:
            self._parse_stat(filename)
        except Exception:
            pass

    def _parse_stat(self, filename):
        # we read whole file into memory because
        # we need to traverse it several times
        text = fu.read_file(filename)

        # parse first part ouf output, which looks like this
        # error requests            0
        lines = StringIO(text)
        is_status_block = False
        for line in iter(lines):
            match = self.KEY_VALUE_RE.match(line)
            if match:
                key = match.group(1).strip()
                val = float(match.group(2))
                self.vars[key] = float(val)
                if is_status_block:
                    self.response_statuses[key] = int(val)
            elif self.STATUS_BLOCK_DIVISOR.match(line):
                is_status_block = True

        # now grab intervals data
        # first, rewind our buffer
        lines.seek(0)
        # now find the 'F range' marker
        for line in lines:
            if line.startswith('F range'):
                break
        # now grab records in form of
        # h[1]  (1000    ..2000)     = 2383 (11.915%), 19947 (99.735%)
        for line in lines:
            match = self.RECORD_RE.match(line)
            if not match:
                continue
            self.records.append(match.groups())

        # grab intervals data
        f_range = re.compile(r'F range[\S| ]+\n([\S| \n]+)')
        line = re.compile(r'\(([\d.]+)%\)[ |\n]+')
        rage_ = re.compile(r'\((\d+)[ |.]+[\d]*\)')

        text_f = f_range.findall(text)
        self.val_intervals = line.findall(text_f[0]) if text_f else []
        self.mcr_sec_intervals = rage_.findall(text_f[0]) if text_f else []
        self.val_intervals = [float(x) for x in self.val_intervals]
        self.mcr_sec_intervals = [float(x) for x in self.mcr_sec_intervals]
