# -*- coding: utf-8 -*-

from sandbox import sdk2
import os
import re
import logging
import gzip

from sandbox.sandboxsdk import process
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects import resource_types
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.common.dolbilka import DolbilkaParametersError

_HTTP_URL = 'http://localhost:12345'
"""
    HTTP-хост используемый по умолчанию
"""

_HTTP_COLLECTION = 'yandsearch'
"""
    HTTP-коллекция по умолчанию
"""

_HTTP_REQUEST_LINE = re.compile(r'GET (/[^?]+)(\?[^\s]+)\s+HTTP/[0-9.]+')
"""
    Шаблон строки запроса
"""


def _ensure_file_exists(file_name, file_type):
    file_name = os.path.abspath(file_name)
    if not os.path.exists(file_name):
        raise DolbilkaParametersError("Cannot find {}: '{}'".format(file_type, file_name))
    if not os.path.isfile(file_name):
        raise DolbilkaParametersError("This path is not a file: '{}'".format(file_name))
    return file_name


class DolbilkaPlanner2(object):
    """
        Обёртка для d-planner
    """

    name = 'd_planner'
    LOADER_TYPES = ('plain', 'eventlog', 'accesslog', 'fullreqs', 'planlog', 'loadlog', 'phantom', 'pcap')
    resource_type = dolbilka_resources.DPLANNER_EXECUTABLE

    def __init__(self):
        self.path = self.get_planner_path()

    @classmethod
    def get_planner_path(cls):
        """
            Синхронизировать ресурс последнего релиза planner-а и получить путь до него
            :return: путь до исполняемого файла
            :rtype: str
        """
        return utils.sync_last_stable_resource(cls.resource_type)

    def create_plan(
            self,
            queries_path,
            result_path=None,
            rps=None,
            loader_type='plain',
            host=None, port=None,
            log_prefix='d_planner_create_plan'
    ):
        """
            Создать план для долбилки

            :param queries_path: путь до файлов с запросами для плана
            :param result_path: путь создаваемого плана.
                                Если не указан, используется путь queries с добавлением в конце .plan
            :param rps: уровень rps в plan режиме в создаваемом плане
            :param loader_type: указывать ли при запуске планнера тип нагрузки (см. DolbilkaPlanner.LOADER_TYPES)
            :param host: название хоста, которое нужно использовать в плане
            :param port: порт, который нужно использовать в плане
            :param log_prefix: префикс для логов
            :return: путь до созданного плана в виде строки
        """
        queries_path = os.path.abspath(queries_path)
        if result_path:
            result_path = os.path.abspath(result_path)
        else:
            result_path = queries_path + '.plan'

        if host is None and loader_type == 'plain':
            # Если сервер и порт не указаны, dplanner считает, что подаваемые на вход строки
            # должны быть валидными урлами вида http://blablabla/some/path?param1=value1&p2=val2...
            # Если строка не удовлетворяет его требованиям, он печатает в stderr ошибку,
            # игнорирует данную строку и продолжает работу, т.е. код возврата равен нулю.
            # Поэтому, если подать на вход неправильные урлы, мы на выходе получим пустой план.
            # Проверка ниже позволяет обнаруживать эту ситуацию чуть раньше и выдавать
            # вменяемое сообщение об ошибке (SEARCH-810).
            if queries_path.endswith(".gz"):
                queries_file = gzip.open(queries_path, 'rb')
            else:
                queries_file = open(queries_path)
            for line in queries_file:
                if '://' not in line:
                    logging.info("Invalid url for dolbilka planner: '%s'", line.strip())
                    # Так как это по сути проблема входных параметров,
                    # данный тип исключения вполне годится в данном случае.
                    # Валить весь таск в FAILURE было бы неправильно.
                    raise DolbilkaParametersError(
                        "Source queries for dolbilka should be valid urls "
                        "when host is not specified. See line sample in task logs. "
                    )

        run_planner_parameters = [self.path, '-l', queries_path, '-o', result_path]
        if loader_type:
            run_planner_parameters.append('-t')
            run_planner_parameters.append(self.check_loader_type(loader_type))
            if rps and loader_type == 'plain':  # параметр rps применим только для plain режима
                run_planner_parameters.append('-q')
                run_planner_parameters.append(self.check_rps(rps))
        if host:
            run_planner_parameters.append('-h')
            run_planner_parameters.append(str(host))
        if port:
            run_planner_parameters.append('-p')
            run_planner_parameters.append(str(port))
        process.run_process(run_planner_parameters, log_prefix=log_prefix)
        return result_path

    def check_loader_type(self, loader_type):
        """
            Проверить значение параметра check_loader_type
        """
        return loader_type

    @staticmethod
    def check_rps(rps):
        """
            Проверить значение параметра rps
        """
        try:
            if not int(rps) > 0:
                raise DolbilkaParametersError()
            return str(rps)
        except (ValueError, DolbilkaParametersError):
            raise DolbilkaParametersError('DPlanner error. Incorrect rps parameter: {}'.format(rps))


class DolbilkaExecutor2(object):
    """
        Обёртка для d-executor
    """

    class Parameters(sdk2.Task.Parameters):
        dolbilo_executor_resource_id = sdk2.parameters.LastReleasedResource(
            'Dolbilo executor',
            resource_type=dolbilka_resources.DEXECUTOR_EXECUTABLE,
            required=False,
        )
        dolbilka_executor_sessions = sdk2.parameters.Integer(
            'Total sessions',
            default=1,
        )

        dolbilka_executor_time_limit = sdk2.parameters.Integer(
            'Overall time limit (in seconds)',
            default=0,
        )
        dolbilka_executor_time_limit.dolbilka_name = "time-limit"

        dolbilka_executor_max_simultaneous_requests = sdk2.parameters.Integer(
            'Max simultaneous requests',
            default=26,  # Reasonable value: ~ 80% NCPU
        )
        dolbilka_executor_max_simultaneous_requests.dolbilka_name = 'simultaneous'

        dolbilka_request_timeout = sdk2.parameters.Integer(
            'Request timeout',
            default_value=600,
        )
        dolbilka_request_timeout.dolbilka_name = 'timeout'

        dolbilka_executor_mode = sdk2.parameters.String(
            'Executor mode',
            default="finger",
        )
        dolbilka_executor_mode.dolbilka_name = "mode"

        with sdk2.parameters.Group("Rarely used") as rarely_used:
            dolbilka_executor_bounds = sdk2.parameters.String(
                'Fixed time bounds in dump, in microseconds',
                default='',
            )
            dolbilka_executor_requests_limit = sdk2.parameters.Integer(
                'Requests limit',
                default=10000,
            )
            dolbilka_executor_requests_limit.dolbilka_name = "queries-limit"

            dolbilka_fixed_rps = sdk2.parameters.Integer(
                'Fixed RPS (use 0 for default)',
                default=0,
            )
            dolbilka_fixed_rps.dolbilka_name = 'rps-fixed'

            dolbilka_augment_url = sdk2.parameters.String(
                'String to append to each request URL',
                default='',
            )
            dolbilka_augment_url.dolbilka_name = 'augmenturl'

            dolbilka_scheduled_rps = sdk2.parameters.String(
                'RPS schedule (leave empty for default)',
                default='',
            )
            dolbilka_scheduled_rps.dolbilka_name = 'rps-schedule'

            dolbilka_executor_delay_multiplier_q = sdk2.parameters.Float(
                'Plan delay multiplier',
            )
            dolbilka_executor_delay_multiplier_q.dolbilka_name = 'multiplier'

            dolbilka_plan_memory = sdk2.parameters.Bool(
                'Load full plan in memory',
                default=False,
            )
            dolbilka_plan_memory.dolbilka_name = 'plan-memory'

            dolbilka_output_lenval32 = sdk2.parameters.Bool(
                'Dump responses in lenval32 format (SEARCH-2041)',
                default=False,
            )
            dolbilka_output_lenval32.dolbilka_name = 'output-lenval32'

    dolbilka_args_params = (
        Parameters.dolbilka_executor_mode,
        Parameters.dolbilka_executor_requests_limit,
        Parameters.dolbilka_executor_time_limit,
        Parameters.dolbilka_executor_max_simultaneous_requests,
        Parameters.dolbilka_fixed_rps,
        Parameters.dolbilka_scheduled_rps,
        Parameters.dolbilka_request_timeout,
        Parameters.dolbilka_augment_url,
        Parameters.dolbilka_plan_memory,
        Parameters.dolbilka_output_lenval32,
    )

    name = 'd_executor'

    resource_type = dolbilka_resources.DEXECUTOR_EXECUTABLE

    def __init__(self, Parameters):
        self.Parameters = Parameters
        executor_resource_id = self.Parameters.dolbilo_executor_resource_id
        if executor_resource_id:
            self.path = str(sdk2.ResourceData(executor_resource_id).path)
        self.mode = self.Parameters.dolbilka_executor_mode

        self.delay_multiplier = None
        self.sessions = self.Parameters.dolbilka_executor_sessions
        self.bounds = self.Parameters.dolbilka_executor_bounds.strip()

        if self.bounds:
            try:
                # Проверка, что границы - отсортированные по возрастанию числа
                bounds = self.bounds.split(",")
                bounds = map(int, bounds)
                previous_bound = 0
                for bound in bounds:
                    if bound <= previous_bound:
                        raise DolbilkaParametersError(
                            'Bound "{}" must be greater than "{}"'.format(bound, previous_bound)
                        )
                    previous_bound = bound
            except ValueError:
                raise DolbilkaParametersError('Bad bounds value "{}"'.format(self.bounds))

        self.__dolbilka_args = {}
        if hasattr(self.Parameters, "dolbilka_param"):
            delattr(self.Parameters, "dolbilka_param")
        logging.debug(DolbilkaExecutor2.Parameters())
        logging.debug(self.Parameters())
        for param in self.dolbilka_args_params:
            value = getattr(self.Parameters, str(param.name))
            if value:
                self.__dolbilka_args[param.dolbilka_name] = value
        self.__dumper = None
        self.__planner = None

    @property
    def dumper(self):
        if self.__dumper is None:
            self.__dumper = DolbilkaDumper2()
        return self.__dumper

    @property
    def planner(self):
        if self.__planner is None:
            self.__planner = DolbilkaPlanner2()
        return self.__planner

    @staticmethod
    def check_plan(plan):
        """
            Проверить план долбилки

            :param plan: путь до файла плана
            :return: абсолютный путь до плана, если всё в порядке;
                     в противном случае выбрасывается исключение
        """
        return _ensure_file_exists(plan, 'plan')

    def run_session(
            self,
            plan,
            dump=None,
            host='localhost',
            port=None,
            log_prefix='d_executor_run_session',
            phantom=False,
            save_answers=False
    ):
        """
           Запустить одну сессию обстрела

           :param plan: путь до файла с планом
           :param dump: путь до дампа результатов
           :param host: заменить в запросах значение хоста на указанное при обстреле
           :param port: заменить в запросах значение порта на указанное при обстреле
           :param log_prefix: префикс для логов
           :param phantom: формировать дамп в формате, совместимом с `phout.txt`
           :param save_answers: сохранять ответы из target
           :return: путь до дампа
        """
        plan = self.check_plan(plan)

        if dump:
            dump = os.path.abspath(dump)
        else:
            dump = plan + '.dump'

        run_executor_parameters = [
            self.path,
            '--plan-file', plan,
            '--output', dump,
        ]

        for key, value in self.__dolbilka_args.iteritems():
            run_executor_parameters.append("--" + key)
            if not isinstance(value, bool):
                run_executor_parameters.append(str(value))

        def append_parameter(name, value):
            if value:
                run_executor_parameters.append(name)
                run_executor_parameters.append(str(value))

        append_parameter('--replace-host', host)
        append_parameter('--replace-port', port)
        if self.mode:
            append_parameter('--mode', self.mode)
            # параметр количества одинаковых потоков только для devastate и finger режимов
            run_executor_parameters.append('--circular')
        if phantom:
            run_executor_parameters.append('--phantom')
        if save_answers:
            run_executor_parameters.append('-d')
        process.run_process(run_executor_parameters, log_prefix=log_prefix)
        return dump

    def run_sessions(
            self,
            task,
            plan,
            target,
            host='localhost',
            run_once=False,
            need_warmup=False,
            callback=None,
            delay_multiplier=1.0,
            phantom=False,
            save_answers=False
    ):
        """
            Запустить несколько сессий обстрела объекта target с планом plan
            :param task: задача, запустившая долбилку
            :param plan: путь до файла с планом
            :param target: поисковый компонент
            :param host: заменить в запросах значение хоста на указанное при обстреле
            :param run_once: нужно ли запускать поисковый компонент один раз или перезапускать после каждой сессии
            :param save_answers: сохранять ответы из target
            :need_warmup: нужно ли запускать дополнительную сессию вначале для прогрева дискового кеша
            :callback: функция будет вызываться после каждой сессии и получать на вход номер сессии начиная с нуля
            :return: список с dict с информацией о прошедших сессиях стрельбы, см. DolbilkaDumper::parse_dolbilka_stat
        """
        result = []
        self.delay_multiplier = delay_multiplier
        sessions = self.sessions
        if need_warmup:
            parsed_stat = self.run_session_and_dumper(plan, target, "warmup", run_once=False, phantom=phantom)
            result.append(parsed_stat)
        if run_once:
            target.start()
            target.wait()
        for i in range(int(sessions)):
            parsed_stat = self.run_session_and_dumper(
                task,
                plan,
                target,
                str(i),
                host,
                run_once,
                phantom=phantom,
                save_answers=save_answers
            )
            result.append(parsed_stat)
            if callback:
                callback(i)
        if run_once:
            target.stop()
        return result

    def run_session_and_dumper(
            self,
            task,
            plan,
            target,
            session_name,
            host='localhost',
            run_once=False,
            phantom=False,
            save_answers=False,
    ):
        """
            Запустить одну сессию обстрела и после распарсить результаты

            :param plan: путь до файла с планом
            :param target: поисковый компонент
            :param session_name: имя сессии, должно быть уникально при каждом вызове
            :param host: заменить в запросах значение хоста на указанное при обстреле
            :param run_once: нужно ли запускать поисковый компонент один раз или перезапускать после каждой сессии
            :param phantom: Использовать выхлопной формат phantom
            :param save_answers: сохранять ответы из target
            :return: dict с информацией о стрельбе, см. DolbilkaDumper::parse_dolbilka_stat
        """
        session_id = 'test_{}_session_{}'.format(target.name, session_name)
        resource_name = 'test {} session {}'.format(target.name, session_name)
        logging.info('Run dolbilka test session %s', session_id)

        if not run_once:
            target.start()
            target.wait()

        target.use_component(
            lambda: self.run_session(
                plan=plan,
                host=host,
                dump="{}.dump".format(session_id),
                port=target.port,
                log_prefix='run_d_executor_{}'.format(session_id),
                phantom=phantom,
                save_answers=save_answers,
            )
        )

        if not run_once:
            target.stop()

        if phantom:
            parsed_stat = {}
            resource_types.EXECUTOR_STAT_PHOUT(
                task,
                resource_name,
                "{}.dump".format(session_id),
            )
        else:
            parsed_stat = self.run_dumper(
                task,
                "{}.dump".format(session_id), resource_name,
                bounds=self.bounds,
                session_name=session_name,
            )
            if save_answers:
                attributes = {'session_name': session_name}
                resource_types.DOLBILKA_RESPONSES(
                    task,
                    'dolbilka_dump_with_target_answers',
                    "{}.dump".format(session_id),
                    attributes=attributes
                )
            else:
                os.remove("{}.dump".format(session_id))
        parsed_stat['session_id'] = session_id

        return parsed_stat

    def run_dumper(
            self, task, dump_path, stat_resource_descr,
            bounds=None,
            session_name=None,
    ):
        result_stat = self.dumper.get_results(dump_path, parse_results=False, bounds=bounds)
        parsed_stat = self.dumper.parse_dolbilka_stat(result_stat)

        stat_file_name = os.path.splitext(os.path.basename(dump_path))[0] + '.stat'

        # stats are useful, store them within 4 months
        attributes = {'ttl': 120}

        if session_name is not None:
            attributes['session_name'] = session_name

        resource = resource_types.EXECUTOR_STAT(
            task,
            stat_resource_descr,
            os.path.abspath(stat_file_name),
            attributes=attributes,
        )
        fu.write_file(resource.path, result_stat)

        return parsed_stat


class DolbilkaDumper2(object):
    """
        Обёртка для d-dumper
    """
    name = 'd_dumper'
    resource_type = dolbilka_resources.DDUMPER_EXECUTABLE

    @classmethod
    def find_parameter_in_dolbilka_stat(cls, stat_text, template):
        """
            Ищет в text по шаблону parameter_template
            Возвращается значение найденной группы или None

            :param stat_text: текст для поиска
            :param template: скомпилированное регулярное выражение
            :return: если по заданному регулярному выражению было что-то найдено, возвращается значение;
                     в противном случае возвращается None
        """
        parameter_template = re.compile(template)
        parameter_value = parameter_template.search(stat_text)
        if parameter_value:
            return parameter_value.group(1).strip()
        else:
            return None

    @classmethod
    def parse_dolbilka_stat(cls, text):
        """
            Анализирует вывод d-dumper-а, переданного в text

            Возвращаемое значение - dict вида:
            {
                'start_time': 'Время начала обстрела',
                'end_time': 'Время завершения обстрела',
                'full_time': 'Время обстреля',
                'data_readed': 'Сколько данных прочитано в байтах',
                'requests': 'Количество запросов',
                'error_requests': 'Количество error запросов',
                'rps': 'Показатель запросов в секунду',
                'average_request_time': 'Среднее вермя обработки запроса',
                'avg req. len': 'Средняя длина запроса',
                'request_time_deviation': 'Отклоннение в вычислении времени работы запроса',
                'requests_moved': 'Запросов перемещено',
                'requests_not_found': 'Запросов не найдено',
                'requests_ok': 'Успешных запросов',
                'bad_gateway': 'Ошибка HTTP-прокси'
                'service_unavailable': 'Поиск недоступен'
            }
            :param text: текст вывода d-dumper-а
            :return: dict с результатами, его ключи:

        """
        result = {}
        parameters_templates = {
            'start_time': 'start time(.*),',
            'end_time': 'end time(.*),',
            'full_time': 'full time(.*),',
            'data_readed': 'data readed(.*)',
            'requests': 'requests(.*)',
            'error_requests': 'error requests(.*)',
            'rps': 'requests/sec(.*)',
            'average_request_time': 'avg req. time(.*)',
            'avg req. len': 'avg req. len(.*)',
            'request_time_deviation': 'req time std deviation(.*)',
            'requests_moved': 'Moved temporarily(.*)',
            'requests_not_found': 'Not found(.*)',
            'bad_gateway': 'Bad gateway(.*)',
            'requests_ok': 'Ok(.*)',
            'service_unavailable': "Service unavailable(.*)",
        }
        for key, value in parameters_templates.items():
            result[key] = cls.find_parameter_in_dolbilka_stat(text, value)
        return result

    def __init__(self):
        self.path = self.get_dumper_path()

    @classmethod
    def get_dumper_path(cls):
        """
            Синхронизировать ресурс последнего релиза dumper-а и получить путь до него
            :return: путь до исполняемого файла
        """
        return utils.sync_last_stable_resource(cls.resource_type)

    @staticmethod
    def check_dump(dump):
        """
            Проверить дамп долбилки
            :param dump: путь до файла дампа
            :return: абсолютный путь до дампа, если всё в порядке; в противном случае выбрасывается исключение
        """
        return _ensure_file_exists(dump, 'dump')

    def get_results(self, dump, parse_results=False, bounds=None):
        """
            Получить результаты из дампа по заданному пути

            :param dump: путь до дампа
            :param bounds: границы времени, по которым считается % запросов
            :return: вывод дампера в виде строки
        """
        dump = self.check_dump(dump)
        run_dumper_parameters = [self.path, '-a', '-f', dump]
        if bounds:
            run_dumper_parameters.extend(['-m', bounds])
        p = process.run_process(run_dumper_parameters, outs_to_pipe=True)
        result = p.stdout.read()
        if parse_results:
            return self.parse_dolbilka_stat(result)
        return result

    def get_responses(self, dump, output_file, parsed_http_body=False):
        """
            Извлечь ответы, которые получил d-executor
            Note: данные будут получены при условии, что
            executor, создавший dump, был запущен
            с флагом -d (--dump-data)
            :param dump: путь до дампа
            :result_path: путь до файла с результатом
        """
        dump = self.check_dump(dump)
        run_dumper_parameters = [self.path, '-u']
        if parsed_http_body:
            run_dumper_parameters.extend(['parsed-http-body'])
        run_dumper_parameters.extend(['-f', dump])
        process.run_process(run_dumper_parameters, stdout=output_file)

    def get_requests(self, dump, output_file):
        """
            Получить сформированные запросы, которыми стрелял d-executor
            Note: данные будут получены при условии, что
            executor, создавший dump, был запущен
            с флагом -d (--dump-data)
            :param dump: путь до дампа
            :result_path: путь до файла с результатом
        """
        dump = self.check_dump(dump)
        run_dumper_parameters = [self.path, '-A', '-f', dump]
        process.run_process(run_dumper_parameters, stdout=output_file)
