# -*- coding: utf-8 -*-
import logging
import os
import subprocess


def unpack(archiver_path, data_path, destination_path):
    """
    Unpack archive content to specified directory
    """

    cmd = [archiver_path, "-u", "-d", destination_path, data_path]
    logging.info('%s', ' '.join(cmd))
    subprocess.check_call(cmd)


def get_md5(archiver_path, path):
    """
    Получить md5 всех моделей в архиве (archiver -m path)

    :param archiver_path: путь к archiver
    :param path: путь к models.archive
    :return: словарь {имя: md5}
    """

    cmd = [archiver_path, "-m", path]
    logging.info('%s', ' '.join(cmd))
    output = subprocess.check_output(cmd)
    models = (line.strip().split('\t', 1) for line in output.splitlines())
    return {name: md5 for name, md5 in models}


def get_matrixnets_md5(archiver_path, path):
    """
    Получить список моделей ранжирования (.info файлы, расположенные в корне) из архива
    :param archiver_path: путь до архиваторв
    :param path: путь до архива
    :return: словарь {имя модели: md5}
    """

    logging.info('Get models from archive %s', path)
    all_md5 = get_md5(archiver_path, path)

    def models_from_root():
        for full_name, md5 in all_md5.iteritems():
            directory, name = os.path.split(full_name)
            if not directory:
                name, ext = os.path.splitext(name)
                if ext == ".info":
                    yield (name, md5)

    return dict(models_from_root())


def get_list(archiver_path, path):
    """
    Получить список всех файлов в архиве (archiver -l path)

    :param archiver_path: путь к archiver
    :param path: путь к models.archive
    :return: список имен файлов в models.archive
    """

    cmd = [archiver_path, "-l", path]
    logging.info('%s', ' '.join(cmd))
    output = subprocess.check_output(cmd)
    return [line.strip() for line in output.splitlines()]


def create(archiver_path, path, deduplicate, *models):
    """
    Создать архив с формулами

    :param archiver_path: Путь до архиватора
    :param path: Путь до архива
    :param deduplicate: Заменять дубликаты файлов на ссылки
    :param models: Пути до файлов и папок (имена должны быть уникальны)
    :return:
    """
    dedup = ["--deduplicate"] if deduplicate else []
    cmd = [archiver_path, "-q", "-r", "-p", "-o", path] + dedup + list(models)
    logging.info("Start creating archive. Cmd: %s", " ".join(cmd))
    subprocess.check_output(cmd)
