# -*- coding: utf-8 -*-
import itertools
import logging
import urllib

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.base_search_quality import basesearch_response_parser


_OLD_FIELDS = (
    'model_name',
    'matrixnet_name',
    'polynom_name',
    'getmodel_matrixnet',
    'getmodel_polynom',
    'matrixnet_memory_source',
    'matrixnet_md5',
    'matrixnet_id',
)

_NEW_FIELDS = (
    'model_name',
    'matrixnet_name',
    'matrixnet_id',
    'matrixnet_memory_source',
    'matrixnet_md5',
    'getmodel_matrixnet',
    'polynom_name',
    'getmodel_polynom',
)


def get_basesearch_models(basesearch_path, archive_path, search_type="", tmp_path="rankingmodels.tmp", old_mode=False):
    """
        Получает информацию о моделях
        от приложения базового поиска
        :param basesearch_path: путь к бинарнику базового поиска
        :param archive_path: путь к models.archive
        :param search_type: тип поиска (например memory)
        :param tmp_path: путь к временному файлу, куда будет записываться stdout базового
    """

    logging.info('Get models from %s basesearch %s' % (search_type, basesearch_path))
    param = '-I%s:rankingmodels:%s' % (search_type, archive_path)

    with open(tmp_path, "w") as f:
        process.run_process([basesearch_path, param], stdout=f, timeout=10 * 60)

    def parse_line(line):
        return dict(
            zip(_OLD_FIELDS if old_mode else _NEW_FIELDS, line.split('\t'))
        )

    with open(tmp_path, "r") as f:
        return [parse_line(line.rstrip()) for line in f]


def create_plain_text_queries(model_ids, description, output_filename='queries.txt'):
    """
        Создаёт ресурс PLAIN_TEXT_QUERIES
        на основе идентификаторов моделей
        :param model_ids: итерируемый объект с идентификаторами моделей
        :param description: описание создаваемого sandbox-ресурса
        :param output_filename: имя файла создаваемого sandbox-ресурса
    """

    r = channel.task.create_resource(
        description,
        output_filename,
        resource_types.PLAIN_TEXT_QUERIES
    )

    with open(r.path, 'w') as output_file:
        for model_id in model_ids:
            param = urllib.quote_plus("fml=:@{0}".format(model_id))
            output_file.write('?text=test&ms=proto&relev={0}\n'.format(param))

    channel.task.mark_resource_ready(r)
    return r.id


def verify_basesearch_responses(responses_file, model_ids):
    """
        Анализирует файл с ответами от поиска
        и убеждается, что отработали правильные формулы
        :param responses_file: путь к файлу с ответами базового
        :param model_ids: список идентификаторов
    """

    responses = basesearch_response_parser.parse_responses(
        responses_file,
        remove_unstable_props=True,
        use_processes=False,
        c_pickle_dump=False,
    )

    eh.ensure(len(model_ids) == len(responses), 'Models count mismatches responses count')

    def get_ranking_mn(response):
        """
            Ищет свойство типа 'SearcherProp' с
            именем 'RankingMn'
            :param response: ответ базового поиска
        """

        searcher_props = response._nodes.get('SearcherProp')
        eh.ensure(searcher_props, "Search response doesn't contain any SearcherProp property")
        for sp in searcher_props:
            if sp.GetPropValue('Key') == 'RankingMn':
                return sp.GetPropValue('Value')
        eh.check_failed('Cannot find RankingMn search property in response')

    diff = []
    for model_id, response in itertools.izip(model_ids, responses):
        ranking_mn = get_ranking_mn(response)
        if model_id != ranking_mn and model_id != ranking_mn.split('[', 1)[0]:
            diff.append((model_id, ranking_mn))

    if not diff:
        # Everything is fine
        return

    failed_models = '\n'.join([str(p) for p in diff])
    channel.task.set_info("Failed models:\n{}".format(failed_models))
    eh.check_failed('Search has chosen a wrong model for the above formulas (expected, actual)')
